﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{main.cpp,PageSampleEftSimpleDemo}

    @brief
    エフェクトライブラリの使い方を説明したシンプルなサンプルプログラム
 */

/**
    @page PageSampleEftSimpleDemo エフェクトライブラリの使い方を説明したシンプルなサンプル
    @tableofcontents

    @brief
    エフェクトライブラリの使い方の解説です。

    @section PageSampleEftSimpleDemo_SectionBrief 概要
    エフェクトライブラリの使い方を理解するためのシンプルなサンプルプログラムです。

    @section PageSampleEftSimpleDemo_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Applications/Nw4f/eft/SimpleDemo
    Samples/Applications/Nw4f/eft/SimpleDemo @endlink 以下にあります。

    @section PageSampleEftSimpleDemo_SectionNecessaryEnvironment 必要な環境
    とくになし

    @section PageSampleEftSimpleDemo_SectionHowToOperate 操作方法
    とくになし

    @section PageSampleEftSimpleDemo_SectionPrecaution 注意事項
    とくになし

    @section PageSampleEftSimpleDemo_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleEftSimpleDemo_SectionDetail 解説
    このサンプルプログラムは、エフェクトライブラリの基本的な使い方を示します。

    サンプルプログラムの処理の流れは以下の通りです。

    - フレームワークの初期化
    - エフェクトシステムの初期化
        - エフェクトヒープの準備
        - エフェクトシステムの初期化
        - エフェクトリソースの読み込み
        - 先頭のエミッタセット生成
    - メインループの実行
        - フレームバッファ設定
        - エフェクトの計算処理
        - エフェクトの描画処理
        - バッファのスワップ
        - VBlank待ち
    - エフェクトシステムの終了処理
        - エフェクトリソースの破棄
        - エフェクトシステムの破棄
    - フレームワークの終了処理
 */

#include "eftdemo_System.h"
#include "eftdemo_FrameBuffer.h"
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftut2_Heap.h>

// Allocator
nw::ut::MemoryAllocator         gNwAllocator;

// System
nw::eftdemo::System*            gSystem = NULL;

// デフォルトフレームバッファ
nw::eftdemo::FrameBuffer        gFrameBuffer64;

// PrimitiveRenderer
nw::dev::PrimitiveRenderer*     gPrimitiveRenderer = NULL;

nw::math::VEC3                  gCameraPosition;            // カメラ位置
nw::math::VEC3                  gCameraLookAt;              // カメラ視点
nw::math::Matrix34              gView;                      // モデルビュー
nw::math::MTX44                 gProjctionMatrix;           // プロジェクション

#define SCREEN_WIDTH 1280
#define SCREEN_HEIGHT 720

//------------------------------------------------------------------------------
//  初期化処理
//------------------------------------------------------------------------------
void Initialize()
{
    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &gNwAllocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
    nw::eftdemo::System::CreateArg    arg;
    arg.allocator   = &gNwAllocator;
    arg.width       = SCREEN_WIDTH;
    arg.height      = SCREEN_HEIGHT;
    arg.drawMeter   = true;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
    arg.clearColor  = 0x333333FF;
    arg.waitVBlank  = 0;
#if defined(NW_PLATFORM_CAFE)
    arg.waitVBlank  = 1;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#endif
    nw::eftdemo::System::LoadConfigFile(&gNwAllocator, &arg);

    gSystem = new nw::eftdemo::System( arg );

    // デモシステムの初期化
    gSystem->Initialize();

    // 入力インターフェースの初期化
    gSystem->InitializeInputInterface();

    // グラフィックシステムの初期化
    gSystem->InitializeGraphicsSystem();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // PrimitiveRenderer の初期化
        gPrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        gPrimitiveRenderer->Initialize( &gNwAllocator );
#else
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = &gNwAllocator;
        u8* binary = fileSystem->Load( loadArg );
        gPrimitiveRenderer->InitializeFromBinary( &gNwAllocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif

    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Finalize()
{
    // PrimitiveRenderer終了処理
    gPrimitiveRenderer->Finalize( &gNwAllocator );

    // デモシステム終了処理
    gSystem->FinalizeGraphicsSystem();
    gSystem->FinalizeInputInterface();
    gSystem->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();
}


//---------------------------
//  エフェクトデモで利用する変数
//---------------------------
nw::eftut2::NwHeap              g_EffectStaticHeap;
nw::eftut2::NwHeap              g_EffectDynamicHeap;
nw::eftut2::NwHeap              g_EffectResourceHeap;
nw::eft2::System*               g_EffectSystem      = NULL;
nw::eft2::Handle                g_EmitterSetHandle;
void*                           g_EffectBinary      = NULL;
u32                             g_EmitterSetID      = 0;
u32                             g_ResourceID        = 0;
u8                              g_GroupID           = 0;

//------------------------------------------------------------------------------
//  Main 関数
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
int PCSDKMain(int argc, char **argv)
#elif defined(NW_PLATFORM_CAFE)
int NwDemoMain(int argc, char **argv)
#endif
{
    NW_UNUSED_VARIABLE( argc );
    NW_UNUSED_VARIABLE( argv );

#if ( EFT_IS_WIN && _DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

    // アロケーターの初期化
    #define HEAP_SIZE ( 512 * 1024 * 1024 )
    void* addr = malloc( HEAP_SIZE );
    gNwAllocator.Initialize( addr, HEAP_SIZE );

    // フレームワーク初期化処理
    Initialize();

    // フレームバッファの初期化
    f32 width  = SCREEN_WIDTH;
    f32 height = SCREEN_HEIGHT;

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::ut::MemoryAllocator* allocator = &gNwAllocator;

        // 標準フレームバッファ Float
        gFrameBuffer64.Initialize( allocator, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // カメラを初期化
    gCameraPosition.Set( 0.0f, 30.0f, 50.0f );
    gCameraLookAt.Set  ( 0.0f, 00.0f, 0.0f );
    nw::math::VEC3 comUp ( 0.0f, 1.0f, 0.0f );
    nw::math::VEC3 target( 0.0f, 0.0f, 0.0f);
    gView.Identity();
    gView.SetLookAt( gCameraPosition, comUp, target );

    // プロジェクションを初期化
    nw::math::MTX44Perspective( &gProjctionMatrix, nw::math::F_PI/3, width/(f32)height, 0.1f, 1000.f);

    //----------------------------------------
    // エフェクト初期化
    //----------------------------------------
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        //----------------------------------------
        // エフェクトヒープの準備
        //----------------------------------------
        g_EffectStaticHeap.SetNwAllocator( &gNwAllocator );
        g_EffectDynamicHeap.SetNwAllocator( &gNwAllocator );
        g_EffectResourceHeap.SetNwAllocator( &gNwAllocator );

        //----------------------------------------
        // エフェクトシステムの初期化
        //----------------------------------------
        nw::eft2::Config  config;

        // システム初期化時に確保されるワークの確保先
        config.SetEffectHeap( &g_EffectStaticHeap );
        // エミッタセット生成時に確保されるパーティクルワークの確保先
        config.SetEffectDynamicHeap( &g_EffectDynamicHeap );
        config.SetEmitterSetNum( 8 );
        config.SetEmitterNum( 32 );
        config.SetResourceNum( 8 );
        config.SetTemporaryBufferSize( 256*1024 );
        g_EffectSystem = new nw::eft2::System( config );

        //----------------------------------------
        // エフェクトバイナリリソース読み込み
        //----------------------------------------
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "demo.ptcl";
        loadArg.allocator = &gNwAllocator;
        loadArg.alignment = nw::eft2::EFT_PTCL_BINARY_ALIGNMENT;
        g_EffectBinary = fileSystem->Load( loadArg );

        //g_EffectBinary = _binaryLoad( "eft\\test.ptcl", &g_EffectResourceHeap, &fileSize );
        EFT_ASSERT( g_EffectBinary != NULL );
        g_EffectSystem->EntryResource( &g_EffectResourceHeap, g_EffectBinary, g_ResourceID );

        //----------------------------------------
        // エフェクトバイナリ内 0番目のエミッタセットを生成
        //----------------------------------------
        g_EmitterSetHandle.Invalidate();
        g_EffectSystem->CreateEmitterSetID( &g_EmitterSetHandle, g_EmitterSetID, g_ResourceID, g_GroupID );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();


    // メインループ
    bool bLoop = true;
    while ( !gSystem->IsExiting() && bLoop )
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        nw::eftdemo::FrameBuffer* currentFrameBuffer = &gFrameBuffer64;
        {
            // フレームバッファ設定
            currentFrameBuffer->Bind();
            currentFrameBuffer->SetClearColor( nw::math::VEC4( 0.0f, 0.0f, 0.0f, 0.0f) );
            currentFrameBuffer->Clear( true, true );

            //----------------------------------------
            // エフェクト定期処理
            //----------------------------------------
            {
                //----------------------------------------
                // エフェクトの計算処理
                //----------------------------------------
                g_EffectSystem->BeginFrame();
                g_EffectSystem->Calc( g_GroupID, 1.0f, true );


#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif
                //----------------------------------------
                // エフェクトの描画処理
                //----------------------------------------
                nw::eft2::ViewParam view;
                view.Set( &gView, &gProjctionMatrix, &gCameraPosition, 0.1f, 10000.f );
                g_EffectSystem->SwapBuffer();
                g_EffectSystem->SetViewParam( &view );
                g_EffectSystem->Draw( g_GroupID );
#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

            }
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

        // バッファのスワップ
        gSystem->SwapBuffer( currentFrameBuffer );

        //  VBlank 待ち
        gSystem->WaitForFlip();
    }


    //----------------------------------------
    // エフェクト終了処理
    //----------------------------------------
    {
        //----------------------------------------
        // エフェクトリソースの破棄
        //----------------------------------------
        g_EffectSystem->ClearResource( &g_EffectResourceHeap, 0 );

        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "demo.ptcl";
        loadArg.allocator = &gNwAllocator;
        loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
        fileSystem->Unload( loadArg, reinterpret_cast<u8*>( g_EffectBinary ) );

        //----------------------------------------
        // エフェクトシステムの破棄
        //----------------------------------------
        if ( g_EffectSystem ) delete g_EffectSystem;
        g_EffectSystem = NULL;
    }

    // フレームワークの終了処理
    Finalize();

    // アロケータ終了処理
    gNwAllocator.Finalize();
    free( addr );

    return 0;
}
