﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "eftdemo_Define.h"
#include "eftdemo_FrameBuffer.h"
#include "eftdemo_System.h"

#include <nw/eft/eftut2_Heap.h>
#include <nw/eft/eftvw2_FileSystem.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    #define WIN32_LEAN_AND_MEAN
    #define NOMINMAX
    #include <windows.h>
    #include <mmsystem.h>
    #include <winext/cafe/os/win32/os_WindowMessage.h>
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

namespace nw
{
    namespace internal
    {
        namespace winext
        {
            // PcSDK で作成されるウィンドウのハンドルです。
            extern HWND ghCommandWnd;
            extern HWND ghFrameWnd;
        }
    }
}

#endif


namespace nw      {
namespace eftdemo {


#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
System* System::sInstance = NULL;

// 設定ファイル名です。
static const char *CONFIG_FILE_NAME = "config.ini";
#endif


//------------------------------------------------------------------------------
//      コンストラクタ
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
System::System( const CreateArg& arg )
: m_Arg( arg ),
  m_MeterDrawer( &m_MeterFrame ),
  m_TextWriterInitialized( false ),
  m_DisplayState( HIDE ),
  m_HWnd( NULL ),
  m_Exit( false )
{
    NW_ASSERT( !sInstance );
    sInstance = this;
}
#endif
#ifdef NW_PLATFORM_CAFE
System::System( const CreateArg& arg )
: m_Arg( arg ),
  m_MeterDrawer( &m_MeterFrame ),
  m_TextWriterInitialized( false ),
  m_DisplayState( HIDE ),
  m_ScanBufferPtr( NULL ),
  m_ContextState( NULL )
{
    m_Flags.SetMaskOn( FLAG_IS_BLACK );
}
#endif


//------------------------------------------------------------------------------
//      システム　初期化処理
//------------------------------------------------------------------------------
void System::Initialize()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    SetupWindow();
#endif
#ifdef NW_PLATFORM_CAFE
    // 高速キャストを有効化します。

#if defined( __ghs__ )
    OSInitFastCast();
#endif

#endif

    // デモファイルシステムの初期化を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
        fileDeviceArg.allocator = m_Arg.allocator;
        fileSystem->Initialize( fileDeviceArg );
    }
}


//------------------------------------------------------------------------------
//      システム　終了処理
//------------------------------------------------------------------------------
void System::Finalize()
{
    // デモファイルシステムの終了処理を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager::GetInstance()->Finalize();
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    OS_SetWindowMessageCallback( NULL );
#endif
}


//------------------------------------------------------------------------------
//      入力関連 初期化処理
//------------------------------------------------------------------------------
void System::InitializeInputInterface()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    nw::dev::JoyPadDeviceWin::GetInstance()->Initialize();
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->Initialize();
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->SetMainWindowHandle( m_HWnd );

    m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadWin ) ) ) nw::dev::PadWin( 0 );
    m_Pad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
    m_Pad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );

    m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::dev::MouseWin ) ) ) nw::dev::MouseWin();
    m_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
#endif
#ifdef NW_PLATFORM_CAFE
    nw::dev::PadDeviceCafe::GetInstance()->Initialize();

    m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadCafe ) ) ) nw::dev::PadCafe( 0 );
    m_Pad->SetPadDevice( nw::dev::PadDeviceCafe::GetInstance() );
#endif

}


//------------------------------------------------------------------------------
//      入力関連 更新処理
//------------------------------------------------------------------------------
void System::UpdateInputInterface()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->Update();
    nw::dev::JoyPadDeviceWin::GetInstance()->Update();
    m_Pad->Update();
    m_Mouse->Update();
#endif
#ifdef NW_PLATFORM_CAFE
    nw::dev::PadDeviceCafe::GetInstance()->Update();
    m_Pad->Update();
#endif
}


//------------------------------------------------------------------------------
//      入力関連 終了処理
//------------------------------------------------------------------------------
void System::FinalizeInputInterface()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    nw::dev::JoyPadDeviceWin::GetInstance()->Finalize();
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->Finalize();

    nw::ut::SafeFree( m_Pad, m_Arg.allocator );
    nw::ut::SafeFree( m_Mouse, m_Arg.allocator );
#endif
#ifdef NW_PLATFORM_CAFE
    nw::dev::PadDeviceCafe::GetInstance()->Finalize();
    nw::ut::SafeFree( m_Pad, m_Arg.allocator );
#endif
}


//------------------------------------------------------------------------------
//      グラフィックスシステム初期化処理
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
void System::InitializeGraphicsSystem( int argc /* = 0 */, char** argv /* = NULL */ )
{
    NW_UNUSED_VARIABLE( argc );
    NW_UNUSED_VARIABLE( argv );
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    // グラフィックスデバイス初期化.
    {
        HDC hdc = GetDC( reinterpret_cast<HWND>( m_HWnd ) );

        // PIXELFORMATDESCRIPTOR準備.
        s32 pixelformat;
        PIXELFORMATDESCRIPTOR pfd;

        ::std::memset( &pfd, 0, sizeof( PIXELFORMATDESCRIPTOR ) );
        pfd.nSize           = sizeof( PIXELFORMATDESCRIPTOR );
        pfd.nVersion        = 1;
        pfd.dwFlags         = PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER | PFD_GENERIC_ACCELERATED;
        pfd.iPixelType      = PFD_TYPE_RGBA;
        pfd.cColorBits      = 32;
        pfd.cDepthBits      = 24;
        pfd.cStencilBits    = 8;

        // ピクセルフォーマット取得.
        if ( ( pixelformat = ChoosePixelFormat( hdc, &pfd ) ) == 0 )
        {
            NW_ERR( "Failed ChoosePixelFormat." );
            ReleaseDC( reinterpret_cast<HWND>( m_HWnd ), hdc );
            return;
        }

        DescribePixelFormat( hdc, pixelformat, pfd.nSize, &pfd );
        if ( !SetPixelFormat( hdc, pixelformat, &pfd ) )
        {
            NW_ERR( "Failed SetPixelFormat." );
            ReleaseDC( reinterpret_cast<HWND>( m_HWnd ), hdc );
            return;
        }

        // gl生成.
        HGLRC hglrc = wglCreateContext( hdc );

        {
            wglMakeCurrent( hdc, hglrc );
            GLenum err = glewInit();
            NW_ASSERT( err == GLEW_OK );
            NW_GL_ASSERT();
            wglMakeCurrent( hdc, NULL );
        }

        //ReleaseDC( mHWnd, hdc );

        // グラフィックスインスタンス生成.
        {
            nw::gfnd::Graphics::CreateArg arg;
            arg.hdc = hdc;
            arg.hglrc = hglrc;

            nw::gfnd::Graphics::SetInstance( new( m_Arg.allocator->Alloc( sizeof(nw::gfnd::Graphics) ) ) nw::gfnd::Graphics( arg ) );
            nw::gfnd::Graphics::GetInstance()->Initialize();

            m_HDC = hdc;
            m_HGLRC = hglrc;
        }

        nw::math::VEC2 frameBufferSize( static_cast<f32>( m_Arg.width ), static_cast<f32>( m_Arg.height ) );
        CreateFrameBuffer( frameBufferSize );
    }

    // PrimitiveRenderer の初期化を行います。
    if ( m_Arg.primitiveRendererInitialize )
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        nw::dev::PrimitiveRenderer::GetInstance()->Initialize( m_Arg.allocator );
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }

    // デバッグ文字列描画用行列を計算します。
    {
        f32 l     = 0.f;
        f32 r     = static_cast<f32>( m_Arg.width );
        f32 t     = 0.f;
        f32 b     = static_cast<f32>( m_Arg.height );
        f32 zNear = 0.0f;
        f32 zFar  = 1.f;
        nw::math::MTX44Ortho( &m_TextWriterProjMtx, l, r, b, t, zNear, zFar );
        nw::math::MTX34Identity( &m_TextWriterViewMtx );
    }

    // デバッグ文字描画のための DevTextWriter を初期化します。
    if ( m_Arg.fontBinary )
    {
        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            NULL,
            0,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }
    else if ( m_Arg.fontPath )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.fontPath;
            loadArg.allocator = m_Arg.allocator;
            loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
            m_Arg.fontBinary = fileSystem->Load( loadArg );
            m_Arg.fontBinarySize = loadArg.readSize;
        }

        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            NULL,
            0,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }

    // 負荷メーターの初期化を行います。
    if ( m_Arg.drawMeter )
    {
        m_MeterFrame.SetName( "Frame" );
        m_MeterFrame.SetColor( nw::ut::Color4u8( DEMO_COLOR_CPU_FRAME ) );
        m_MeterCalc.SetName( "Calc" );
        m_MeterCalc.SetColor( ut::Color4u8( DEMO_COLOR_CPU_CALC ) );
        m_MeterDraw.SetName( "Draw" );
        m_MeterDraw.SetColor( ut::Color4u8( DEMO_COLOR_CPU_DRAW ) );
        m_MeterGPU.SetName( "GPU" );
        m_MeterGPU.SetColor( ut::Color4u8( DEMO_COLOR_GPU ) );
        m_MeterGPU.Initialize();

        m_MeterDrawer.AttachLoadMeter( &m_MeterCalc );
        m_MeterDrawer.AttachLoadMeter( &m_MeterDraw );
        m_MeterDrawer.AttachLoadMeter( &m_MeterGPU );

        m_MeterDrawer.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_MeterDrawer.SetTextWriter( m_TextWriterInitialized ? &m_DevTextWriter : NULL );
        m_MeterDrawer.SetFrameRate( m_Arg.waitVBlank > 0 ? 60.f / m_Arg.waitVBlank : m_Arg.fps );
        m_MeterDrawer.SetMinimizeBarSize( 0.f );

        // 負荷メーターの位置、大きさを調整します。
        {
            const f32 METER_WIDTH = m_Arg.width - 10.f;
            const f32 METER_HEIGHT = m_MeterDrawer.GetAdjustedHeight();

            m_MeterDrawer.SetWidth( METER_WIDTH );
            m_MeterDrawer.SetHeight( METER_HEIGHT );
            m_MeterDrawer.SetPosition( nw::math::VEC2( 5.f, m_Arg.height - ( METER_HEIGHT + 5.f ) ) );
        }
    }
}
#endif
#ifdef NW_PLATFORM_CAFE
void System::InitializeGraphicsSystem( int argc /* = 0 */, char** argv /* = NULL */ )
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    {
        MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
        u32 mem1Size = MEMGetAllocatableSizeForFrmHeapEx(hMEM1, 4);
        void* mem1 = MEMAllocFromFrmHeapEx(hMEM1, mem1Size, 4);
        m_VRAMAllocator.Initialize(mem1, mem1Size);
    }

    // GX2初期化
    u32 gx2InitAttribs[] =
    {
        GX2_INIT_ATTRIB_ARGC,    (u32)argc,  // number of args
        GX2_INIT_ATTRIB_ARGV,    (u32)argv,  // command-line args
        GX2_INIT_ATTRIB_NULL                 // terminates the list
    };
    GX2Init(gx2InitAttribs);

    // GX2のコンテキストを作成し、Graphicsに設定
    m_ContextState = static_cast<GX2ContextState*>( m_Arg.allocator->Alloc(sizeof(GX2ContextState), GX2_CONTEXT_STATE_ALIGNMENT) );

    GX2SetupContextState( m_ContextState );

    // グラフィックスインスタンス生成.
    {
        nw::gfnd::Graphics::SetInstance( new( m_Arg.allocator->Alloc( sizeof( nw::gfnd::Graphics ) ) ) nw::gfnd::Graphics() );
        nw::gfnd::Graphics::GetInstance()->Initialize();
        nw::gfnd::Graphics::GetInstance()->SetGX2ContextState(m_ContextState);
    }

    // スキャンバッファの設定
    {
        GX2Boolean scaleNeeded;

        // スキャンバッファのサイズを計算する
        GX2CalcTVSize( GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_BUFFERING_DOUBLE, &m_ScanSize, &scaleNeeded );
        // スキャンバッファのメモリを取得する。
        m_ScanBufferPtr = m_Arg.allocator->Alloc( m_ScanSize, GX2_SCAN_BUFFER_ALIGNMENT );
        // スキャンバッファを設定する
        GX2SetTVBuffer( m_ScanBufferPtr, m_ScanSize, GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_BUFFERING_DOUBLE );

        // スキャンバッファのサイズを計算する
        GX2CalcTVSize( GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, GX2_BUFFERING_DOUBLE, &m_LinearScanSize, &scaleNeeded );
        // スキャンバッファのメモリを取得する。
        m_LinearScanBufferPtr = m_Arg.allocator->Alloc( m_LinearScanSize, GX2_SCAN_BUFFER_ALIGNMENT );
        // スキャンバッファを設定する
        GX2SetTVBuffer( m_LinearScanBufferPtr, m_LinearScanSize, GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, GX2_BUFFERING_DOUBLE );
    }

    GX2SetTVScale( m_Arg.width, m_Arg.height );

    GX2SetSwapInterval( m_Arg.waitVBlank );

    // PrimitiveRenderer の初期化を行います。
    {
        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();

        if ( m_Arg.primitiveRendererShaderBinary )
        {
            renderer->InitializeFromBinary( m_Arg.allocator, m_Arg.primitiveRendererShaderBinary, m_Arg.primitiveRendererShaderBinarySize );
        }
        else if ( m_Arg.primitiveRendererShaderPath )
        {
            nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.primitiveRendererShaderPath;
            loadArg.allocator = m_Arg.allocator;
            u8* binary = fileSystem->Load( loadArg );

            renderer->InitializeFromBinary( m_Arg.allocator, binary, loadArg.readSize );

            fileSystem->Unload( loadArg, binary );
        }
    }

    // デバッグ文字列描画用行列を計算します。
    {
        f32 l     = 0.f;
        f32 r     = static_cast<f32>( m_Arg.width );
        f32 t     = 0.f;
        f32 b     = static_cast<f32>( m_Arg.height );
        f32 zNear = 0.0f;
        f32 zFar  = 1.f;
        nw::math::MTX44Ortho( &m_TextWriterProjMtx, l, r, b, t, zNear, zFar );
        nw::math::MTX34Identity( &m_TextWriterViewMtx );
    }

    // デバッグ文字描画のための DevTextWriter を初期化します。
    if ( m_Arg.fontBinary )
    {
        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            m_Arg.fontShaderBinary,
            m_Arg.fontShaderBinarySize,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }
    else if ( m_Arg.fontPath )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        u8* fontShaderBinary = NULL;
        u32 fontShaderBinarySize = 0;

        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.fontPath;
            loadArg.allocator = m_Arg.allocator;
            loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
            m_Arg.fontBinary = fileSystem->Load( loadArg );
            m_Arg.fontBinarySize = loadArg.readSize;
        }

        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.fontShaderPath;
            loadArg.allocator = m_Arg.allocator;
            loadArg.alignment = 128;
            fontShaderBinary = fileSystem->Load( loadArg );
            fontShaderBinarySize = loadArg.readSize;
        }

        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            fontShaderBinary,
            fontShaderBinarySize,
            m_Arg.allocator
        );

        m_Arg.fontShaderBinary       = fontShaderBinary;
        m_Arg.fontShaderBinarySize   = fontShaderBinarySize;

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;


    }

    // 負荷メーターの初期化を行います。
    if( m_Arg.drawMeter )
    {
        m_MeterFrame.SetName( "Frame" );
        m_MeterFrame.SetColor( nw::ut::Color4u8( DEMO_COLOR_CPU_FRAME ) );
        m_MeterCalc.SetName( "Calc" );
        m_MeterCalc.SetColor( ut::Color4u8( DEMO_COLOR_CPU_CALC ) );
        m_MeterDraw.SetName( "Draw" );
        m_MeterDraw.SetColor( ut::Color4u8( DEMO_COLOR_CPU_DRAW ) );
        m_MeterGPU.SetName( "GPU" );
        m_MeterGPU.SetColor( ut::Color4u8( DEMO_COLOR_GPU ) );

        m_GpuCounters = reinterpret_cast<u64*>( m_Arg.allocator->Alloc( m_MeterGPU.GetBufferSize(), GX2_DEFAULT_BUFFER_ALIGNMENT ) );
        m_MeterGPU.Initialize( m_GpuCounters );

        m_MeterDrawer.AttachLoadMeter( &m_MeterCalc );
        m_MeterDrawer.AttachLoadMeter( &m_MeterDraw );
        m_MeterDrawer.AttachLoadMeter( &m_MeterGPU );

        m_MeterDrawer.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_MeterDrawer.SetTextWriter( m_TextWriterInitialized ? &m_DevTextWriter : NULL );
        m_MeterDrawer.SetFrameRate( 60.f / m_Arg.waitVBlank );
        m_MeterDrawer.SetMinimizeBarSize( 0.f );

        // 負荷メーターの位置、大きさを調整します。
        // 安全フレームを考慮して、やや小さくします。
        {
            const f32 METER_WIDTH = m_Arg.width * 0.965f;
            const f32 METER_HEIGHT = m_MeterDrawer.GetAdjustedHeight();
            const f32 METER_POS_X = (m_Arg.width - METER_WIDTH ) / 2.f;
            const f32 METER_POS_Y = m_Arg.height - ( METER_HEIGHT + m_Arg.height / 50.f );

            m_MeterDrawer.SetWidth( METER_WIDTH );
            m_MeterDrawer.SetHeight( METER_HEIGHT );
            m_MeterDrawer.SetPosition( nw::math::VEC2( METER_POS_X, METER_POS_Y) );
        }
    }
}
#endif


//------------------------------------------------------------------------------
//      グラフィックスシステム終了処理
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
void System::FinalizeGraphicsSystem()
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Finalize();
    }

    if ( m_Arg.fontPath && m_Arg.fontBinary )
    {
        m_Arg.allocator->Free( m_Arg.fontBinary );
    }

    if ( m_Arg.primitiveRendererInitialize )
    {
        nw::dev::PrimitiveRenderer::GetInstance()->Finalize( m_Arg.allocator );
    }

    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();
    nw::ut::SafeFree( gfx, m_Arg.allocator );
}
#endif
#ifdef NW_PLATFORM_CAFE
void System::FinalizeGraphicsSystem()
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    if ( m_Arg.drawMeter )
    {
        m_Arg.allocator->Free( m_GpuCounters );
    }

    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Finalize();
    }

    if ( m_Arg.fontPath && m_Arg.fontBinary )
    {
        m_Arg.allocator->Free( m_Arg.fontBinary );
    }

    if ( m_Arg.fontShaderBinary )
    {
         m_Arg.allocator->Free( m_Arg.fontShaderBinary );
    }

    if ( m_Arg.primitiveRendererShaderBinary || m_Arg.primitiveRendererShaderPath )
    {
        nw::dev::PrimitiveRenderer::GetInstance()->Finalize( m_Arg.allocator );
    }

    GX2SetTVEnable( GX2_FALSE );

    nw::ut::SafeFree( m_ScanBufferPtr, m_Arg.allocator );

    nw::ut::SafeFree( m_ContextState, m_Arg.allocator );

    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();
    nw::ut::SafeFree( gfx, m_Arg.allocator );

    GX2Shutdown();

    m_VRAMAllocator.Finalize();
    MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
    MEMFreeToFrmHeap(hMEM1, MEM_FRMHEAP_FREE_ALL);
}
#endif


//------------------------------------------------------------------------------
//      インターバル設定
//------------------------------------------------------------------------------
bool System::SetWaitVBlankInterval(u32 interval)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    {
        bool vsync_ok = nw::gfnd::Graphics::GetInstance()->SetVBlankWaitInterval( interval );

        if ( !vsync_ok )
        {
            return false;
        }
        else
        {
            m_Arg.waitVBlank = interval;
            return true;
        }
    }
#endif
#ifdef NW_PLATFORM_CAFE
    {
        m_Arg.waitVBlank = interval;
        return true;
    }
#endif
}


//------------------------------------------------------------------------------
//      Fps設定
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
void System::SetFps( f32 fps )
{
    // 1フレーム相当の時間をあらかじめ計算しておく.
    m_FrameTime = nw::ut::TimeSpan::FromMicroSeconds( static_cast<s64>( 1000000 / fps ) );
    m_Arg.fps = fps;
}
#endif


//------------------------------------------------------------------------------
//      終了処理へ移行するか
//------------------------------------------------------------------------------
bool System::IsExiting() const
{
    // メモリーリーク検出が有効な場合、パッド操作で終了判定を true にします。
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#ifdef NW_DEBUG_CHECK_MEMORY_LEAK
    if( m_Pad->IsHoldAll( nw::dev::Pad::MASK_L | nw::dev::Pad::MASK_R | nw::dev::Pad::MASK_X ) )
    {
        return true;
    }

    // 普通に終了した場合は、再入テストに入らずに素直に終了する。
    if (m_Exit)
    {
        nw::demo::DebugUtility::DisableReentryTest(true);
    }
#endif //   NW_DEBUG_CHECK_MEMORY_LEAK

    return m_Exit;

#endif //   NW_PLATFORM_WIN32

#ifdef NW_PLATFORM_CAFE

    //Cafe版では常にパッド操作で終了判定を true にします。
    if( m_Pad->IsHoldAll( nw::dev::Pad::MASK_L | nw::dev::Pad::MASK_R | nw::dev::Pad::MASK_X ) )
    {
        return true;
    }

    return false;
#endif //   NW_PLATFORM_CAFE

}


//------------------------------------------------------------------------------
//      画面更新
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
void System::SwapBuffer( FrameBuffer* frameBuffer )
{
    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

    gfx->LockDrawContext();
    {
        if ( frameBuffer != NULL )
        {
            frameBuffer->CopyToDisplayBuffer( m_CurrentScanBuffer );

            if ( m_CurrentScanBuffer == &m_ScanBufferSRGB )
            {
                glEnable( GL_FRAMEBUFFER_SRGB_EXT);
            }

            m_CurrentScanBuffer->CopyToDisplayBuffer( NULL );

            if ( m_CurrentScanBuffer == &m_ScanBufferSRGB )
            {
                glDisable( GL_FRAMEBUFFER_SRGB_EXT);
            }
        }
    }
    gfx->UnlockDrawContext();

    // 転送.
    SwapBuffers( reinterpret_cast<HDC>( m_HDC ) );

    // 表示開始.
    if ( GetDisplayState() == READY )
    {
        RECT workAreaRect;
        SystemParametersInfo( SPI_GETWORKAREA, 0, &workAreaRect, 0 );

        RECT commandWindowRect;
        GetWindowRect( nw::internal::winext::ghCommandWnd, &commandWindowRect );

        // ウィンドウ位置を調整
        SetWindowPos(
            nw::internal::winext::ghCommandWnd,
            NULL,
            workAreaRect.right - ( commandWindowRect.right - commandWindowRect.left ),
            workAreaRect.bottom - ( commandWindowRect.bottom - commandWindowRect.top ),
            0,
            0,
            SWP_NOSIZE);

        SetWindowPos(
            reinterpret_cast<HWND>( m_HWnd ),
            HWND_TOP,
            workAreaRect.left,
            workAreaRect.top,
            0,
            0,
            SWP_NOSIZE | SWP_SHOWWINDOW);

        m_DisplayState = SHOW;
    }
}
#endif
#ifdef NW_PLATFORM_CAFE
void System::SwapBuffer( FrameBuffer* frameBuffer )
{
    // カラーバッファをスキャンバッファに転送して画面をスワップ
    GX2SwapBuffers( frameBuffer->GetGX2ColorBuffer());

    // context stateを戻す
    GX2SetContextState( nw::gfnd::Graphics::GetInstance()->GetGX2ContextState() );

    // カラーバッファを再設定、この処理はスワップの後に必ず行う必要がある
    frameBuffer->Bind();

    // スワップコマンドが確実に実行されるようにコマンドをフラッシュする
    GX2Flush();

    //  画面表示切り替え
    if ( m_Flags.IsMaskOn( FLAG_CHANGE_BLACK ) )
    {
        SetBlack( ! IsBlack());
        m_Flags.SetMaskOff( FLAG_CHANGE_BLACK );
    }

    // カラーバッファへの描画完了を待つ
    GX2DrawDone();

    // 表示開始.
    if ( GetDisplayState() == READY )
    {
        ReserveSetBlack( false );
        m_DisplayState = SHOW;
    }
}
#endif


//------------------------------------------------------------------------------
//      フリップ待ち
//------------------------------------------------------------------------------
void System::WaitForFlip()
{
#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

    // VBlank 待ちの間、メッセージ処理を行います。
    for(;;)
    {
        MSG msg;
        if( PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) )
        {
            // 処理するメッセージがあるときはそれを優先.
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
        else
        {
            // メッセージがないときは、経過時間を見て判断.
            // vsyncが有効な時は時間を無視してフレーム処理.
            nw::ut::TimeSpan dt = nw::ut::Tick::GetSystemCurrent() - m_LastUpdateTime;
            if ( m_Arg.waitVBlank > 0 || dt > m_FrameTime )
            {
                m_LastDiffTime = dt;
                m_LastUpdateTime = nw::ut::Tick::GetSystemCurrent();

                nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

                // Windowsでは、次のlockDrawContextで初めてVBlank待ちが行われる模様.
                gfx->LockDrawContext();
                gfx->UnlockDrawContext();

                Sleep( 0 );

                break;
            }
            else if ( dt + nw::ut::TimeSpan::FromMilliSeconds( 10 ) > m_FrameTime )
            {
                // 残り10.0msec以下のときはSleep(0).
                Sleep( 0 );
            }
            else
            {
                // 残り10.0msec以上ならSleep(1)で下位スレッドに処理を回す.
                timeBeginPeriod( 1 );
                Sleep( 1 );
                timeEndPeriod( 1 );
            }
        }
    }

#elif defined(NW_PLATFORM_CAFE)

    GX2WaitForFlip();

#endif
}


//------------------------------------------------------------------------------
//      フレーム開始処理
//------------------------------------------------------------------------------
void System::BeginFrame()
{
    m_MeterFrame.BeginMeasure();
}


//------------------------------------------------------------------------------
//      フレーム終了処理
//------------------------------------------------------------------------------
void System::EndFrame()
{
    ClearTextWriter();
    m_MeterFrame.EndMeasure();
    m_MeterDrawer.EndFrame();
}


//------------------------------------------------------------------------------
//      コストメータ描画
//------------------------------------------------------------------------------
void System::DrawLoadMeters()
{
    if( m_Arg.drawMeter )
    {
        m_MeterDrawer.Draw();
    }
}


//------------------------------------------------------------------------------
//      テキストライタクリア
//------------------------------------------------------------------------------
void System::ClearTextWriter()
{
    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Clear();
    }
}


//------------------------------------------------------------------------------
//      コンフィグファイルのロード
//------------------------------------------------------------------------------
void System::LoadConfigFile( nw::ut::MemoryAllocator *allocator, nw::eftdemo::System::CreateArg *arg )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

    char *buf = NULL;
    u32        bufSize;

    nw::eftut2::NwHeap heap;
    heap.SetNwAllocator(allocator);
    bool success = nw::eftvw2::FileSystem::Load(&heap, CONFIG_FILE_NAME, (void**)&buf, &bufSize);

    if ( !success || !buf )
    {
        NW_LOG( "[EftDemoSystem] File load error : " );
        return;
    }

    int frameRate = 0;
    int resolution = 0;
    f32 clearColor[4];
    char* nexttoken;

    const char *c = strtok_s( buf, "\n", &nexttoken );

    int line = 1;
    while ( c )
    {
        NW_LOG(c);
        switch(line++)
        {
        case 1: frameRate = atoi(c);            break;
        case 2: resolution = atoi(c);           break;
        case 3: clearColor[0] = (f32)atof(c);   break;
        case 4: clearColor[1] = (f32)atof(c);   break;
        case 5: clearColor[2] = (f32)atof(c);   break;
        case 6: clearColor[3] = (f32)atof(c);   break;
        default: break;
        }
        c = strtok_s( NULL, "\n",  &nexttoken );
    }

    switch (frameRate)
    {
    case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_60:
        arg->fps = 60.0f;
        break;
    case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_30:
        arg->fps = 30.0f;
        break;
    case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_20:
        arg->fps = 20.0f;
        break;
    case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_15:
        arg->fps = 15.0f;
        break;
    }

    switch (resolution)
    {
    case nw::eftdemo::INIT_SETTINGS_RESOLUTION_960_540:
        arg->width  = 960;
        arg->height = 540;
        break;
    case nw::eftdemo::INIT_SETTINGS_RESOLUTION_640_480:
        arg->width  = 640;
        arg->height = 480;
        break;
    case nw::eftdemo::INIT_SETTINGS_RESOLUTION_1280_720:
        arg->width  = 1280;
        arg->height = 720;
        break;
    }

    heap.Free((void*)buf);
    heap.Finalize();
#endif

#if defined(NW_PLATFORM_CAFE)
    NW_UNUSED_VARIABLE( allocator );
    NW_UNUSED_VARIABLE( arg );
#endif
}


//------------------------------------------------------------------------------
//      コンフィグファイルのセーブ
//------------------------------------------------------------------------------
void System::SaveConfigFile(    INIT_SETTINGS_FRAME_RATE frameRate,
                                INIT_SETTINGS_RESOLUTION resolution,
                                f32 r, f32 g, f32 b, f32 a )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    char s[1024];
    sprintf_s(s, 1024, "%d\n%d\n%f\n%f\n%f\n%f\n",
        frameRate,
        resolution,
        r, g, b, a);
    nw::eftvw2::FileSystem::Write( CONFIG_FILE_NAME, s, strlen(s), 1, false );
#endif
}


//------------------------------------------------------------------------------
//      ウインドウ関連
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
void System::SetupWindow()
{
    // PcSDK で作成されるウィンドウへの設定です。
    {
        m_HWnd = nw::internal::winext::ghFrameWnd;
        OS_SetWindowMessageCallback( reinterpret_cast<WindowMessageCallback>( System::MsgProc ) );

        DWORD style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_CLIPCHILDREN;
        RECT rect;
        s32 width;
        s32 height;

        SetRect( &rect, 0, 0, m_Arg.width, m_Arg.height );
        AdjustWindowRect( &rect, style, FALSE );

        width = rect.right - rect.left;
        height = rect.bottom - rect.top;

        SetWindowPos( reinterpret_cast<HWND>( m_HWnd ), HWND_TOP, 0, 0, width, height, SWP_NOMOVE );
    }

    // フレーム管理初期化。
    {
        SetFps( m_Arg.fps );

        m_LastUpdateTime = nw::ut::Tick::GetSystemCurrent();
        m_LastUpdateTime -= nw::ut::Tick(m_FrameTime);
    }
}

//------------------------------------------------------------------------------
//      MsgProc
//------------------------------------------------------------------------------
_W64 long System::MsgProc( void* hWnd, unsigned int msg, _W64 unsigned wParam, _W64 long lParam, BOOL *handled )
{
    NW_ASSERT_NOT_NULL( sInstance );

    return sInstance->MsgProcImpl( hWnd, msg, wParam, lParam, handled );
}

_W64 long System::MsgProcImpl( void* hWnd, unsigned int msg, _W64 unsigned int wParam, _W64 long lParam, BOOL *handled )
{
    NW_UNUSED_VARIABLE( hWnd );
    NW_UNUSED_VARIABLE( lParam );

    switch ( msg )
    {
    case WM_CREATE:
        break;

    case WM_CLOSE:
        // ×ボタン押下.
        m_Exit = true;
        *handled = TRUE;
        return 1;

    case WM_CHAR:
        if ( wParam == 0x11 )
        {
            // Ctrl + Q キー押下.
            m_Exit = true;
            *handled = TRUE;
            return 1;
        }
        break;

    case WM_SYSKEYDOWN:
        *handled = TRUE;
        return 1;
    }

    return 0;
}


//------------------------------------------------------------------------------
//      フレームバッファ生成
//------------------------------------------------------------------------------
void System::CreateFrameBuffer( const nw::math::VEC2& frameBufferSize )
{
    NW_UNUSED_VARIABLE( frameBufferSize );

    // デフォルトのフレームバッファを設定します。
    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

    gfx->LockDrawContext();
    {
        f32 width  = static_cast<f32>( m_Arg.width );
        f32 height = static_cast<f32>( m_Arg.height );

        m_ScanBuffer.Initialize(     NULL, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGBA32 );
        m_ScanBufferSRGB.Initialize( NULL, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_SRGB );

        m_CurrentScanBuffer = &m_ScanBuffer;

        // 初期設定。
        bool vsync_ok = SetWaitVBlankInterval( m_Arg.waitVBlank );
        if ( !vsync_ok )
        {
            m_Arg.waitVBlank = 0;
        }
        NW_GL_ASSERT();

    }
    gfx->UnlockDrawContext();
}
#endif


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
#ifdef NW_PLATFORM_CAFE
void System::SetBlack( bool is_black )
{
    if (m_Flags.IsMaskOn( FLAG_IS_BLACK ) != is_black)
    {
        GX2SetTVEnable(static_cast<GX2Boolean>( ! is_black));
        m_Flags.ChangeMask( FLAG_IS_BLACK, is_black );
    }
}
#endif


//---------------------------------------------------------------------------
//! @brief        スキャンバッファの切り替え処理を行います。
//---------------------------------------------------------------------------
void System::SetScanBuffer( bool linearScanBuffer )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    if ( linearScanBuffer )
    {
        m_CurrentScanBuffer = &m_ScanBufferSRGB;
    }
    else
    {
        m_CurrentScanBuffer = &m_ScanBuffer;
    }
#endif
#ifdef NW_PLATFORM_CAFE
    if ( linearScanBuffer )
    {
        GX2SetTVBuffer( m_LinearScanBufferPtr, m_LinearScanSize, GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, GX2_BUFFERING_DOUBLE );
    }
    else
    {
        GX2SetTVBuffer( m_ScanBufferPtr, m_ScanSize, GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_BUFFERING_DOUBLE );
    }
#endif
}

} // namespace eftdemo
} // namespace nw
