#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import os
import re
import argparse
import multiprocessing
import codecs
import StringIO

from subprocess import call

# cpplint をモジュールとしてインポート
_APP_ROOT    = os.path.dirname( os.path.abspath(__file__) )
_SIGLO_ROOT  = os.path.normpath( _APP_ROOT + '/../../../..' )
_WORK_ROOT   = _SIGLO_ROOT
_CPPLINT_DIR = os.path.normpath( _SIGLO_ROOT + '/Externals/google-styleguide/cpplint' )
sys.path.append( _CPPLINT_DIR )
import cpplint

# 規約違反の適用例外モジュールを定義するリストのパスです。
_LIST_DIR = os.path.normpath( _SIGLO_ROOT + '/Common/Build/CodeInspection' )
_EXCLUDE_PATH_LIST = os.path.normpath( _LIST_DIR + '/ExcludePath.list' )
_EXCLUDE_PATH_EXHAUSTIVE_LIST = os.path.normpath( _LIST_DIR + '/ExcludePath.exhaustive.list' )
_EXCLUDE_PATH_COPYRIGHT_LIST = os.path.normpath( _LIST_DIR + '/ExcludePathForCopyright.list' )

# 有効な拡張子のリストです。
_VALID_EXTENSIONS = set(['.cc', '.h', '.cpp', '.cu', '.cuh', '.hpp'])

# コピーライトのチェックのみを行う拡張子のリストです。
_COPYRIGHT_CHECK_EXTENSIONS = set(['.dts', '.dtsi'])

# 自動生成ファイルの命名パターンです。
_AUTO_GENERATED_FILE_PATTENRS = [
  re.compile(r".+\.sfdl\.h"),
  re.compile(r".+_Result\.private\.h"),
  re.compile(r".+_Result\.public\.h"),
]

# 配列 target を n 個ずつの要素に分割した配列を返す
def split_array(target, n):
    for i in xrange(0, len(target), n):
        yield target[i:i+n]

def create_exclude_file_list(path):

    try:
        raw_body  = codecs.open(path, 'r', 'utf_8', 'strict').read()
    except:
        print "cannot open the exclude list file."
        quit()

    if len(raw_body) > 0 and raw_body[0] == u'\ufeff':
        body = raw_body[1:]
    else:
        body = raw_body
    lines = body.splitlines()

    files = []
    for line in lines:
        files.append(os.path.normpath(_WORK_ROOT + '/' + line))
    return files

def is_ignored_source(path, ignores):
    name, ext = os.path.splitext(path)
    if ext not in _VALID_EXTENSIONS:
        if ext not in _COPYRIGHT_CHECK_EXTENSIONS:
            return True
    for pattern in _AUTO_GENERATED_FILE_PATTENRS:
        if pattern.match(path):
            return True
    for ignore in ignores:
        if path.startswith(ignore):
            return True
    return False

def is_check_copyright_source(path):
    name, ext = os.path.splitext(path)
    if ext in _COPYRIGHT_CHECK_EXTENSIONS:
        return True
    return False

# multiprocessing に渡す処理関数
def cpplint_worker(filenames, copyright_ignores):
  ignores = create_exclude_file_list(_EXCLUDE_PATH_COPYRIGHT_LIST)

  # stderr 出力を string として得る
  sys.stderr = StringIO.StringIO()

  cpplint._cpplint_state.ResetErrorCounts()
  for filename in filenames:
    if is_ignored_source(filename, copyright_ignores):
      cpplint._SetFilters('-style/copyright')
    elif is_check_copyright_source(filename):
      cpplint._valid_extensions |= set(['dts', 'dtsi'])
      cpplint._SetFilters('-analysis, -branch, -c++11, -class, -extension, -extra, -file, -function, -impl, -lint, -name, -others, -preprocessor, -style, -type, -variable, +style/copyright')
    else:
      cpplint._SetFilters('')
    cpplint.ProcessFile(filename, 1)

  # stderr とエラー数を親プロセスに返す
  return [sys.stderr.getvalue(), cpplint._cpplint_state.error_count]


def cpplint_parallel(filenames, copyright_ignores, unit_count = 10):
    # 物理コア数分の子プロセスを起動する
    cpu_count = multiprocessing.cpu_count()
    pool = multiprocessing.Pool(cpu_count)

    # unit_count 個のファイルずつ子プロセスに処理を依頼する
    results = []
    for file_list_partial in split_array(filenames, unit_count):
        results.append(pool.apply_async(cpplint_worker, (file_list_partial, copyright_ignores)))

    # 依頼した順番で処理の完了を待つ
    total_count = 0
    for result in results:
        while True:
            try:
                # 先頭の結果を取得する（ポーリングしないと Ctrl+C で止められない）
                output, count = result.get(3)
                # 順番を守りながら stderr に出力
                sys.stderr.write("".join(output))
                total_count += count
                break
            except multiprocessing.TimeoutError:
                # get がタイムアウトしたら即リトライ
                pass

    # cpplint を使って結果を出力
    cpplint._cpplint_state.error_count = total_count
    cpplint._cpplint_state.PrintErrorCounts()
    return total_count

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Cpp coding rule checker.')

    # 互換性のために残しておく
    parser.add_argument('-C', action='append')
    parser.add_argument('-d', action='append')

    parser.add_argument('--excludedir', action='append')
    parser.add_argument('--exhaustive', action='store_true')
    parser.add_argument('--version', action='version', version='%(prog)s 1.5')
    parser.add_argument('--basepath')

    parser.add_argument('targets', nargs='*', default=[os.getcwd()])

    args = parser.parse_args()

    if args.basepath:
        _WORK_ROOT = os.path.normpath(args.basepath)

    if args.exhaustive:
        ignores = create_exclude_file_list(_EXCLUDE_PATH_EXHAUSTIVE_LIST)
    else:
        ignores = create_exclude_file_list(_EXCLUDE_PATH_LIST)
    if args.excludedir:
        for dir in args.excludedir:
            ignores.append(os.path.realpath(dir))

    copyright_ignores = create_exclude_file_list(_EXCLUDE_PATH_COPYRIGHT_LIST)

    specific_path = []
    if not args.C is None:
        for arg in args.C:
            dst = os.path.realpath(arg)
            specific_path.append(dst)
    if not args.d is None:
        for arg in args.d:
            dst = os.path.realpath(arg)
            specific_path.append(dst)
    for arg in args.targets:
        dst = os.path.realpath(arg)
        specific_path.append(dst)

    print "targets: " + str(specific_path)

    file_list = []

    for rootPath in specific_path:
        if os.path.isfile(rootPath):
            path = os.path.normpath(rootPath)
            if not is_ignored_source(path, ignores):
                file_list.append(path)
        else:
            for root, dirs, files in os.walk(rootPath):
                for file in files:
                    path = os.path.normpath(os.path.join(root,file))
                    if not is_ignored_source(path, ignores):
                        file_list.append(path)

    sys.stderr = codecs.StreamReaderWriter(sys.stderr,
                                           codecs.getreader('utf8'),
                                           codecs.getwriter('utf8'),
                                           'replace')

    total_count = cpplint_parallel(file_list, copyright_ignores)
    sys.exit(total_count)
