﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.IO;
using System.Text;

namespace NintendoWare.Spy
{
    /// <summary>
    /// ストリームの入出力をログファイルに記録します。
    /// </summary>
    public class LoggingStream : Stream
    {
        private readonly Stream _baseStream;
        private readonly TextWriter _textWriter;
        private readonly bool _disposeBaseStream;
        private bool _disposed;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="baseStream">ログを記録する対象のストリーム。</param>
        /// <param name="logFilePath">ログファイルのパス。</param>
        /// <param name="disposeBaseStream"><see cref="Dispose"/> 時に <paramref name="baseStream"/> も破棄するか。 </param>
        public LoggingStream(Stream baseStream, string logFilePath, bool disposeBaseStream = true)
        {
            Ensure.Argument.NotNull(baseStream, nameof(baseStream));
            Ensure.Argument.True(!string.IsNullOrWhiteSpace(logFilePath));

            _baseStream = baseStream;
            _textWriter = new StreamWriter(logFilePath);
            _disposeBaseStream = disposeBaseStream;

            _textWriter.WriteLine($"Create {DateTime.Now}");
        }

        /// <summary>
        /// ログを記録する対象のストリームを取得します。
        /// <para>
        /// <see cref="BaseStream"/> に対して直接に行った操作はログに記録されないので注意してください。
        /// </para>
        /// </summary>
        public Stream BaseStream
        {
            get { return _baseStream; }
        }

        /// <summary>
        /// ログの記録に使用する <see cref="TextWriter"/> を取得します。
        /// <para>
        /// ログファイルに情報を追加したいときに使用できます。
        /// </para>
        /// </summary>
        public TextWriter TextWriter
        {
            get { return _textWriter; }
        }

        public override bool CanRead
        {
            get { return _baseStream.CanRead; }
        }

        public override bool CanSeek
        {
            get { return _baseStream.CanSeek; }
        }

        public override bool CanWrite
        {
            get { return _baseStream.CanWrite; }
        }

        public override long Length
        {
            get { return _baseStream.Length; }
        }

        public override long Position
        {
            get { return _baseStream.Position; }
            set { _baseStream.Position = value; }
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);

            if (_disposed)
            {
                return;
            }

            _disposed = true;

            if (disposing)
            {
                try
                {
                    if (_disposeBaseStream)
                    {
                        _baseStream.Close();
                    }
                }
                finally
                {
                    _textWriter.Close();
                }
            }
        }

        public override void Flush()
        {
            _baseStream.Flush();
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            try
            {
                _textWriter.WriteLine($"Read {count}");
                var result = _baseStream.Read(buffer, offset, count);
                this.DumpBytes(buffer, offset, result);
                _textWriter.WriteLine($"<{result}>");
                return result;
            }
            catch (Exception e)
            {
                _textWriter.WriteLine($"Exception {e}");
                throw;
            }
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            try
            {
                _textWriter.WriteLine($"Seek {offset} {origin}");
                var result = _baseStream.Seek(offset, origin);
                _textWriter.WriteLine($"<{result}>");
                return result;
            }
            catch (Exception e)
            {
                _textWriter.WriteLine($"Exception {e}");
                throw;
            }
        }

        public override void SetLength(long value)
        {
            try
            {
                _textWriter.WriteLine($"SetLength {value}");
                _baseStream.SetLength(value);
            }
            catch (Exception e)
            {
                _textWriter.WriteLine($"Exception {e}");
                throw;
            }
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            try
            {
                _textWriter.WriteLine($"Write {count}");
                _baseStream.Write(buffer, offset, count);
                this.DumpBytes(buffer, offset, count);
            }
            catch (Exception e)
            {
                _textWriter.WriteLine($"Exception {e}");
                throw;
            }
        }

        private void DumpBytes(byte[] bytes, int offset, int count)
        {
            const int BytesPerLine = 16;

            for (int i = 0; i < count;)
            {
                int j = 0;
                for (; i + j < count && j < BytesPerLine; j++)
                {
                    if (j > 0)
                    {
                        _textWriter.Write(" ");

                        if (j % sizeof(int) == 0)
                        {
                            _textWriter.Write(" ");
                        }
                    }

                    _textWriter.Write("{0:X2}", bytes[offset + i + j]);
                }

                for (int k = j; k < BytesPerLine; k++)
                {
                    if (k > 0)
                    {
                        _textWriter.Write(" ");

                        if (k % sizeof(int) == 0)
                        {
                            _textWriter.Write(" ");
                        }
                    }

                    _textWriter.Write("  ");
                }

                _textWriter.Write(" : ");

                var charArray = Encoding.ASCII.GetChars(bytes, offset + i, j);
                foreach (var c in charArray)
                {
                    if (char.IsControl(c))
                    {
                        _textWriter.Write(".");
                    }
                    else
                    {
                        _textWriter.Write(c);
                    }
                }

                _textWriter.WriteLine();

                i += j;
            }
        }
    }
}
