﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace NintendoWare.Spy
{
    public static class IpcUtility
    {
        public class IpcUtilityException : ApplicationException
        {
            public IpcUtilityException(string message)
                : base(message)
            { }
        }

        /// <summary>
        /// メッセージを送信します。
        /// </summary>
        /// <param name="hWnd"></param>
        /// <param name="message"></param>
        /// <returns></returns>
        /// <exception cref="IpcUtilityException">PInvoke 呼び出しでエラーが起きた場合。</exception>
        public static int SendStringMessage(IntPtr hWnd, string message)
        {
            Ensure.Argument.NotNull(message, nameof(message));

            var hGlobals = new List<IntPtr>();
            try
            {
                Win32.COPYDATASTRUCT cds = new Win32.COPYDATASTRUCT();
                cds.dwData = IntPtr.Zero;
                cds.cbData = message.Length * sizeof(char);
                cds.lpData = Marshal.StringToHGlobalUni(message);
                hGlobals.Add(cds.lpData);

                Win32.SetLastError(0);
                int result = Win32.SendMessage(hWnd, Win32.WM.WM_COPYDATA, IntPtr.Zero, ref cds);

                if (result == Win32.FALSE)
                {
                    int win32Error = Win32.GetLastError();
                    if (win32Error != Win32.ERROR_SUCCESS)
                    {
                        throw new IpcUtilityException("Unexpected Win32 Error: " + win32Error);
                    }
                }

                return result;
            }
            finally
            {
                hGlobals.ForEach(it => Marshal.FreeHGlobal(it));
            }
        }

        public static string ReceiveStringMessage(IntPtr hWnd, int msg, IntPtr wParam, IntPtr lParam)
        {
            var cds = Marshal.PtrToStructure<Win32.COPYDATASTRUCT>(lParam);

            if (cds.cbData < 0 || cds.cbData % sizeof(char) != 0)
            {
                return null;
            }

            var length = cds.cbData / 2;
            var message = Marshal.PtrToStringUni(cds.lpData, length);

            return message;
        }
    }
}
