﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace NintendoWare.Spy
{
    /// <summary>
    /// フレーム値を管理します。
    /// </summary>
    public struct Frame : IComparable<Frame>
    {
        public static Frame Zero = new Frame(0);
        public static Frame InvalidValue = new Frame(long.MinValue);

        //-----------------------------------------------------------------

        private long _value;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">フレーム値を指定します。</param>
        public Frame(long value)
        {
            _value = value;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 数値を取得します。
        /// </summary>
        public long Value
        {
            get { return _value; }
        }

        /// <summary>
        /// 有効な数値かどうかを調べます。
        /// </summary>
        public bool IsValid
        {
            get { return _value != Frame.InvalidValue; }
        }

        //-----------------------------------------------------------------

        public override string ToString()
        {
            return this.Value.ToString();
        }

        public override bool Equals(object obj)
        {
            if (obj is Frame)
            {
                return this == (Frame)obj;
            }
            else
            {
                return false;
            }
        }

        public override int GetHashCode()
        {
            return this.Value.GetHashCode();
        }

        public static implicit operator long(Frame source)
        {
            return source._value;
        }

        public static bool operator ==(Frame left, Frame right)
        {
            return left.Value == right.Value;
        }

        public static bool operator !=(Frame left, Frame right)
        {
            return left.Value != right.Value;
        }

        public static Frame operator +(Frame left, Frame right)
        {
            if (left == Frame.InvalidValue || right == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            else
            {
                return new Frame(left.Value + right.Value);
            }
        }

        public static Frame operator -(Frame left, Frame right)
        {
            if (left == Frame.InvalidValue || right == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            else
            {
                return new Frame(left.Value - right.Value);
            }
        }

        public static Frame operator +(Frame left, long right)
        {
            if (left == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            else
            {
                return new Frame(left.Value + right);
            }
        }

        public static Frame operator -(Frame left, long right)
        {
            if (left == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            else
            {
                return new Frame(left.Value - right);
            }
        }

        public int CompareTo(Frame other)
        {
            return this.Value.CompareTo(other.Value);
        }

        public static Frame Parse(string label)
        {
            try
            {
                if (label == "?")
                {
                    return InvalidValue;
                }
                else
                {
                    return new Frame(long.Parse(label));
                }
            }
            catch
            {
                return InvalidValue;
            }
        }

        public string ToLabelString()
        {
            return (this.IsValid) ? this.ToString() : "?";
        }
    }
}
