﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace NintendoWare.Spy
{
    /// <summary>
    /// 波形データ Spy モデルです。
    /// </summary>
    public sealed class WaveformSpyModel : SpyModel, ILoudnessProvider
    {
        public const int BitsPerSample = 16;
        public const int ChannelCountMax = 6;

        public const int MainOutChannelCount = 6;

        public const string MainOutName = "Main";

        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct WaveformMetadataPacket {
        ///     u8 packetType = PacketType.WaveformMetadata;
        ///     u8 sampleFormat = SampleFormatType.PcmInt16;
        ///     u8 channelCount;
        ///     u8 reserved;
        ///     s32 sampleRate;
        ///     s32 sampleCountPerAudioFrame;
        ///     ChannelInfo channelInfos[channelCount];
        /// }
        ///
        /// struct WaveformPacket {
        ///     u8 packetType = PacketType.Waveform;
        ///     u8 reserved;
        ///     u16 sampleCount;
        ///     s32 sampleIndexOnTick;
        ///     u16 sampleData[channelCount][sampleCount];
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// バージョン 0.1.1.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct DeviceOutputVolumePacket {
        ///     u8 packetType = PacketType.DeviceOutputVolume;
        ///     u8 reserved[3];
        ///     f32 volume; // 0.0 ~ 128.0
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_1_0 = new Version(0, 1, 1, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        // 最新の SpyDataBlock がファイルに保存されるまで、一時的に保持するキャッシュの最大数。
        private const int MaxDataBlockCacheCount = 100;

        // Waveform のキャッシュするインスタンスの最大数です。
        private const int MaxWaveformCacheCount = 100;

        // LoudnessUpdateEvent の通知間隔です(ミリ秒)。
        private const int LoudnessUpdateEventInterval = 500;

        private readonly PeakDataManager _peakDataManager = new PeakDataManager();

        public PeakDataManager PeakDataManager => _peakDataManager;

        //-----------------------------------------------------------------

        private readonly List<LoudnessInfo> _loudnessList = new List<LoudnessInfo>();

        private SpyTime _beginTime = SpyTime.InvalidValue;
        private SpyTime _endTime = SpyTime.InvalidValue;
        private int _waveformCount;
        private int _loudnessCount;

        private bool _isWaveformMetadataReady;
        private int _sampleRate;
        private bool _isLoudnessAvailable;
        private int _loudnessStepCount;

        private readonly LoudnessAccumulator _loudnessAccumulator = new LoudnessAccumulator();

        public delegate void UpdateEventHandler(object sender, EventArgs e);
        public event UpdateEventHandler UpdateEvent;

        /// <summary>
        /// バックグラウンドで新しいラウドネス情報が計算されたときに発生するイベントです。
        /// このイベントは非UIスレッドで実行されます。
        /// </summary>
        public event EventHandler LoudnessUpdateEvent;

        private DateTime _loudnessUpdateEventNotifyTime = DateTime.Now;

        // 最新の SpyDataBlock をキャッシュします。
        // キーは SpyDataBlock.ID です。
        private readonly CacheDictionary<long, SpyDataBlock> _dataBlockCache = new CacheDictionary<long, SpyDataBlock>(MaxDataBlockCacheCount);

        // 受信した波形データの概要を保持します。
        private readonly List<WaveformInfo> _waveformInfos = new List<WaveformInfo>();

        // Waveform のインスタンスをキャッシュします。
        // キーは waveformIndex になります。
        private readonly CacheDictionary<int, Waveform> _waveformCache = new CacheDictionary<int, Waveform>(MaxWaveformCacheCount);

        private readonly object _getWaveformLockObject = new object();

        private readonly object _infoLockObject = new object();

        /// <summary>
        /// ラウドネスを計算するTaskです。
        /// </summary>
        private Task _calcLoudnessTask;

        /// <summary>
        /// ラウドネス計算タスクを終了させるときに使います。
        /// </summary>
        private readonly CancellationTokenSource _calcLoudnessCancellationTokenSource = new CancellationTokenSource();

        /// <summary>
        /// ラウドネス計算要求の待ち行列です。
        /// </summary>
        private readonly ConcurrentQueue<CalcLoudnessRequest> _calcLoudnessRequests = new ConcurrentQueue<CalcLoudnessRequest>();

        /// <summary>
        /// ラウドネス計算の結果です。UIスレッドでloudnessListに統合されるまで蓄積されます。
        /// </summary>
        private readonly ConcurrentQueue<CalcLoudnessResult> _calcLoudnessResults = new ConcurrentQueue<CalcLoudnessResult>();

        private readonly object _calcLoudnessLock = new object();

        /// <summary>
        /// オーディオデバイス(TV)の出力ボリュームです(0.0 ~ 128.0)。
        /// 波形サンプルに積算されます。
        /// </summary>
        private float _deviceOutputVolume = 1.0f;

        //-----------------------------------------------------------------

        private class CacheDictionary<TKey, TValue>
        {
            private readonly Dictionary<TKey, TValue> _dictionary = new Dictionary<TKey, TValue>();
            private readonly List<TKey> _list = new List<TKey>();
            private int _capacity;
            private object _lockObject = new object();

            public CacheDictionary(int capacity)
            {
                Ensure.Argument.True(capacity > 0);
                _capacity = capacity;
            }

            public void Add(TKey key, TValue value)
            {
                lock (_lockObject)
                {
                    _dictionary.Add(key, value);
                    _list.Add(key);

                    if (_list.Count > _capacity)
                    {
                        _dictionary.Remove(_list[0]);
                        _list.RemoveAt(0);
                    }
                }
            }

            public TValue GetValue(TKey key)
            {
                lock (_lockObject)
                {
                    TValue value;
                    _dictionary.TryGetValue(key, out value);
                    return value;
                }
            }
        }

        /// <summary>
        /// 受信した波形データの概要情報です。
        /// 波形データそのもの（サンプル）はサイズが大きいので、必要に応じて GetWaveform() で取得します。
        /// </summary>
        private class WaveformInfo
        {
            /// <summary>
            /// Waveform の０から始まる通し番号です。
            /// </summary>
            public int WaveformIndex { get; internal set; }

            /// <summary>
            /// 全波形データのサンプルにおける、Waveform の先頭サンプルのインデックスです。
            /// </summary>
            public long SampleIndex { get; }

            /// <summary>
            /// Waveform が持つサンプル数です。
            /// </summary>
            public int SampleCount { get; }

            /// <summary>
            /// サンプル値に積算するボリュームです。
            /// </summary>
            public float Volume { get; }

            /// <summary>
            /// DataBlockID です。
            /// </summary>
            public long DataBlockID { get; }

            public WaveformInfo(
                long sampleIndex,
                int sampleCount,
                float volume,
                long dataBlockID)
            {
                this.SampleIndex = sampleIndex;
                this.SampleCount = sampleCount;
                this.Volume = volume;
                this.DataBlockID = dataBlockID;
            }
        }

        public enum ChannelIndex : int
        {
            MainFrontLeft,
            MainFrontRight,
            MainRearLeft,
            MainRearRight,
            MainFrontCenter,
            MainLfe,
            DrcLeft,
            DrcRight,
            Max,
            Invalid = -1
        }

        /// <summary>
        /// サンプルデータです。
        /// </summary>
        public class SampleData
        {
            public short[][] Samples { get; }

            public SampleData(int sampleCount)
            {
                Samples = new short[ChannelCountMax][].Populate(_ => new short[sampleCount]);
            }
        }

        /// <summary>
        /// チャンネルごとの波形の概要情報です。
        /// </summary>
        public class ChannelInfo
        {
            public int MaxSampleValue { get; set; }
            public int MinSampleValue { get; set; }
            public float TotalMeanSquare { get; set; }

            public ChannelInfo()
            {
            }
        }

        /// <summary>
        /// 波形データです。
        /// </summary>
        public class Waveform
        {
            /// <summary>
            /// 波形データの再生開始時間です。
            /// </summary>
            public SpyGlobalTime Time { get; set; } = SpyGlobalTime.InvalidValue;

            /// <summary>
            /// 波形データの０から始まる通し番号です。
            /// </summary>
            public int WaveformIndex { get; set; }

            /// <summary>
            /// 全波形データのサンプルにおける、波形の先頭サンプルのインデックスです。
            /// WaveformSpyModel.BeginTime が全波形データの再生開始時間を表しています。
            /// </summary>
            public long SampleIndex { get; set; }

            /// <summary>
            /// 波形データに含まれるサンプルの数です。
            /// </summary>
            public int SampleCount { get; }

            /// <summary>
            /// サンプルレート（1秒当たりのサンプル数）です。
            /// WaveformSpyModel.SampleRate と同じです。
            /// すべての波形データで一定です。
            /// </summary>
            public int SampleRate { get; }

            /// <summary>
            /// サンプルデータです。
            /// </summary>
            public SampleData SampleData { get; }

            /// <summary>
            /// チャンネルごとの概要情報です。
            /// </summary>
            public ChannelInfo[] ChannelInfoList { get; } =
                new ChannelInfo[ChannelCountMax].Populate(_ => new ChannelInfo());

            public Waveform(int sampleRate, int sampleCount)
            {
                SampleRate = sampleRate;
                SampleCount = sampleCount;
                SampleData = new SampleData(sampleCount);
            }
        }

        public class LoudnessInfo
        {
            public long SampleIndex { get; set; }
            public int LoudnessIndex { get; set; }
            public float MomentaryLoudnessValue { get; set; }
            public float ShortTermLoudnessValue { get; set; }
            public float AbsGatedLoudnessValue { get; set; }
            public float RelGatedLoudnessValue { get; set; }
            public ChannelLoudnessInfo[] Channels { get; } =
                new ChannelLoudnessInfo[ChannelCountMax].Populate(_ => new ChannelLoudnessInfo());

            public LoudnessInfo()
            {
            }
        }

        public class ChannelLoudnessInfo
        {
            public float PeakValue { get; set; }
            public float TruePeakValue { get; set; }
            public float RmsValue { get; set; }

            public ChannelLoudnessInfo()
            {
            }
        }

        /// <summary>
        /// ラウドネス情報をバックグラウンドで計算するタスクに与えるリクエストです。
        /// </summary>
        private class CalcLoudnessRequest
        {
            /// <summary>
            /// 波形データです。
            /// </summary>
            public Waveform Waveform { get; }

            public CalcLoudnessRequest(Waveform waveform)
            {
                this.Waveform = waveform;
            }
        }

        /// <summary>
        /// ラウドネス情報のバックグラウンドでの計算結果です。
        /// </summary>
        private class CalcLoudnessResult
        {
            /// <summary>
            /// 計算結果です。
            /// </summary>
            public LoudnessInfo LoudnessInfo { get; }

            public CalcLoudnessResult(LoudnessInfo loudnessInfo)
            {
                this.LoudnessInfo = loudnessInfo;
            }
        }

        private enum PacketType : byte
        {
            WaveformMetadata = 1,
            Waveform,
            DeviceOutputVolume,
        }

        private enum SampleFormatType : byte
        {
            PcmIn16 = 2,
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 波形の概要情報を受信したかを示します。
        /// </summary>
        public bool IsWaveformMetadataReady
        {
            get { return _isWaveformMetadataReady; }
            private set { this.SetPropertyValue(ref _isWaveformMetadataReady, value); }
        }

        /// <summary>
        /// サンプルレートです。
        /// IsWaveformMetadataReady プロパティが true でないと取得できません。
        /// </summary>
        public int SampleRate
        {
            get
            {
                Ensure.Operation.True(IsWaveformMetadataReady);
                return _sampleRate;
            }
            private set
            {
                _sampleRate = value;
            }
        }

        /// <summary>
        /// １サンプルあたりの秒数(SampleRate の逆数)です。
        /// </summary>
        private double SecondPerSample { get; set; }

        /// <summary>
        /// サンプルの型および格納形式の情報です。
        /// </summary>
        private SampleFormatType SampleFormat { get; set; }

        /// <summary>
        /// 波形データのチャンネル数です。
        /// </summary>
        private int ChannelCount { get; set; }

        /// <summary>
        /// オーディオフレームあたりのサンプル数です。
        /// （nn::audio::AudioRenderer の設定値）
        /// </summary>
        private int SampleCountPerAudioFrame { get; set; }

        /// <summary>
        /// 波形データのチャンネルの格納順です。
        /// </summary>
        private ChannelIndex[] ChannelIndices { get; set; }

        /// <summary>
        /// 波形データのある最初の時間です。
        /// 波形データが無いときは SpyTime.InvalidValue となります。
        /// </summary>
        public SpyTime BeginTime
        {
            get { return _beginTime; }
            private set { this.SetPropertyValue(ref _beginTime, value); }
        }

        /// <summary>
        /// 波形データのある最後の時間です。
        /// 波形データが無いときは SpyTime.InvalidValue となります。
        /// </summary>
        public SpyTime EndTime
        {
            get { return _endTime; }
            private set { this.SetPropertyValue(ref _endTime, value); }
        }

        /// <summary>
        /// 波形データの総数です。
        /// </summary>
        public int WaveformCount
        {
            get { return _waveformCount; }
            private set { this.SetPropertyValue(ref _waveformCount, value); }
        }

        /// <summary>
        /// ラウドネスが計算されるかを示します。
        /// </summary>
        public bool IsLoudnessAvailable
        {
            get { return _isLoudnessAvailable; }
            private set { this.SetPropertyValue(ref _isLoudnessAvailable, value); }
        }

        /// <summary>
        /// LoudnessInfo １つあたりの波形のサンプル数です。
        /// </summary>
        public int LoudnessStepSampleCount
        {
            get { return _loudnessStepCount; }
            private set { this.SetPropertyValue(ref _loudnessStepCount, value); }
        }

        /// <summary>
        /// ラウドネスデータの総数です。
        /// </summary>
        public int LoudnessCount
        {
            get { return _loudnessCount; }
            private set { this.SetPropertyValue(ref _loudnessCount, value); }
        }

        /// <summary>
        /// 受信した総サンプル数です。欠落したサンプル数も含みます。
        /// </summary>
        private long TotalSampleCount { get; set; }

        /// <summary>
        /// 受信中に例外が発生したことを示します。
        /// 一度エラーが発生すると、以後の受信データは無視されます。
        /// </summary>
        private bool Error { get; set; }

        //-----------------------------------------------------------------

        public WaveformSpyModel()
        {
        }

        protected override void Dispose(bool disposing)
        {
            _calcLoudnessCancellationTokenSource.Cancel();
            _calcLoudnessTask = null;
            base.Dispose(disposing);
        }

        /// <summary>
        /// 指定時間を区間に含む波形データのインデックスを取得します。
        /// </summary>
        /// <param name="time"></param>
        /// <returns>
        /// time を区間に含む波形データがあるときは、そのフレームインデックスを返します。
        /// time を区間に含む波形データが無いときは、
        /// それ以後の最初の波形データのインデックスの１の補数(負の値)を返します。
        /// time 以後の波形データが無いときは、WaveformCount プロパティの１の補数(負の値)を返します。
        /// </returns>
        public int GetWaveformIndex(SpyGlobalTime time)
        {
            int waveformIndex = 0;
            long sampleIndex = 0;
            WaveformInfo waveformInfo = null;

            lock (_infoLockObject)
            {
                Assertion.Operation.True(this.WaveformCount == _waveformInfos.Count);

                if (this.WaveformCount == 0)
                {
                    return ~0;
                }

                Assertion.Operation.True(this.BeginTime.Timestamp != SpyGlobalTime.InvalidValue);
                if (time < this.BeginTime.Timestamp)
                {
                    return ~0;
                }

                // 指定時刻、またはそれ以後の最初のサンプルのインデックスを得ます。
                sampleIndex = (long)Math.Ceiling((time - this.BeginTime.Timestamp).Seconds * this.SampleRate);

                // sampleIndex を区間に含むフレーム、またはそれ以降のフレームのインデックスを得ます。
                waveformIndex = BinarySearchUtility.BinarySearch(_waveformInfos, sampleIndex, it => it.SampleIndex);
                if (waveformIndex < 0)
                {
                    waveformIndex = ~waveformIndex - 1;

                    if (waveformIndex < 0)
                    {
                        return ~0;
                    }
                }

                waveformInfo = _waveformInfos[waveformIndex];
                Assertion.Operation.True(waveformInfo.SampleIndex <= sampleIndex);
            }

            // time を区間に含まない場合は次のフレームのインデックスの１の補数を返します。
            if (waveformInfo.SampleIndex + waveformInfo.SampleCount <= sampleIndex)
            {
                return ~(waveformIndex + 1);
            }

            return waveformIndex;
        }

        /// <summary>
        /// 波形データを取得します。
        /// この関数はファイルからの読み込みを伴うため時間がかかります。
        /// 非同期処理を検討してください。
        /// </summary>
        /// <param name="waveformIndex"></param>
        /// <returns>情報が無い場合は null を返します。</returns>
        public Waveform GetWaveform(int waveformIndex)
        {
            lock (_getWaveformLockObject)
            {
                Waveform waveform = _waveformCache.GetValue(waveformIndex);
                if (waveform != null)
                {
                    return waveform;
                }

                WaveformInfo waveformInfo = null;
                lock (_infoLockObject)
                {
                    Assertion.Operation.True(this.WaveformCount == _waveformInfos.Count);

                    if (waveformIndex < 0 || this.WaveformCount <= waveformIndex)
                    {
                        return null;
                    }

                    waveformInfo = _waveformInfos[waveformIndex];
                }

                var dataBlock = _dataBlockCache.GetValue(waveformInfo.DataBlockID);
                if (dataBlock != null)
                {
                    var reader = CreateDataReader(dataBlock);
                    if (reader != null)
                    {
                        waveform = ReadWaveform(reader, waveformInfo);
                    }
                }

                if (waveform == null)
                {
                    waveform = ReadWaveform(waveformInfo);
                }

                if (waveform == null)
                {
                    return null;
                }

                _waveformCache.Add(waveformIndex, waveform);

                return waveform;
            }
        }

        private Waveform ReadWaveform(WaveformInfo waveformInfo)
        {
            BinaryReader reader = CreateDataReader(ReadData(waveformInfo.DataBlockID));
            if (reader == null)
            {
                return null;
            }

            var waveform = ReadWaveform(reader, waveformInfo);

            return waveform;
        }

        private Waveform ReadWaveform(BinaryReader reader, WaveformInfo waveformInfo)
        {
            reader.ReadByte(); // packetType
            reader.ReadByte(); // reserved
            var sampleCount = reader.ReadInt16();
            int sampleIndexOnTick = reader.ReadInt32();

            var waveform = new Waveform(this.SampleRate, sampleCount);
            waveform.Time = this.BeginTime.Timestamp + SpyGlobalTime.FromSeconds(waveformInfo.SampleIndex * this.SecondPerSample);
            waveform.WaveformIndex = waveformInfo.WaveformIndex;
            waveform.SampleIndex = waveformInfo.SampleIndex;
            SampleData sampleData = waveform.SampleData;

            foreach (var channelIndex in this.ChannelIndices)
            {
                if (channelIndex == ChannelIndex.Invalid)
                {
                    continue;
                }

                var samples = sampleData.Samples[(int)channelIndex];
                for (int index = 0; index < sampleCount; index++)
                {
                    samples[index] = (short)MathUtility.Clamp(reader.ReadInt16() * waveformInfo.Volume, short.MinValue, short.MaxValue);
                }
            }

            for (int channelIndex = 0; channelIndex < ChannelCountMax; channelIndex++)
            {
                ChannelInfo channelInfo = waveform.ChannelInfoList[channelIndex];

                channelInfo.MaxSampleValue = int.MinValue;
                channelInfo.MinSampleValue = int.MaxValue;
                channelInfo.TotalMeanSquare = 0;

                for (int index = 0; index < sampleCount; index++)
                {
                    int sampleValue = (int)sampleData.Samples[channelIndex][index];
                    channelInfo.MaxSampleValue = Math.Max(channelInfo.MaxSampleValue, sampleValue);
                    channelInfo.MinSampleValue = Math.Min(channelInfo.MinSampleValue, sampleValue);
                    channelInfo.TotalMeanSquare += sampleValue * sampleValue;
                }
            }

            return waveform;
        }

        /// <summary>
        /// バックグラウンドで計算したラウドネス情報を取得可能にします。
        /// </summary>
        public void UpdateLoudness()
        {
            CalcLoudnessResult result;
            while (_calcLoudnessResults.TryDequeue(out result))
            {
                var loudnessInfo = result.LoudnessInfo;
                loudnessInfo.LoudnessIndex = _loudnessList.Count;
                _loudnessList.Add(loudnessInfo);
            }

            this.LoudnessCount = _loudnessList.Count;
        }

        /// <summary>
        /// ラウドネス情報を取得します。
        /// </summary>
        /// <param name="loudnessIndex"></param>
        /// <returns>情報が無い場合は null を返します。</returns>
        public LoudnessInfo GetLoudness(int loudnessIndex)
        {
            if (loudnessIndex < 0 || _loudnessList.Count <= loudnessIndex)
            {
                return null;
            }
            else
            {
                return _loudnessList[loudnessIndex];
            }
        }

        private SpyTime GetTime(SpyGlobalTime timestamp)
        {
            var belongingFrame = GetBelongingFrame(timestamp);
            return new SpyTime(timestamp, belongingFrame);
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.Error)
            {
                return;
            }

            if (this.DataVersion >= VersionUnexpected)
            {
                this.Error = true;
                return;
            }

            try
            {
                var reader = CreateDataReader(dataBlock);
                var packetType = (PacketType)reader.ReadByte();

                switch (packetType)
                {
                    case PacketType.WaveformMetadata:
                        OnPushWaveformMetadataPacket(dataBlock, reader);
                        break;

                    case PacketType.Waveform:
                        OnPushWaveformPacket(dataBlock, reader);
                        break;

                    case PacketType.DeviceOutputVolume:
                        OnPushDeviceOutputVolume(dataBlock, reader);
                        break;

                    default:
                        break;
                }
            }
            catch
            {
                this.Error = true;
            }
        }

        private void OnPushWaveformMetadataPacket(SpyDataBlock dataBlock, BinaryReader reader)
        {
            var sampleFormat = (SampleFormatType)reader.ReadByte();
            var channelCount = reader.ReadByte();
            reader.ReadByte(); // reserved
            var sampleRate = reader.ReadInt32();
            var sampleCountPerAudioFrame = reader.ReadInt32();
            var channelIndices = new ChannelIndex[channelCount];
            if (channelCount > 0)
            {
                for (int i = 0; i < channelCount; ++i)
                {
                    var channelType = reader.ReadByte();
                    reader.ReadByte(); // reserved
                    reader.ReadByte(); // reserved
                    reader.ReadByte(); // reserved

                    if (channelType < (int)ChannelIndex.Max)
                    {
                        channelIndices[i] = (ChannelIndex)channelType;
                    }
                    else
                    {
                        channelIndices[i] = ChannelIndex.Invalid;
                    }
                }
            }

            reader.Close();

            if (!this.IsWaveformMetadataReady)
            {
                this.IsWaveformMetadataReady = true;

                this.SampleFormat = sampleFormat;
                this.ChannelCount = channelCount;
                this.SampleRate = sampleRate;
                this.SecondPerSample = 1.0 / sampleRate;
                this.SampleCountPerAudioFrame = sampleCountPerAudioFrame;
                this.ChannelIndices = channelIndices;

                Loudness.Parameter parameter;
                if (Loudness.Parameters.TryGetValue(this.SampleRate, out parameter))
                {
                    this.IsLoudnessAvailable = true;
                    this.LoudnessStepSampleCount = parameter.StepSampleCount;
                    _loudnessAccumulator.SetParameter(parameter);
                }
                else
                {
                    this.IsLoudnessAvailable = false;
                }

                _peakDataManager.Initialize(this.SampleRate, this.ChannelCount);
            }
            else
            {
                Ensure.Operation.True(this.SampleFormat == sampleFormat);
                Ensure.Operation.True(this.ChannelCount == channelCount);
                Ensure.Operation.True(this.SampleRate == sampleRate);
                Ensure.Operation.True(this.SampleCountPerAudioFrame == sampleCountPerAudioFrame);
                Ensure.Operation.True(this.ChannelIndices.Length == channelIndices.Length);
                Ensure.Operation.True(this.ChannelIndices.SequenceEqual(channelIndices));
            }
        }

        private void OnPushWaveformPacket(SpyDataBlock dataBlock, BinaryReader reader)
        {
            // データファイルに保存されたパケットが読み出し可能になるまでキャッシュする。
            _dataBlockCache.Add(dataBlock.ID, dataBlock);

            reader.ReadByte(); // reserved
            int sampleCount = reader.ReadUInt16();
            int sampleIndexOnTick = reader.ReadInt32();

            var beginTime = dataBlock.Timestamp - SpyGlobalTime.FromSeconds(sampleIndexOnTick * this.SecondPerSample);
            if (this.BeginTime.Timestamp == SpyGlobalTime.InvalidValue)
            {
                this.BeginTime = this.GetTime(beginTime);
                this.EndTime = this.BeginTime;
                this.TotalSampleCount = 0;
            }
            else
            {
                // 波形の開始時刻を推定します。
                //
                // dataBlock.Timestamp には nn::audio::RequestUpdateAudioRenderer() が
                // 呼ばれた時刻が与えられますが、
                // 実際の再生時刻とは±１オーディオフレーム程度の誤差があります。
                //
                // そこで、最初に受信した波形の時刻と、この波形の時刻を比較したとき、
                // ２オーディオフレーム以上の遅れがあったときにデータをロスしたと判断します。

                var newIndex = (long)((beginTime - this.BeginTime.Timestamp).Seconds * this.SampleRate);
                newIndex -= newIndex % this.SampleCountPerAudioFrame;
                if (newIndex >= this.TotalSampleCount + 2 * this.SampleCountPerAudioFrame)
                {
                    this.TotalSampleCount = newIndex;
                }

                beginTime = this.BeginTime.Timestamp + SpyGlobalTime.FromSeconds(this.TotalSampleCount * this.SecondPerSample);
            }

            var duration = SpyGlobalTime.FromSeconds(sampleCount * this.SecondPerSample);
            var endTime = beginTime + duration;

            if (this.EndTime.Timestamp < endTime)
            {
                this.EndTime = this.GetTime(endTime);
            }

            var frameInfo = new WaveformInfo(
                this.TotalSampleCount,
                sampleCount,
                _deviceOutputVolume,
                dataBlock.ID);

            this.TotalSampleCount += sampleCount;

            lock (_infoLockObject)
            {
                frameInfo.WaveformIndex = this.WaveformCount++;
                _waveformInfos.Add(frameInfo);
            }

            var waveform = this.ReadWaveform(CreateDataReader(dataBlock), frameInfo);

            // 波形データのキャッシュを作成します。
            _peakDataManager.Add(waveform);

            // ラウドネスを非同期に計算します。
            if (this.IsLoudnessAvailable)
            {
                var request = new CalcLoudnessRequest(waveform);
                lock (_calcLoudnessLock)
                {
                    _calcLoudnessRequests.Enqueue(request);
                    if (_calcLoudnessTask == null)
                    {
                        var cancellationToken = _calcLoudnessCancellationTokenSource.Token;
                        _calcLoudnessTask = Task.Factory.StartNew(
                            CalcLoudnessFunc,
                            (object)cancellationToken,
                            cancellationToken);
                    }
                }
            }

            if (UpdateEvent != null)
            {
                UpdateEvent(this, EventArgs.Empty);
            }
        }

        private void OnPushDeviceOutputVolume(SpyDataBlock dataBlock, BinaryReader reader)
        {
            reader.ReadBytes(3); // padding
            _deviceOutputVolume = reader.ReadSingle();
        }

        /// <summary>
        /// バックグランドでラウドネス情報を計算するTaskです。
        /// </summary>
        /// <param name="state"></param>
        private void CalcLoudnessFunc(object state)
        {
            var cancellationToken = (CancellationToken)state;

            bool needUpdate = false;
            while (!cancellationToken.IsCancellationRequested)
            {
                CalcLoudnessRequest args;
                lock (_calcLoudnessLock)
                {
                    if (!_calcLoudnessRequests.TryDequeue(out args))
                    {
                        _calcLoudnessTask = null;
                        break;
                    }
                }

                foreach (var loudness in _loudnessAccumulator.Accumulate(args.Waveform))
                {
                    needUpdate = true;
                    _calcLoudnessResults.Enqueue(new CalcLoudnessResult(loudness));
                }

                if (needUpdate)
                {
                    if (this.LoudnessUpdateEvent != null)
                    {
                        var now = DateTime.Now;
                        if ((now - _loudnessUpdateEventNotifyTime).TotalMilliseconds >= LoudnessUpdateEventInterval)
                        {
                            _loudnessUpdateEventNotifyTime = now;
                            this.LoudnessUpdateEvent(this, EventArgs.Empty);
                            needUpdate = false;
                        }
                    }
                }
            }

            if (cancellationToken.IsCancellationRequested)
            {
                return;
            }

            if (needUpdate)
            {
                if (this.LoudnessUpdateEvent != null)
                {
                    _loudnessUpdateEventNotifyTime = DateTime.Now;
                    this.LoudnessUpdateEvent(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// ラウドネスの計算処理をまとめたクラスです。
        /// ラウドネスの計算に必要なサンプル数は時間毎のサンプル数よりも多いので
        /// 計算途中の状態を保持します。
        /// </summary>
        private class LoudnessAccumulator
        {
            private class WaveformRegion
            {
                public Waveform Waveform { get; }
                public int From { get; }
                public int To { get; }

                public WaveformRegion(Waveform waveform, int from, int to)
                {
                    this.Waveform = waveform;
                    this.From = from;
                    this.To = to;
                }
            }

            private readonly LoudnessCalculator _loudnessCalculator = new LoudnessCalculator();
            private readonly Loudness.LoudnessContext[] _loudnessContext = new Loudness.LoudnessContext[ChannelCountMax].Populate(_ => new Loudness.LoudnessContext());
            private readonly float[] _channelValue = new float[ChannelCountMax];
            private readonly float[] _sampleSquareSum = new float[ChannelCountMax];
            private readonly float[] _samplePeak = new float[ChannelCountMax];
            private readonly float[] _sampleTruePeak = new float[ChannelCountMax];
            private readonly List<WaveformRegion> _waveformRegions = new List<WaveformRegion>();
            private int _accumulatedSampleCount;
            private Loudness.Parameter _parameter;

            public LoudnessAccumulator()
            {
                Ensure.Operation.True(Loudness.ChannelCount <= ChannelCountMax);
            }

            public void SetParameter(Loudness.Parameter parameter)
            {
                _parameter = parameter;
                _loudnessCalculator.SetParameter(parameter);
            }

            public IEnumerable<LoudnessInfo> Accumulate(Waveform waveform)
            {
                int sampleIndex = 0;
                while (sampleIndex < waveform.SampleCount)
                {
                    // ラウドネスの計算に足りないサンプル数だけ累積
                    int sampleCount = Math.Min(waveform.SampleCount - sampleIndex, _parameter.StepSampleCount - _accumulatedSampleCount);
                    Assertion.Operation.True(sampleCount > 0);

                    _waveformRegions.Add(new WaveformRegion(waveform, sampleIndex, sampleIndex + sampleCount));
                    sampleIndex += sampleCount;
                    _accumulatedSampleCount += sampleCount;

                    // ラウドネスの計算にサンプル数が足りないときはここで終了
                    if (_accumulatedSampleCount < _parameter.StepSampleCount)
                    {
                        break;
                    }

                    // ラウドネスを計算
                    {
                        this.AccumulateSample();

                        float totalMeanSquare = 0.0f;
                        float[] sampleSquareSumArray = new float[Loudness.ChannelCount];
                        float[] samplePeakArray = new float[Loudness.ChannelCount];
                        float[] sampleTruePeakArray = new float[Loudness.ChannelCount];
                        for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                        {
                            totalMeanSquare += _channelValue[channelIndex] * Loudness.ChannelWeightArray[channelIndex];
                            sampleSquareSumArray[channelIndex] = _sampleSquareSum[channelIndex];
                            samplePeakArray[channelIndex] = _samplePeak[channelIndex];
                            sampleTruePeakArray[channelIndex] = _sampleTruePeak[channelIndex];
                        }

                        _loudnessCalculator.AddStep(totalMeanSquare, sampleSquareSumArray, samplePeakArray, sampleTruePeakArray);
                    }

                    var loudnessInfo = ReadLoudnessInfo();

                    // 計算の完了したサンプルのインデックスを記録します。
                    loudnessInfo.SampleIndex = waveform.SampleIndex + sampleIndex;

                    ResetValue();

                    yield return loudnessInfo;
                }
            }

            private void ResetValue()
            {
                for (int i = 0; i < ChannelCountMax; ++i)
                {
                    _channelValue[i] = 0;
                    _sampleSquareSum[i] = 0;
                    _samplePeak[i] = 0;
                    _sampleTruePeak[i] = 0;
                }

                _accumulatedSampleCount = 0;
            }

            private void AccumulateSample()
            {
                Parallel.For(0, ChannelCountMax, channelIndex =>
                {
                    var samplePeak = float.MinValue;
                    var sampleTruePeak = float.MinValue;
                    var sampleSquareSum = 0.0f;
                    var channelValue = 0.0f;
                    var loudnessContext = _loudnessContext[channelIndex];

                    foreach (var region in _waveformRegions)
                    {
                        short[] sample = region.Waveform.SampleData.Samples[channelIndex];

                        for (int i = region.From; i < region.To; i++)
                        {
                            float sampleValue = sample[i] / 32768.0f;

                            samplePeak = Math.Max(samplePeak, Math.Abs(sampleValue));
                            sampleTruePeak = Math.Max(sampleTruePeak, Loudness.CalcTruePeak(loudnessContext, sampleValue));
                            sampleSquareSum += sampleValue * sampleValue;
                            float kOut = Loudness.CalcKWeightingFilter(loudnessContext, sampleValue, _parameter);
                            channelValue += kOut * kOut;
                        }
                    }

                    _samplePeak[channelIndex] = Math.Max(_samplePeak[channelIndex], samplePeak);
                    _sampleTruePeak[channelIndex] = Math.Max(_sampleTruePeak[channelIndex], sampleTruePeak);
                    _sampleSquareSum[channelIndex] += sampleSquareSum;
                    _channelValue[channelIndex] += channelValue;
                });

                _waveformRegions.Clear();
            }

            private LoudnessInfo ReadLoudnessInfo()
            {
                var loudness = new LoudnessInfo();

                loudness.MomentaryLoudnessValue = _loudnessCalculator.MomentaryLoudnessValue;
                loudness.ShortTermLoudnessValue = _loudnessCalculator.ShortTermLoudnessValue;
                loudness.AbsGatedLoudnessValue = _loudnessCalculator.AbsGatedLoudnessValue;
                loudness.RelGatedLoudnessValue = _loudnessCalculator.RelGatedLoudnessValue;

                for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                {
                    var channelInfo = loudness.Channels[channelIndex];
                    channelInfo.PeakValue = _loudnessCalculator.PeakValue[channelIndex];
                    channelInfo.TruePeakValue = _loudnessCalculator.TruePeakValue[channelIndex];
                    channelInfo.RmsValue = _loudnessCalculator.RmsValue[channelIndex];
                }

                return loudness;
            }
        }
    }
}
