﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Text;
using System.Windows.Media;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Marker Spy モデルです。
    /// </summary>
    public sealed class MarkerSpyModel : SpyModel
    {
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_0_0_0 = new Version(1, 0, 0, 0);

        /// <summary>
        /// バージョン 1.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct ValuePacket {
        ///     u64 id;
        ///     u8 color[4];
        ///     StringData(ut 255) description;
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_1_0_0 = new Version(1, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(1, 2, 0, 0);
        private const int MaxTextLength = 255;

        private readonly ObservableDictionary<ulong, Marker> _markers =
            new ObservableDictionary<ulong, Marker>();

        private bool _errorUnexpectedDataVersion = false;

        public IDictionary<ulong, Marker> Markers
        {
            get { return _markers; }
        }

        /// <summary>
        /// サポートしないバージョンのデータを受信すると true に設定されます。
        /// </summary>
        public bool ErrorUnexpectedDataVersion
        {
            get { return _errorUnexpectedDataVersion; }
            private set { this.SetPropertyValue(ref _errorUnexpectedDataVersion, value); }
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                this.ErrorUnexpectedDataVersion = true;
                return;
            }

            var reader = CreateDataReader(dataBlock);

            var belongingFrame = this.GetBelongingFrame(dataBlock.Timestamp);

            ulong id = 0;
            if (this.DataVersion >= Version_1_1_0_0)
            {
                id = reader.ReadUInt64();
            }
            else
            {
                id = reader.ReadUInt32();
            }

            var r = reader.ReadByte();
            var g = reader.ReadByte();
            var b = reader.ReadByte();
            var a = reader.ReadByte();

            var descriptionLength = reader.ReadByte();
            string description = string.Empty;

            if (descriptionLength > 0)
            {
                description = Encoding.UTF8.GetString(reader.ReadBytes(MaxTextLength), 0, descriptionLength);
            }

            Marker marker;
            _markers.TryGetValue(id, out marker);

            // 未検出の ID は新規作成する
            if (marker == null)
            {
                marker = new Marker(id);
                _markers.Add(id, marker);
            }

            marker.Values.Add(
                new MarkerValue(marker, dataBlock.Timestamp, belongingFrame, Color.FromArgb(a, r, g, b), description));
        }

        public class Marker
        {
            private readonly IList<MarkerValue> _values = new ObservableList<MarkerValue>();

            public Marker(ulong id)
            {
                this.ID = id;
            }

            public ulong ID { get; private set; }

            public IList<MarkerValue> Values
            {
                get { return _values; }
            }
        }

        public class MarkerValue
        {
            public MarkerValue(Marker owner, SpyGlobalTime timestamp, SpyTime belongingFrame, Color color, string description)
            {
                this.Owner = owner;
                this.Timestamp = timestamp;
                this.BelongingFrame = belongingFrame;
                this.Color = color;
                this.Description = description;
            }

            public Marker Owner { get; private set; }
            public SpyGlobalTime Timestamp { get; private set; }
            public SpyTime BelongingFrame { get; private set; }
            public Frame AudioFrame { get { return this.BelongingFrame.AudioFrame; } }
            public Frame AppFrame { get { return this.BelongingFrame.AppFrame; } }
            public Color Color { get; private set; }
            public string Description { get; private set; }

            public override string ToString()
            {
                return string.Format("AudioFrame={0}, AppFrame={1}", this.AudioFrame, this.AppFrame);
            }
        }

        private enum MarkerDataType : byte
        {
            Value = 0,
            Metadata
        }
    }
}
