﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics.CodeAnalysis;
using System.Text;

namespace NintendoWare.Spy
{
    /// <summary>
    /// API コール Spy モデルです。
    /// </summary>
    public sealed class ApiCallSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.2.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct LogPacketData {
        ///     char message[256];
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_2_0_0 = new Version(0, 2, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 3, 0, 0);

        /// <summary>
        /// ログの行情報です。
        /// </summary>
        public class LogLine
        {
            public LogLine(SpyGlobalTime timestamp, SpyTime belongingFrame, int lineNumber, string text)
            {
                this.Timestamp = timestamp;
                this.BelongingFrame = belongingFrame;
                this.LineNumber = lineNumber;
                this.Text = text;
            }

            public int LineNumber { get; private set; }

            public SpyGlobalTime Timestamp { get; private set; }
            public SpyTime BelongingFrame { get; private set; }
            public string Text { get; private set; }
        }

        //-----------------------------------------------------------------

        private readonly ObservableCollection<LogLine> _logLines = new ObservableCollection<LogLine>();

        private bool _errorUnexpectedDataVersion = false;

        //-----------------------------------------------------------------

        /// <summary>
        /// ログラインを取得します。
        /// </summary>
        public IList<LogLine> LogLines
        {
            get { return _logLines; }
        }

        /// <summary>
        /// サポートしないバージョンのデータを受信すると true に設定されます。
        /// </summary>
        public bool ErrorUnexpectedDataVersion
        {
            get { return _errorUnexpectedDataVersion; }
            private set { this.SetPropertyValue(ref _errorUnexpectedDataVersion, value); }
        }

        //-----------------------------------------------------------------

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                this.ErrorUnexpectedDataVersion = true;
                return;
            }

            var reader = CreateDataReader(dataBlock);

            var data = reader.ReadBytes((int)(reader.BaseStream.Length - reader.BaseStream.Position));
            int textLength = Array.IndexOf(data, (byte)'\0');
            string text = Encoding.UTF8.GetString(data, 0, textLength);

            var newLine = new LogLine(
                dataBlock.Timestamp,
                GetBelongingFrame(dataBlock.Timestamp),
                _logLines.Count + 1,
                text);

            _logLines.Add(newLine);
        }
    }
}
