﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using System.Windows;
using System.Windows.Controls.Primitives;
using System.Windows.Interactivity;

namespace NintendoWare.Spy.Windows
{
    public class ThumbResizeBehavior : Behavior<Thumb>
    {
        /// <summary>
        /// リサイズの対象を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty TargetProperty = DependencyProperty.Register(
            nameof(Target),
            typeof(FrameworkElement),
            typeof(ThumbResizeBehavior),
            new FrameworkPropertyMetadata(null));

        /// <summary>
        /// 横方向のリサイズの可否を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty IsHorizontalResizeEnabledProperty = DependencyProperty.Register(
            nameof(IsHorizontalResizeEnabled),
            typeof(bool),
            typeof(ThumbResizeBehavior),
            new FrameworkPropertyMetadata(true));

        /// <summary>
        /// 縦方向のリサイズの可否を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty IsVerticalResizeEnabledProperty = DependencyProperty.Register(
            nameof(IsVerticalResizeEnabled),
            typeof(bool),
            typeof(ThumbResizeBehavior),
            new FrameworkPropertyMetadata(true));

        /// <summary>
        /// リサイズの対象を取得または設定します。
        /// </summary>
        public FrameworkElement Target
        {
            get { return (FrameworkElement)GetValue(TargetProperty); }
            set { SetValue(TargetProperty, value); }
        }

        /// <summary>
        /// 横方向のリサイズの可否を取得または設定します。
        /// </summary>
        public bool IsHorizontalResizeEnabled
        {
            get { return (bool)GetValue(IsHorizontalResizeEnabledProperty); }
            set { SetValue(IsHorizontalResizeEnabledProperty, value); }
        }

        /// <summary>
        /// 縦方向のリサイズの可否を取得または設定します。
        /// </summary>
        public bool IsVerticalResizeEnabled
        {
            get { return (bool)GetValue(IsVerticalResizeEnabledProperty); }
            set { SetValue(IsVerticalResizeEnabledProperty, value); }
        }

        protected override void OnAttached()
        {
            base.OnAttached();

            this.AssociatedObject.DragDelta += this.UpdateSize;
        }

        protected override void OnDetaching()
        {
            base.OnDetaching();

            this.AssociatedObject.DragDelta -= this.UpdateSize;
        }

        private void UpdateSize(object sender, DragDeltaEventArgs e)
        {
            if (this.Target == null)
            {
                return;
            }

            if (IsHorizontalResizeEnabled)
            {
                this.Target.Width = MathUtility.Clamp(this.Target.ActualWidth + e.VerticalChange, this.Target.MinWidth, this.Target.MaxWidth);
            }

            if (IsVerticalResizeEnabled)
            {
                this.Target.Height = MathUtility.Clamp(this.Target.ActualHeight + e.VerticalChange, this.Target.MinHeight, this.Target.MaxHeight);
            }
        }
    }
}
