﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Interactivity;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    public class TextBoxBehavior : Behavior<TextBox>
    {
        public static readonly DependencyProperty PlaceHolderTextProperty = DependencyProperty.Register(
            nameof(PlaceHolderText),
            typeof(string),
            typeof(TextBoxBehavior),
            new PropertyMetadata((d, e) => ((TextBoxBehavior)d).SetPlaceHolderText()));

        /// <summary>
        /// TextBoxの内容が空のときに代わりに表示される文字列です。
        /// </summary>
        public string PlaceHolderText
        {
            get { return (string)GetValue(PlaceHolderTextProperty); }
            set { SetValue(PlaceHolderTextProperty, value); }
        }

        public static readonly DependencyProperty UpdateSourceOnTextChangedProperty = DependencyProperty.Register(
            nameof(UpdateSourceOnTextChanged),
            typeof(bool),
            typeof(TextBoxBehavior));

        /// <summary>
        /// TextBoxへの入力をただちにソースに反映したい場合に使います。
        /// {Binding UpdateSourceTrigger=Explicit}とセットで指定します。
        /// <remarks>
        /// {Binding UpdateSourceTrigger=PropertyChanged}だとATOKを使った場合に未確定文字が消えてしまいます。
        /// {Binding UpdateSourceTrigger=PropertyChanged, Mode=OneWayToSource}でも良いですが、
        /// ソースからビューへの反映ができなくなります。
        /// </remarks>
        /// </summary>
        public bool UpdateSourceOnTextChanged
        {
            get { return (bool)GetValue(UpdateSourceOnTextChangedProperty); }
            set { SetValue(UpdateSourceOnTextChangedProperty, value); }
        }

        public static readonly DependencyProperty UpdateSourceOnReturnProperty = DependencyProperty.Register(
            nameof(UpdateSourceOnReturn),
            typeof(bool),
            typeof(TextBoxBehavior));

        /// <summary>
        /// TextBoxでリターンキーを押したときにソースに反映したい場合に使います。
        /// </summary>
        public bool UpdateSourceOnReturn
        {
            get { return (bool)GetValue(UpdateSourceOnReturnProperty); }
            set { SetValue(UpdateSourceOnReturnProperty, value); }
        }

        public TextBoxBehavior()
        {
            this.PlaceHolderText = "(PlaceHolderText)";
        }

        protected override void OnAttached()
        {
            base.OnAttached();

            this.AssociatedObject.TextChanged += HandleTextChangedEvent;
            this.AssociatedObject.KeyUp += HandleKeyUpEvent;
            this.SetPlaceHolderText();
        }

        protected override void OnDetaching()
        {
            base.OnDetaching();

            this.AssociatedObject.TextChanged -= HandleTextChangedEvent;
            this.AssociatedObject.KeyUp -= HandleKeyUpEvent;
        }

        private void HandleTextChangedEvent(object sender, TextChangedEventArgs e)
        {
            this.SetPlaceHolderText();
            if (this.UpdateSourceOnTextChanged)
            {
                this.UpdateSource();
            }
        }

        private void HandleKeyUpEvent(object sender, KeyEventArgs e)
        {
            switch (e.Key)
            {
                case Key.Return:
                    if (this.UpdateSourceOnReturn)
                    {
                        this.UpdateSource();
                    }
                    break;

                default:
                    break;
            }
        }

        private void SetPlaceHolderText()
        {
            var textBox = this.AssociatedObject;

            if (textBox == null)
            {
                return;
            }

            if (string.IsNullOrEmpty(this.PlaceHolderText))
            {
                return;
            }

            if (string.IsNullOrEmpty(textBox.Text))
            {
                textBox.Background = this.CreateVisualBrush(this.PlaceHolderText);
            }
            else
            {
                textBox.Background = new SolidColorBrush(Colors.Transparent);
            }
        }

        private void UpdateSource()
        {
            var binding = this.AssociatedObject?.GetBindingExpression(TextBox.TextProperty);
            if (binding != null)
            {
                binding.UpdateSource();
            }
        }

        private VisualBrush CreateVisualBrush(string placeHolderText)
        {
            var visual = new Label()
            {
                Content = placeHolderText,
                Padding = new Thickness(5, 1, 1, 1),
                Foreground = new SolidColorBrush(Colors.LightGray),
                HorizontalAlignment = HorizontalAlignment.Left,
                VerticalAlignment = VerticalAlignment.Center,
            };

            var visualBrush = new VisualBrush(visual)
            {
                Stretch = Stretch.None,
                TileMode = TileMode.None,
                AlignmentX = AlignmentX.Left,
                AlignmentY = AlignmentY.Center,
            };

            return visualBrush;
        }
    }
}
