﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Windows;
using System.Windows.Data;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// SpyTime、またはフレーム値を時間の表示単位に応じた表示用文字列に変換します。
    /// 時間の表示単位は TimeUnit プロパティ、またはコンバータ・パラメータで指定します。
    /// </summary>
    public sealed class SpyTimeToStringValueConverter : IValueConverter
    {
        /// <summary>
        /// Timestamp から TimestampUsec の表現に自動的に切り替わる Scale の閾値。
        /// </summary>
        private const double TimestampUsecScaleThreshold = 0.1;

        public class Parameter
        {
            /// <summary>
            /// <see cref="Convert"/>の結果を制御します。
            /// <list type="bullet">
            ///   <item>
            ///     <term><see cref="SpyTimeUnit.Timestamp"/>, <see cref="SpyTimeUnit.TimestampUsec"/></term>
            ///     <description>
            ///       <para>hms 形式の時間表現に変換します。</para>
            ///       <para>value 引数は <see cref="SpyTime"/>、<see cref="SpyGlobalTime"/>、または long に変換可能な値をサポートします。</para>
            ///       <para>変換結果の精度にも影響します。</para>
            ///     </description>
            ///   </item>
            ///   <item>
            ///     <term><see cref="SpyTimeUnit.AppFrame"/>, <see cref="SpyTimeUnit.AudioFrame"/></term>
            ///     <description>
            ///       <para>シンプルな数値表現に変換します。</para>
            ///       <para>value 引数は <see cref="SpyTime"/>、<see cref="Frame"/>、または long に変換可能な値をサポートします。</para>
            ///     </description>
            ///   </item>
            /// </list>
            /// </summary>
            public SpyTimeUnit? TimeUnit { get; set; }

            /// <summary>
            /// <see cref="NintendoWare.Spy.Windows.Primitives.PlotFrameGrid"/> 等でフレーム値に適用されているスケールを指定します。
            /// 変換結果の精度に影響します。
            /// </summary>
            public double? Scale { get; set; }
        }

        public SpyTimeToStringValueConverter()
        {
            this.TimeUnit = SpyTimeUnit.Timestamp;
        }

        /// <summary>
        /// 時間の表示単位。
        /// このプロパティよりもコンバータ・パラメータが優先されます。
        /// </summary>
        public SpyTimeUnit TimeUnit { get; set; }

        /// <summary>
        /// 時間の表示単位に応じた表示用文字列に変換します。
        /// </summary>
        /// <param name="value">変換のソースです。解釈は <paramref name="parameter"/> によって変わります。</param>
        /// <param name="targetType"><see cref="string"/> のみサポートします。</param>
        /// <param name="parameter">
        /// パラメータです。指定できる値の型は以下の通りです。
        /// <list type="bullet" >
        ///   <item>
        ///     <term><see cref="Parameter"/> 型</term>
        ///     <description>
        ///       <para>パラメータを指定します。詳細は <see cref="Parameter"/> を参照してください。</para>
        ///     </description>
        ///   </item>
        ///   <item>
        ///     <term><see cref="SpyTimeUnit"/> 型</term>
        ///     <description>
        ///         <para>効果は <see cref="Parameter"/> で <see cref="Parameter.TimeUnit"/> プロパティを指定した場合と同様です。</para>
        ///     </description>
        ///   </item>
        ///   <item>
        ///     <term><see cref="string"/> 型</term>
        ///     <description>
        ///       <para><see cref="SpyTimeUnit"/> に変換されます。</para>
        ///       <para>効果は <see cref="Parameter"/> で <see cref="Parameter.TimeUnit"/> プロパティを指定した場合と同様です。</para>
        ///     </description>
        ///   </item>
        /// </list>
        /// </param>
        /// <param name="culture">使用しません。</param>
        /// <returns>時間の表示単位に応じた表示用文字列を返します。</returns>
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            SpyTimeUnit timeUnit = this.TimeUnit;
            double? scale = null;

            if (parameter != null)
            {
                if (parameter is Parameter)
                {
                    var p = (Parameter)parameter;

                    if (p.TimeUnit.HasValue)
                    {
                        timeUnit = p.TimeUnit.Value;
                    }

                    if (p.Scale.HasValue)
                    {
                        scale = p.Scale.Value;
                    }
                }
                else if (parameter is SpyTimeUnit)
                {
                    timeUnit = (SpyTimeUnit)parameter;
                }
                else
                {
                    if (!Enum.TryParse(parameter.ToString(), out timeUnit))
                    {
                        timeUnit = this.TimeUnit;
                    }
                }
            }

            if (targetType == typeof(string))
            {
                if (value is SpyTime)
                {
                    if (timeUnit == SpyTimeUnit.Timestamp &&
                        scale.HasValue &&
                        scale.Value >= TimestampUsecScaleThreshold)
                    {
                        return ((SpyTime)value).Timestamp.ToMicroSecondsLabelString();
                    }
                    else
                    {
                        return ((SpyTime)value).ToLabelString(timeUnit);
                    }
                }
                else
                {
                    switch (timeUnit)
                    {
                        case SpyTimeUnit.AppFrame:
                        case SpyTimeUnit.AudioFrame:
                            return ToFrame(value).ToLabelString();

                        case SpyTimeUnit.Timestamp:
                            if (scale.HasValue && scale.Value >= TimestampUsecScaleThreshold)
                            {
                                return ToSpyGlobalTime(value).ToMicroSecondsLabelString();
                            }
                            else
                            {
                                return ToSpyGlobalTime(value).ToLabelString();
                            }

                        case SpyTimeUnit.TimestampUsec:
                            return ToSpyGlobalTime(value).ToMicroSecondsLabelString();

                        default:
                            throw new InvalidOperationException();
                    }
                }
            }
            else
            {
                throw new NotImplementedException();
            }
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            if (value is string)
            {
                switch (this.TimeUnit)
                {
                    case SpyTimeUnit.Timestamp:
                    case SpyTimeUnit.TimestampUsec:
                        try
                        {
                            if (targetType == typeof(SpyGlobalTime))
                            {
                                return SpyGlobalTime.Parse((string)value);
                            }
                            else
                            {
                                return System.Convert.ChangeType(SpyGlobalTime.Parse((string)value).MicroSeconds, targetType);
                            }
                        }
                        catch
                        {
                            return DependencyProperty.UnsetValue;
                        }

                    case SpyTimeUnit.AppFrame:
                    case SpyTimeUnit.AudioFrame:
                        try
                        {
                            if (targetType == typeof(Frame))
                            {
                                return Frame.Parse((string)value);
                            }
                            else
                            {
                                return System.Convert.ChangeType(Frame.Parse((string)value).Value, targetType);
                            }
                        }
                        catch
                        {
                            return DependencyProperty.UnsetValue;
                        }

                    default:
                        throw new NotImplementedException();
                }
            }
            else
            {
                throw new NotImplementedException();
            }
        }

        private static Frame ToFrame(object value)
        {
            if (value is Frame)
            {
                return (Frame)value;
            }
            else
            {
                return new Frame(System.Convert.ToInt64(value));
            }
        }

        private static SpyGlobalTime ToSpyGlobalTime(object value)
        {
            if (value is SpyGlobalTime)
            {
                return (SpyGlobalTime)value;
            }
            else
            {
                return SpyGlobalTime.FromMicroSeconds(System.Convert.ToInt64(value));
            }
        }
    }
}
