﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Windows.Controls;
using System;
using System.Collections;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows.Primitives
{
    /// <summary>
    /// マーカーを表示するコントロールです。
    /// </summary>
    [TemplatePart(Name = TemplatePartMarkerLayerName, Type = typeof(Canvas))]
    public sealed class PlotMarker : Control
    {
        private const string TemplatePartMarkerLayerName = "PlotMarker_Canvas";

        public static readonly DependencyProperty OriginXProperty
            = DependencyProperty.Register(
                nameof(OriginX),
                typeof(double),
                typeof(PlotMarker),
                new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnOriginXChanged(e)));

        public static readonly DependencyProperty ScaleXProperty
            = DependencyProperty.Register(
                nameof(ScaleX),
                typeof(double),
                typeof(PlotMarker),
                new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnScaleXChanged(e)));

        /// <summary>
        /// 縦方向マーカーリストの元になるオブジェクトリストを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty VerticalMarkersSourceProperty = DependencyProperty.Register(
            nameof(VerticalMarkersSource),
            typeof(IList),
            typeof(PlotMarker),
            new FrameworkPropertyMetadata(null, (d, e) => Self(d).OnVerticalMarkersSourceChanged(e)));

        /// <summary>
        /// 縦方向マーカーのスタイルを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty VerticalMarkerStyleProperty = DependencyProperty.Register(
            nameof(VerticalMarkerStyle),
            typeof(Style),
            typeof(PlotMarker),
            new FrameworkPropertyMetadata(null, (d, e) => Self(d).OnVerticalMarkerStyleChanged(e)));

        /// <summary>
        /// 縦方向マーカーの結合範囲を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty CombiningRangeForVerticalMarkersProperty = DependencyProperty.Register(
            nameof(CombiningRangeForVerticalMarkers),
            typeof(double),
            typeof(PlotMarker),
            new FrameworkPropertyMetadata(10.0, (d, e) => Self(d).OnCombiningRangeForVerticalMarkersChanged(e)));

        private readonly SortedListDecorator<double, ILineChartMarkerData> _verticalMarkerDataList;

        private Canvas _markerLayer;
        private int _lastMarkerCount;

        private readonly MergedRequestDispatcher _updateMarkerLayerDispatcher = new MergedRequestDispatcher();
        private bool _isMarkerReoredered = false;

        static PlotMarker()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(PlotMarker),
                new FrameworkPropertyMetadata(typeof(PlotMarker)));
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PlotMarker()
        {
            _verticalMarkerDataList =
                new SortedListDecorator<double, ILineChartMarkerData>(marker => marker.Position);

            _verticalMarkerDataList.CollectionChanged += (sender, e) =>
            {
                this.RequestUpdateMarkerLayer(reorder: true);
            };
        }

        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        private double ValidatedOriginX
        {
            get { return PlotCanvas.GetValidatedOriginX(this.OriginX); }
        }

        private double ValidatedScaleX
        {
            get { return PlotCanvas.GetValidatedScaleX(this.ScaleX); }
        }

        /// <summary>
        /// マーカーリストの元になるオブジェクトリストを取得または設定します。
        /// </summary>
        public IList VerticalMarkersSource
        {
            get { return (IList)this.GetValue(VerticalMarkersSourceProperty); }
            set { this.SetValue(VerticalMarkersSourceProperty, value); }
        }

        /// <summary>
        /// マーカーのスタイルを取得または設定します。
        /// </summary>
        public Style VerticalMarkerStyle
        {
            get { return (Style)this.GetValue(VerticalMarkerStyleProperty); }
            set { this.SetValue(VerticalMarkerStyleProperty, value); }
        }

        /// <summary>
        /// 縦方向マーカーの結合範囲を取得または設定します。
        /// </summary>
        public double CombiningRangeForVerticalMarkers
        {
            get { return (double)this.GetValue(CombiningRangeForVerticalMarkersProperty); }
            set { this.SetValue(CombiningRangeForVerticalMarkersProperty, value); }
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _markerLayer = this.Template.FindName(TemplatePartMarkerLayerName, this) as Canvas;
            _lastMarkerCount = 0;

            this.RequestUpdateMarkerLayer(reorder: true);
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.RequestUpdateMarkerLayer();
        }

        private void OnOriginXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateMarkerLayer();
        }

        private void OnScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateMarkerLayer();
        }

        private void OnVerticalMarkersSourceChanged(DependencyPropertyChangedEventArgs e)
        {
            this.AttachVerticalMarkersSource();
        }

        private void OnVerticalMarkerStyleChanged(DependencyPropertyChangedEventArgs e)
        {
            this.ApplyMarkerStyle();
        }

        private void OnCombiningRangeForVerticalMarkersChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateMarkerLayer();
        }

        private void AttachVerticalMarkersSource()
        {
            this.DetachVerticalMarkersSource();

            if (this.VerticalMarkersSource != null)
            {
                _verticalMarkerDataList.SetSource(
                    new GenericListDecorator<ILineChartMarkerData>(this.VerticalMarkersSource));
            }

            this.RequestUpdateMarkerLayer(reorder: true);
        }

        private void DetachVerticalMarkersSource()
        {
            _verticalMarkerDataList.SetSource(null);
            if (_markerLayer != null)
            {
                _markerLayer.Children.Clear();
                _lastMarkerCount = 0;
            }
        }

        private void RequestUpdateMarkerLayer(bool reorder = false)
        {
            _isMarkerReoredered = reorder;
            _updateMarkerLayerDispatcher.Request(UpdateMarkerLayer, DispatcherPriority.Render);
        }

        private void UpdateMarkerLayer()
        {
            if (_markerLayer == null)
            {
                return;
            }

            if (_isMarkerReoredered)
            {
                _isMarkerReoredered = false;
                _verticalMarkerDataList.Sort();
            }

            // _verticalMarkerDataList から表示範囲のマーカーを選択し、
            // _markerLayer.Children のコンテントに設定します。
            var markerCount = this.SelectVisibleMarkers();

            if (markerCount < _markerLayer.Children.Count)
            {
                // 余分な未使用マーカーを破棄します。
                var newChildrenCount = Math.Max(
                    markerCount + 100,
                    Math.Min(_markerLayer.Children.Count, markerCount * 2));

                if (newChildrenCount < _markerLayer.Children.Count)
                {
                    _markerLayer.Children.RemoveRange(newChildrenCount, _markerLayer.Children.Count - newChildrenCount);
                }

                // 未使用マーカーを非表示にします。
                if (markerCount < _lastMarkerCount)
                {
                    _markerLayer.Children
                        .Cast<PlotMarkerItem>()
                        .Skip(markerCount)
                        .Take(Math.Min(_markerLayer.Children.Count, _lastMarkerCount) - markerCount)
                        .ForEach(marker => this.CollapseMarker(marker));
                }
            }

            // 無駄に非表示の設定をくりかえす必要がないように、使用している範囲を覚えておきます。
            _lastMarkerCount = markerCount;
        }

        /// <summary>
        /// 画面範囲内のマーカーを選択し _markerLayer.Children のコンテントに設定します。
        /// _markerLayer.Children の要素は再利用され、必要に応じて追加されます。
        /// </summary>
        /// <returns>_markerLayer.Children の使用した要素数です。</returns>
        private int SelectVisibleMarkers()
        {
            if (_verticalMarkerDataList.Count == 0)
            {
                return 0;
            }

            var reciprocalScaleX = 1.0 / this.ValidatedScaleX;

            var markerDistance = this.CombiningRangeForVerticalMarkers * reciprocalScaleX;
            if (markerDistance < 0)
            {
                markerDistance = 0;
            }

            var minX = this.ValidatedOriginX;
            var maxX = minX + this.ActualWidth * reciprocalScaleX;

            // CombiningRangeForVerticalMarkers 単位でマーカーを統合するため、ビューポートより前にあるマーカーも含めて処理します。
            // ビューポートより前にあるマーカーは、あとで Visibility = Collapsed に設定します。
            double currentMarkerMaxX = this.GetLowerMultipleOf(minX, markerDistance);
            var firstMarkerDataIndex = this.FindMarkerDataIndex(currentMarkerMaxX);

            if (firstMarkerDataIndex < 0)
            {
                return 0;
            }

            var markerCount = 0;
            PlotMarkerItem currentMarker = null;

            foreach (var markerDataIndex in Enumerable.Range(firstMarkerDataIndex, _verticalMarkerDataList.Count - firstMarkerDataIndex))
            {
                var currentMarkerData = _verticalMarkerDataList[markerDataIndex];

                // ビューポートの範囲を超えたら終了
                if (maxX <= currentMarkerData.Position)
                {
                    break;
                }

                // 近傍のマーカーを統合
                if (markerDistance == 0)
                {
                    currentMarker = null;
                }
                else if (currentMarkerMaxX <= currentMarkerData.Position)
                {
                    currentMarkerMaxX = (Math.Floor(currentMarkerData.Position / markerDistance) + 1) * markerDistance;
                    currentMarker = null;
                }

                if (currentMarker == null)
                {
                    // マーカーの最大数を超えたら、新しいマーカーインスタンスを作成する
                    if (_markerLayer.Children.Count <= markerCount)
                    {
                        var newMarker = new PlotMarkerItem()
                        {
                            Style = this.VerticalMarkerStyle,
                        };

                        _markerLayer.Children.Add(newMarker);
                    }

                    currentMarker = (PlotMarkerItem)_markerLayer.Children[markerCount];
                    currentMarker.DataContext = currentMarkerData;
                    currentMarker.IsCombined = false;

                    markerCount++;
                }
                else
                {
                    // 複数ラベルを統合したときに "..." を付加する
                    currentMarker.IsCombined = true;
                }
            }

            if (markerCount > 0)
            {
                // 近傍のマーカーを統合するために含めていた表示範囲外のマーカーを非表示にします。
                // 表示範囲内のマーカーを更新します。
                _markerLayer.Children
                    .Cast<PlotMarkerItem>()
                    .Take(markerCount)
                    .ForEach(marker =>
                    {
                        if ((marker.DataContext as ILineChartMarkerData).Position < minX)
                        {
                            this.CollapseMarker(marker);
                        }
                        else
                        {
                            marker.Visibility = System.Windows.Visibility.Visible;
                        }
                    });
            }

            return markerCount;
        }

        private void CollapseMarker(PlotMarkerItem marker)
        {
            marker.Visibility = System.Windows.Visibility.Collapsed;
            marker.DataContext = null;
        }

        private int FindMarkerDataIndex(double position)
        {
            var resultMin = _verticalMarkerDataList.BinarySearchByKeyNtf(position, marker => marker.Position);

            // Viewport 内にマーカーが存在するか？
            if (resultMin.PrevIndex == _verticalMarkerDataList.Count - 1)
            {
                return -1;
            }

            return resultMin.GetMatchOrNextIndex() ?? -1;
        }

        private void ApplyMarkerStyle()
        {
            if (_markerLayer != null)
            {
                _markerLayer.Children
                    .OfType<PlotMarkerItem>()
                    .ForEach(item => item.Style = this.VerticalMarkerStyle);
            }
        }

        private double GetLowerMultipleOf(double value, double round)
        {
            return Math.Floor(value / round) * round;
        }

        private static PlotMarker Self(DependencyObject d)
        {
            return (PlotMarker)d;
        }
    }
}
