﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows.Primitives
{
    public sealed class PlotFrameGrid : Control
    {
        private const int MaxGridCount = 500;

        /// <summary>
        /// ラベルを表示する位置です。
        /// </summary>
        public enum LabelPositionType
        {
            Top,
            Bottom,
        }

        /// <summary>
        /// 依存プロパティの変化をイベントで通知します。
        /// </summary>
        private class DependencyPropertyChangedNotifier : DependencyObject
        {
            private static readonly DependencyProperty ValueProperty
                = DependencyProperty.Register(
                    "Value",
                    typeof(object),
                    typeof(DependencyPropertyChangedNotifier),
                    new FrameworkPropertyMetadata(
                        null,
                        (d, e) => Self(d).OnValueChanged(e)));

            private WeakReference<DependencyObject> _source;

            public event EventHandler<DependencyPropertyChangedEventArgs> PropertyChanged;

            public DependencyPropertyChangedNotifier(DependencyObject source, DependencyProperty property)
            {
                Ensure.Argument.NotNull(source, nameof(source));
                Ensure.Argument.NotNull(property, nameof(property));

                _source = new WeakReference<DependencyObject>(source);

                BindingOperations.SetBinding(this, ValueProperty, new Binding()
                {
                    Path = new PropertyPath(property),
                    Mode = BindingMode.OneWay,
                    Source = source,
                });
            }

            private void OnValueChanged(DependencyPropertyChangedEventArgs arg)
            {
                DependencyObject s;
                if (_source.TryGetTarget(out s))
                {
                    this.PropertyChanged?.Invoke(s, arg);
                }
            }

            private static DependencyPropertyChangedNotifier Self(object obj)
            {
                return (DependencyPropertyChangedNotifier)obj;
            }
        }

        public static readonly DependencyProperty FrameSyncSpyModelProperty
            = DependencyProperty.Register(
                nameof(FrameSyncSpyModel),
                typeof(FrameSyncSpyModel),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    null,
                    FrameworkPropertyMetadataOptions.AffectsRender,
                    (d, e) => Self(d).OnFrameSyncSpyModelChanged(e)));

        public static readonly DependencyProperty OriginXProperty
            = DependencyProperty.Register(
                nameof(OriginX),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    0.0,
                    FrameworkPropertyMetadataOptions.AffectsRender),
                IsValidFiniteDouble);

        public static readonly DependencyProperty ScaleXProperty
            = DependencyProperty.Register(
                nameof(ScaleX),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    1.0,
                    FrameworkPropertyMetadataOptions.AffectsRender),
                v => IsValidFiniteDouble(v, d => d > 0));

        public static readonly DependencyProperty TimeUnitProperty
            = DependencyProperty.Register(
                nameof(TimeUnit),
                typeof(SpyTimeUnit),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    SpyTimeUnit.Timestamp,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty GridFrequencyProperty
            = DependencyProperty.Register(
                nameof(GridFrequency),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    100.0,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty FrameStringFormatProperty
            = DependencyProperty.Register(
                nameof(FrameStringFormat),
                typeof(string),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    string.Empty,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty FrameToStringValueConverterProperty
            = DependencyProperty.Register(
                nameof(FrameToStringValueConverter),
                typeof(IValueConverter),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    null,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty TimestampStringFormatProperty
            = DependencyProperty.Register(
                nameof(TimestampStringFormat),
                typeof(string),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    string.Empty,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty TimestampToStringValueConverterProperty
            = DependencyProperty.Register(
                nameof(TimestampToStringValueConverter),
                typeof(IValueConverter),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    null,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty GridThicknessProperty
            = DependencyProperty.Register(
                nameof(GridThickness),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    1.0,
                    FrameworkPropertyMetadataOptions.AffectsRender,
                    (d, e) => Self(d).OnGridThicknessChanged(e)),
                v => IsValidFiniteDouble(v, d => d >= 0));

        public static readonly DependencyProperty GridDashStyleProperty
            = DependencyProperty.Register(
                nameof(GridDashStyle),
                typeof(DashStyle),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    DashStyles.Dash,
                    FrameworkPropertyMetadataOptions.AffectsRender,
                    (d, e) => Self(d).OnGridDashStyleChanged(e)));

        public static readonly DependencyProperty GridPenProperty
            = DependencyProperty.Register(
                nameof(GridPen),
                typeof(Pen),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    null,
                    FrameworkPropertyMetadataOptions.AffectsRender,
                    (d, e) => Self(d).OnGridPenChanged(e)));

        public static readonly DependencyProperty LabelPositionProperty
            = DependencyProperty.Register(
                nameof(LabelPosition),
                typeof(LabelPositionType),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    LabelPositionType.Top,
                    FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty LabelOffsetProperty
            = DependencyProperty.Register(
                nameof(LabelOffset),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    2.0,
                    FrameworkPropertyMetadataOptions.AffectsRender),
                IsValidFiniteDouble);

        public static readonly DependencyProperty LabelMarginProperty
            = DependencyProperty.Register(
                nameof(LabelMargin),
                typeof(double),
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(
                    4.0,
                    FrameworkPropertyMetadataOptions.AffectsRender),
                IsValidFiniteDouble);

        /// <summary>
        /// フレーム情報を供給するモデルクラスです。
        /// </summary>
        public FrameSyncSpyModel FrameSyncSpyModel
        {
            get { return (FrameSyncSpyModel)this.GetValue(FrameSyncSpyModelProperty); }
            set { this.SetValue(FrameSyncSpyModelProperty, value); }
        }

        /// <summary>
        /// 表示範囲左端におけるX値です。
        /// </summary>
        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        /// <summary>
        /// X値をスクリーン座標に変換するスケール値です。
        /// </summary>
        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        /// <summary>
        /// 表示するフレームの種別です。
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            get { return (SpyTimeUnit)this.GetValue(TimeUnitProperty); }
            set { this.SetValue(TimeUnitProperty, value); }
        }

        /// <summary>
        /// AppFrame または AudioFrame の文字列フォーマットです。
        /// <see cref="FrameToStringValueConverter" />が優先されます。
        /// </summary>
        public string FrameStringFormat
        {
            get { return (string)this.GetValue(FrameStringFormatProperty); }
            set { this.SetValue(FrameStringFormatProperty, value); }
        }

        /// <summary>
        /// AppFrame または AudioFrame から文字列へのコンバーターです。
        /// <see cref="FrameStringFormat" />よりも優先されます。
        /// </summary>
        public IValueConverter FrameToStringValueConverter
        {
            get { return (IValueConverter)this.GetValue(FrameToStringValueConverterProperty); }
            set { this.SetValue(FrameToStringValueConverterProperty, value); }
        }

        /// <summary>
        /// Timestamp または TimestampUsec の文字列フォーマットです。
        /// <see cref="TimestampToStringValueConverter" />が優先されます。
        /// </summary>
        public string TimestampStringFormat
        {
            get { return (string)this.GetValue(TimestampStringFormatProperty); }
            set { this.SetValue(TimestampStringFormatProperty, value); }
        }

        /// <summary>
        /// Timestamp または TimestampUsec から文字列へのコンバーターです。
        /// <see cref="TimestampStringFormat" />よりも優先されます。
        /// </summary>
        public IValueConverter TimestampToStringValueConverter
        {
            get { return (IValueConverter)this.GetValue(TimestampToStringValueConverterProperty); }
            set { this.SetValue(TimestampToStringValueConverterProperty, value); }
        }

        /// <summary>
        /// 保証すべきグリッド間隔(ピクセル)です。
        /// 実際のグリッド間隔はこの値以上になります。
        /// </summary>
        public double GridFrequency
        {
            get { return (double)this.GetValue(GridFrequencyProperty); }
            set { this.SetValue(GridFrequencyProperty, value); }
        }

        /// <summary>
        /// グリッド線の太さです。
        /// <see cref="GridPen"/> が優先されます。
        /// </summary>
        public double GridThickness
        {
            get { return (double)this.GetValue(GridThicknessProperty); }
            set { this.SetValue(GridThicknessProperty, value); }
        }

        /// <summary>
        /// グリッド線のスタイルです。
        /// <see cref="GridPen"/> が優先されます。
        /// </summary>
        public DashStyle GridDashStyle
        {
            get { return (DashStyle)this.GetValue(GridDashStyleProperty); }
            set { this.SetValue(GridDashStyleProperty, value); }
        }

        /// <summary>
        /// グリッド線の描画に使うペンです。
        /// <see cref="GridThickness" /> および <see cref="GridDashStyle"/> よりも優先されます。
        /// </summary>
        public Pen GridPen
        {
            get { return (Pen)this.GetValue(GridPenProperty); }
            set { this.SetValue(GridPenProperty, value); }
        }

        /// <summary>
        /// グリッドラベルを表示する位置です。
        /// </summary>
        public LabelPositionType LabelPosition
        {
            get { return (LabelPositionType)this.GetValue(LabelPositionProperty); }
            set { this.SetValue(LabelPositionProperty, value); }
        }

        /// <summary>
        /// グリッドラベルを表示するボーダーからの距離です。
        /// </summary>
        public double LabelOffset
        {
            get { return (double)this.GetValue(LabelOffsetProperty); }
            set { this.SetValue(LabelOffsetProperty, value); }
        }

        /// <summary>
        /// グリッド線からグリッドラベルまでの距離です。
        /// </summary>
        public double LabelMargin
        {
            get { return (double)this.GetValue(LabelMarginProperty); }
            set { this.SetValue(LabelMarginProperty, value); }
        }

        private readonly object _observerOwner = new object();
        private DependencyPropertyChangedNotifier _foregroundChangedNotifier;
        private Pen _gridPenCache;
        private SpyTime _lastMaximumFrame;
        private readonly MergedRequestDispatcher _requestUpdateFrameSync = new MergedRequestDispatcher();

        static PlotFrameGrid()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(PlotFrameGrid),
                new FrameworkPropertyMetadata(typeof(PlotFrameGrid)));
        }

        public PlotFrameGrid()
        {
            _foregroundChangedNotifier = new DependencyPropertyChangedNotifier(this, ForegroundProperty);
            _foregroundChangedNotifier.PropertyChanged += (d, e) => this.OnForegroundChanged(e);
        }

        private void OnFrameSyncSpyModelChanged(DependencyPropertyChangedEventArgs e)
        {
            if (e.OldValue != null)
            {
                PropertyChangedObservation.RemoveObservers(_observerOwner);
            }

            if (e.NewValue != null)
            {
                var newModel = (FrameSyncSpyModel)e.NewValue;
                PropertyChangedObservation.GetObserver(_observerOwner, newModel).AddHandler(
                    nameof(newModel.MaximumFrame),
                    (target, args) => this.OnFrameSyncUpdated());
            }

            _lastMaximumFrame = null;
        }

        private void OnFrameSyncUpdated()
        {
            _requestUpdateFrameSync.Request(this.UpdateFrameRange);
        }

        private void UpdateFrameRange()
        {
            if (this.FrameSyncSpyModel == null)
            {
                return;
            }

            if (this.ActualWidth == 0)
            {
                return;
            }

            // フレーム情報が無効化されたときは一度だけ再描画をリクエストします。
            if (!this.FrameSyncSpyModel.MaximumFrame.Timestamp.IsValid)
            {
                if (_lastMaximumFrame != null)
                {
                    _lastMaximumFrame = null;
                    InvalidateVisual();

                    return;
                }
            }

            // 表示範囲にフレーム情報が追加されたら再描画をリクエストします。
            {
                if (_lastMaximumFrame == null)
                {
                    _lastMaximumFrame = this.FrameSyncSpyModel.MinimumFrame;
                }

                var viewLeftX = this.OriginX;
                var viewRightX = this.OriginX + this.ActualWidth / this.ScaleX;

                var leftX = _lastMaximumFrame.GetMicroSeconds();

                _lastMaximumFrame = this.FrameSyncSpyModel.MaximumFrame;
                var rightX = _lastMaximumFrame.GetMicroSeconds();

                if (viewLeftX <= rightX && leftX <= viewRightX)
                {
                    this.InvalidateVisual();
                }
            }
        }

        private void OnForegroundChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateGridPenCache();
        }

        private void OnGridThicknessChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateGridPenCache();
        }

        private void OnGridDashStyleChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateGridPenCache();
        }

        private void OnGridPenChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateGridPenCache();
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            base.OnRender(drawingContext);

            if (this.ActualWidth == 0 || this.ActualHeight == 0)
            {
                return;
            }

            if (this.Visibility != Visibility.Visible)
            {
                return;
            }

            drawingContext.DrawRectangle(
                this.Background,
                null /* pen */,
                new Rect(0, 0, this.ActualWidth, this.ActualHeight));

            if (!this.DrawFrameGrid(drawingContext))
            {
                this.DrawTimestampGrid(drawingContext);
            }
        }

        private bool DrawFrameGrid(DrawingContext drawingContext)
        {
            var frameSync = this.FrameSyncSpyModel;
            if (frameSync == null)
            {
                return false;
            }

            var originX = this.OriginX;
            var scaleX = this.ScaleX;

            IList<SpyTime> frameSyncList = null;
            switch (this.TimeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    frameSyncList = frameSync.AppFrames;
                    break;

                case SpyTimeUnit.AudioFrame:
                    frameSyncList = frameSync.AudioFrames;
                    break;

                default:
                    return false;
            }

            if (frameSyncList == null || frameSyncList.Count == 0)
            {
                return false;
            }

            var pen = this.GetGridPen();

            Typeface typeface = CreateTypeface();

            var getLabel = this.CreateFrameLabelFunctiion();

            Func<SpyTime, double> getXValue = t => t.GetMicroSeconds();

            var timeUnit = this.TimeUnit;
            Func<SpyTime, double> getFrameValue = t => t.SelectFrameValue(timeUnit);

            double gridDistanceF;
            {
                var firstTime = frameSyncList.First();
                var lastTime = frameSyncList.Last();
                var rangeX = getXValue(lastTime) - getXValue(firstTime);
                var rangeF = getFrameValue(lastTime) - getFrameValue(firstTime);
                var minDistanceF = this.GridFrequency / this.ScaleX * Math.Max(1, rangeF) / Math.Max(1, rangeX);
                gridDistanceF = this.CalcGridDistance(minDistanceF);
            }

            var originIndex = BinarySearchUtility.BinarySearch(frameSyncList, originX, getXValue, BinarySearchUtility.Options.SmallestIndex);
            if (originIndex < 0)
            {
                originIndex = ~originIndex - 1;
            }

            var originTime = frameSyncList[Math.Max(0, originIndex)];
            var startF = getFrameValue(originTime);

            var lastLabelRight = 0d;
            if (originIndex >= 0)
            {
                if (this.FontSize > 0)
                {
                    // 左端のフレーム値を表示します。
                    var label = getLabel(getFrameValue(originTime));
                    var text = this.CreateFormattedText(label, typeface);
                    var origin = this.CreateLabelOrition(text, 0);

                    drawingContext.DrawText(text, origin);

                    lastLabelRight = origin.X + text.Width;
                }

                // 左端にフレーム値が表示されているので、次のグリッドから描画します。
                startF = Math.Ceiling(startF / gridDistanceF) * gridDistanceF;
            }
            else
            {
                // 左端にフレーム値が表示されないので、最初のグリッドから描画します。
                startF = Math.Floor(startF / gridDistanceF) * gridDistanceF;
            }

            double lastFrame = double.NaN;
            for (var i = 0; i < MaxGridCount; i++)
            {
                var gridFrame = startF + i * gridDistanceF;

                var index = BinarySearchUtility.BinarySearch(frameSyncList, gridFrame, getFrameValue, BinarySearchUtility.Options.SmallestIndex);
                if (index < 0)
                {
                    index = ~index;
                }

                if (index == frameSyncList.Count)
                {
                    break;
                }

                var time = frameSyncList[index];
                var frame = getFrameValue(time);

                if (lastFrame == frame)
                {
                    continue;
                }

                var x = getXValue(time);
                var screenX = Math.Round((x - originX) * scaleX);

                if (screenX > this.ActualWidth)
                {
                    break;
                }

                if (pen.Thickness > 0)
                {
                    var offset = pen.Thickness % 2 >= 1 ? 0.5 : 0.0;
                    drawingContext.DrawLine(
                        pen,
                        new Point(screenX - offset, 0),
                        new Point(screenX - offset, this.ActualHeight));
                }

                if (this.FontSize > 0)
                {
                    if (screenX >= lastLabelRight)
                    {
                        var label = getLabel(frame);
                        var text = CreateFormattedText(label, typeface);
                        var origin = this.CreateLabelOrition(text, screenX);

                        drawingContext.DrawText(text, origin);

                        lastLabelRight = origin.X + text.Width;
                    }
                }

                lastFrame = frame;
            }

            return true;
        }

        private void DrawTimestampGrid(DrawingContext drawingContext)
        {
            var originX = this.OriginX;
            var scaleX = this.ScaleX;
            var gridDistance = this.CalcGridDistance();

            var pen = this.GetGridPen();

            var typeface = this.CreateTypeface();

            var getLabel = CreateTimestampLabelFunction();

            var startX = Math.Floor(originX / gridDistance) * gridDistance;
            var endX = originX + this.ActualWidth / scaleX;

            for (int i = 0; i <= MaxGridCount; i++)
            {
                var x = startX + i * gridDistance;
                var screenX = Math.Round((x - originX) * scaleX);

                if (screenX > this.ActualWidth)
                {
                    break;
                }

                if (pen.Thickness > 0)
                {
                    var offset = pen.Thickness % 2 >= 1 ? 0.5 : 0.0;
                    drawingContext.DrawLine(
                        pen,
                        new Point(screenX - offset, 0),
                        new Point(screenX - offset, this.ActualHeight));
                }

                if (this.FontSize > 0)
                {
                    var label = getLabel(x);
                    var text = this.CreateFormattedText(label, typeface);
                    var origin = this.CreateLabelOrition(text, screenX);

                    drawingContext.DrawText(text, origin);
                }
            }
        }

        private Typeface CreateTypeface()
        {
            return new Typeface(
                this.FontFamily,
                this.FontStyle,
                this.FontWeight,
                this.FontStretch);
        }

        private FormattedText CreateFormattedText(string label, Typeface typeface)
        {
            return new FormattedText(
                label,
                CultureInfo.CurrentCulture,
                this.FlowDirection,
                typeface,
                this.FontSize,
                this.Foreground);
        }

        private Pen GetGridPen()
        {
            if (this.GridPen != null)
            {
                return this.GridPen;
            }
            else
            {
                if (_gridPenCache == null)
                {
                    _gridPenCache = new Pen(this.Foreground, this.GridThickness) { DashStyle = this.GridDashStyle };
                    _gridPenCache.Freeze();
                }

                return _gridPenCache;
            }
        }

        private void InvalidateGridPenCache()
        {
            _gridPenCache = null;
        }

        private Func<double, string> CreateFrameLabelFunctiion()
        {
            if (this.FrameToStringValueConverter != null)
            {
                var formatter = this.FrameToStringValueConverter;
                return x => (string)formatter.Convert(x, typeof(string), null, CultureInfo.CurrentCulture);
            }
            else if (!string.IsNullOrEmpty(this.FrameStringFormat))
            {
                var format = this.FrameStringFormat;
                return x => string.Format(format, x);
            }
            else
            {
                return x => x.ToString();
            }
        }

        private Func<double, string> CreateTimestampLabelFunction()
        {
            if (this.TimestampToStringValueConverter != null)
            {
                var formatter = this.TimestampToStringValueConverter;
                object parameter = null;
                if (formatter is SpyTimeToStringValueConverter)
                {
                    parameter = new SpyTimeToStringValueConverter.Parameter()
                    {
                        Scale = this.ScaleX,
                    };
                }
                return x => (string)formatter.Convert(x, typeof(string), parameter, CultureInfo.CurrentCulture);
            }
            else if (!string.IsNullOrEmpty(this.TimestampStringFormat))
            {
                var format = this.TimestampStringFormat;
                return x => string.Format(format, x);
            }
            else
            {
                return x => x.ToString();
            }
        }

        private Point CreateLabelOrition(FormattedText text, double screenX)
        {
            if (this.LabelPosition == LabelPositionType.Bottom)
            {
                return new Point(screenX + this.LabelMargin, this.ActualHeight - this.LabelOffset - text.Height);
            }

            return new Point(screenX + this.LabelMargin, this.LabelOffset);
        }

        /// <summary>
        /// 目盛間隔を求めます。
        /// </summary>
        private double CalcGridDistance()
        {
            return CalcGridDistance(Math.Max(1, this.GridFrequency) / this.ScaleX);
        }

        /// <summary>
        /// 目盛間隔を求めます。
        /// </summary>
        /// <param name="minDistance">グリッド間隔として保証する最低の幅(X座標)</param>
        private double CalcGridDistance(double minDistance)
        {
            // minDistance 以下の 10 のべき乗のうち、最も大きい値を求めます。
            var distanceBase = Math.Pow(10.0, Math.Floor(Math.Log10(minDistance)));

            if (minDistance <= distanceBase * 1)
            {
                return distanceBase * 1;
            }
            else if (minDistance <= distanceBase * 2)
            {
                return distanceBase * 2;
            }
            else if (minDistance <= distanceBase * 5)
            {
                return distanceBase * 5;
            }
            else
            {
                return distanceBase * 10;
            }
        }

        private static bool IsValidFiniteDouble(object v)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidFiniteDouble(object v, Func<double, bool> predicate)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d) && predicate(d);
            }
            else
            {
                return false;
            }
        }

        private static PlotFrameGrid Self(object d)
        {
            return (PlotFrameGrid)d;
        }
    }
}
