﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using Nintendo.ToolFoundation.Windows.Primitives.Geometries;
using Nintendo.ToolFoundation.Windows.Primitives.Shapes;
using NintendoWare.Spy.Windows.Input;
using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Input;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows.Primitives
{
    /// <summary>
    /// キャンバスに対するユーザの操作に対応するクラスです。
    /// </summary>
    public class PlotCanvasController : ContentControl
    {
        private const double AutoScrollInterval = 20;
        private const double AutoScrollPixels = 20;
        private const string PartAdornerButtonZoomIn = "PART_AdornerButtonZoomIn";
        private const string PartAdornerButtonZoomOut = "PART_AdornerButtonZoomOut";

        /// <summary>
        /// X 座標のスケール中心のモードです。
        /// </summary>
        public enum ScaleCenterModeX
        {
            /// <summary>
            /// マウスカーソルの位置をスケール中心になります。
            /// </summary>
            Cursor,

            /// <summary>
            /// 表示範囲の左端がスケール中心になります。
            /// </summary>
            OriginX,
        }

        /// <summary>
        /// 現在の X 座標を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty CurrentXProperty = DependencyProperty.Register(
            nameof(CurrentX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                0.0,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnCurrentXChanged(e),
                (s, v) => Self(s).CoerceCurrentX((double)v)),
                ValidateCurrentX);

        /// <summary>
        /// X 軸方向の最小値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty MinimumXProperty = DependencyProperty.Register(
            nameof(MinimumX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                double.NaN,
                (s, e) => Self(s).OnMinimumXChanged(e)));

        /// <summary>
        /// X 軸方向の最大値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty MaximumXProperty = DependencyProperty.Register(
            nameof(MaximumX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                double.NaN,
                (s, e) => Self(s).OnMaximumXChanged(e),
                (s, v) => Self(s).CoerceMaximumX((double)v)));

        /// <summary>
        /// 表示範囲の左端(X 座標)を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty OriginXProperty = DependencyProperty.Register(
            nameof(OriginX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                0.0,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnOriginXChanged(e),
                (s, v) => Self(s).CoerceViewportLeft((double)v)),
                ValidateOriginX);

        /// <summary>
        /// X 軸方向のスケールを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty ScaleXProperty = DependencyProperty.Register(
            nameof(ScaleX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                1.0,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnScaleXChanged(e),
                (s, v) => Self(s).CoerceScaleX((double)v)));

        /// <summary>
        /// X 軸方向のスケールの最小値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty MinimumScaleXProperty = DependencyProperty.Register(
            nameof(MinimumScaleX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                double.NaN,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnMinimumScaleXChanged(e)));

        /// <summary>
        /// X 軸方向のスケールの最大値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty MaximumScaleXProperty = DependencyProperty.Register(
            nameof(MaximumScaleX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                32.0,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnMaximumScaleXChanged(e),
                (s, v) => Self(s).CoerceMaximumScaleX((double)v)));

        /// <summary>
        /// マウスホイールによるズーム操作における X 座標のスケール中心のモードを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty MouseWheelZoomCenterModeXProperty = DependencyProperty.Register(
            nameof(MouseWheelScaleCenterModeX),
            typeof(ScaleCenterModeX),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(ScaleCenterModeX.Cursor));

        /// <summary>
        /// Y 軸方向のスケールを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty ScaleYProperty = DependencyProperty.Register(
            nameof(ScaleY),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                1.0,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (s, e) => Self(s).OnScaleYChanged(e)));

        /// <summary>
        /// アイテムのX 軸方向の最小値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty ItemsMinimumXProperty = DependencyProperty.Register(
            nameof(ItemsMinimumX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                double.NaN,
                (s, e) => Self(s).OnItemsMinimumXChanged(e)));

        /// <summary>
        /// アイテムのX 軸方向の最大値を示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty ItemsMaximumXProperty = DependencyProperty.Register(
            nameof(ItemsMaximumX),
            typeof(double),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(
                double.NaN,
                (s, e) => Self(s).OnItemsMaximumXChanged(e),
                (s, v) => Self(s).CoerceItemsMaximumX((double)v)));

        /// <summary>
        /// マウスホイールによる横方向スケーリング操作用の修飾キーを示す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty HorizontalScalingModifierKeysProperty = DependencyProperty.Register(
            nameof(HorizontalScalingModifierKeys),
            typeof(ModifierKeys),
            typeof(PlotCanvasController),
            new FrameworkPropertyMetadata(ModifierKeys.Control));

        private readonly ShapeViewTransform _viewTransform = new ShapeViewTransform();

        private LineChartScale _currentScaleX = LineChartScale.NoScaling;
        private double? _screenScaleCenterX = null;
        private double _currentScaleY = 1.0f;

        private Point? _dragOrigin;
        private DispatcherTimer _autoScrollTimer;

        static PlotCanvasController()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(PlotCanvasController), new FrameworkPropertyMetadata(typeof(PlotCanvasController)));
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PlotCanvasController()
        {
            _viewTransform = new ShapeViewTransform()
            {
                Screen = new GeometryRectangle(new Vector2(0, 0), 0, 0),
                World = new GeometryRectangle(new Vector2(0, 0), 0, 0),
            };

            AddHandler(ButtonBase.ClickEvent, (RoutedEventHandler)OnButtonClicked);
            AddHandler(MouseTilt.MouseTiltEvent, (MouseTilt.MouseTiltEventHandler)OnMouseTilt);
        }

        /// <summary>
        /// 現在の X 座標を取得または設定します。
        /// </summary>
        public double CurrentX
        {
            get { return (double)this.GetValue(CurrentXProperty); }
            set { this.SetValue(CurrentXProperty, value); }
        }

        /// <summary>
        /// X 軸方向の最小値を取得または設定します。
        /// 初期値は、double.NaN です。
        /// </summary>
        public double MinimumX
        {
            get { return (double)this.GetValue(MinimumXProperty); }
            set { this.SetValue(MinimumXProperty, value); }
        }

        /// <summary>
        /// X 軸方向の最大値を取得または設定します。
        /// 初期値は、double.NaN です。
        /// </summary>
        public double MaximumX
        {
            get { return (double)this.GetValue(MaximumXProperty); }
            set { this.SetValue(MaximumXProperty, value); }
        }

        /// <summary>
        /// 表示範囲の左端(X 座標)を取得または設定します。
        /// 初期値は 0 です。
        /// </summary>
        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        /// <summary>
        /// X 軸方向のスケールを取得または設定します。
        /// </summary>
        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        /// <summary>
        /// Y 軸方向のスケールを取得または設定します。
        /// </summary>
        public double ScaleY
        {
            get { return (double)this.GetValue(ScaleYProperty); }
            set { this.SetValue(ScaleYProperty, value); }
        }

        /// <summary>
        /// アイテムの X 軸方向の最小値を取得または設定します。
        /// 初期値は、double.Nan です。
        /// </summary>
        public double ItemsMinimumX
        {
            get { return (double)this.GetValue(ItemsMinimumXProperty); }
            set { this.SetValue(ItemsMinimumXProperty, value); }
        }

        /// <summary>
        /// アイテムの X 軸方向の最大値を取得または設定します。
        /// 初期値は、double.NaN です。
        /// </summary>
        public double ItemsMaximumX
        {
            get { return (double)this.GetValue(ItemsMaximumXProperty); }
            set { this.SetValue(ItemsMaximumXProperty, value); }
        }

        /// <summary>
        /// マウスホイールによる横方向スケーリング操作用の修飾キーを取得または設定します。
        /// </summary>
        public ModifierKeys HorizontalScalingModifierKeys
        {
            get { return (ModifierKeys)this.GetValue(HorizontalScalingModifierKeysProperty); }
            set { this.SetValue(HorizontalScalingModifierKeysProperty, value); }
        }

        /// <summary>
        /// X 軸方向のスケールの最小値を取得または設定します。
        /// </summary>
        public double MinimumScaleX
        {
            get { return (double)this.GetValue(MinimumScaleXProperty); }
            set { this.SetValue(MinimumScaleXProperty, value); }
        }

        /// <summary>
        /// X 軸方向のスケールの最大値を取得または設定します。
        /// </summary>
        public double MaximumScaleX
        {
            get { return (double)this.GetValue(MaximumScaleXProperty); }
            set { this.SetValue(MaximumScaleXProperty, value); }
        }

        /// <summary>
        /// マウスホイールによるズーム操作における X 座標のスケール中心のモードを取得または設定します。
        /// </summary>
        public ScaleCenterModeX MouseWheelScaleCenterModeX
        {
            get { return (ScaleCenterModeX)this.GetValue(MouseWheelZoomCenterModeXProperty); }
            set { this.SetValue(MouseWheelZoomCenterModeXProperty, value); }
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            _viewTransform.Screen.Width = sizeInfo.NewSize.Width;
            _viewTransform.Screen.Height = sizeInfo.NewSize.Height;

            this.SetScaleCenterX(null);
            this.UpdateView();
        }

        protected override void OnMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            base.OnMouseLeftButtonDown(e);

            var position = e.MouseDevice.GetPosition(this);
            if (position.X < 0 || this.ActualWidth < position.X)
            {
                return;
            }

            var worldPositionX = this.TransformScreenXToWorldX(position.X);

            this.SetCurrentValue(CurrentXProperty, worldPositionX);

            this.CaptureMouse();
        }

        protected override void OnMouseLeftButtonUp(MouseButtonEventArgs e)
        {
            if (this.IsMouseCaptured)
            {
                this.ReleaseMouseCapture();
            }

            base.OnMouseLeftButtonUp(e);
        }

        protected override void OnMouseRightButtonDown(MouseButtonEventArgs e)
        {
            base.OnMouseRightButtonDown(e);

            var position = e.MouseDevice.GetPosition(this);
            if (position.X < 0 || this.ActualWidth < position.X)
            {
                return;
            }

            _dragOrigin = position;

            this.CaptureMouse();
        }

        protected override void OnMouseRightButtonUp(MouseButtonEventArgs e)
        {
            if (this.IsMouseCaptured)
            {
                this.ReleaseMouseCapture();
            }

            base.OnMouseRightButtonUp(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.IsMouseCaptured)
            {
                if (e.LeftButton == MouseButtonState.Pressed)
                {
                    this.OnMouseLeftButtonDrag(e);
                }
                else if (e.RightButton == MouseButtonState.Pressed)
                {
                    this.OnMouseRightButtonDrag(e);
                }
            }

            base.OnMouseMove(e);
        }

        protected override void OnPreviewMouseWheel(MouseWheelEventArgs e)
        {
            base.OnPreviewMouseWheel(e);

            var position = e.MouseDevice.GetPosition(this);
            if (position.X < 0 || this.ActualWidth < position.X)
            {
                return;
            }

            // 拡大
            if (e.Delta > 0)
            {
                if (Keyboard.Modifiers == this.HorizontalScalingModifierKeys)
                {
                    this.SetScaleXOnPreviewMouseWheel(_currentScaleX.ScaleUp(), position.X);
                    e.Handled = true;
                }
            }
            // 縮小
            else
            {
                if (Keyboard.Modifiers == this.HorizontalScalingModifierKeys)
                {
                    this.SetScaleXOnPreviewMouseWheel(_currentScaleX.ScaleDown(), position.X);
                    e.Handled = true;
                }
            }
        }

        protected override void OnIsMouseCapturedChanged(DependencyPropertyChangedEventArgs e)
        {
            if (!this.IsMouseCaptured)
            {
                _dragOrigin = null;
            }

            this.StopAutoScroll();

            base.OnIsMouseCapturedChanged(e);
        }

        private void OnButtonClicked(object sender, RoutedEventArgs e)
        {
            if (e.OriginalSource is ButtonBase button)
            {
                switch (button.Name)
                {
                    case PartAdornerButtonZoomIn:
                        ZoomIn();
                        e.Handled = true;
                        break;

                    case PartAdornerButtonZoomOut:
                        ZoomOut();
                        e.Handled = true;
                        break;
                }
            }
        }

        private void OnMouseTilt(object sender, MouseTiltEventArgs args)
        {
            var lastValue = this.OriginX;

            var value = this.TransformScreenXToWorldX(AutoScrollPixels * -args.Tilt / MouseTilt.Delta);
            this.SetCurrentValue(OriginXProperty, value);

            if (lastValue != this.OriginX)
            {
                args.Handled = true;
            }
        }

        private void OnCurrentXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.BringXIntoView(this.CurrentX);
        }

        private static bool ValidateCurrentX(object value)
        {
            return !double.IsNaN((double)value);
        }

        private void OnMinimumXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumXProperty);
            this.CoerceValue(CurrentXProperty);
            this.CoerceValue(OriginXProperty);
        }

        private void OnMaximumXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(CurrentXProperty);
            this.CoerceValue(OriginXProperty);
        }

        private double CoerceMaximumX(double value)
        {
            var minimumX = this.MinimumX;
            if (!double.IsNaN(minimumX))
            {
                return Math.Max(minimumX, value);
            }
            else
            {
                return value;
            }
        }

        private void OnOriginXChanged(DependencyPropertyChangedEventArgs e)
        {
            _viewTransform.World.X = this.OriginX;
        }

        private static bool ValidateOriginX(object value)
        {
            return !double.IsNaN((double)value);
        }

        private void OnScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.UpdateView();
        }

        private void OnScaleYChanged(DependencyPropertyChangedEventArgs e)
        {
            this.SetScaleY(this.ScaleY);
        }

        private void OnItemsMinimumXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(ItemsMaximumXProperty);
            this.CoerceValue(CurrentXProperty);
            this.CoerceValue(OriginXProperty);
        }

        private void OnItemsMaximumXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(CurrentXProperty);
            this.CoerceValue(OriginXProperty);
        }

        private void OnMinimumScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumScaleXProperty);
            this.CoerceValue(ScaleXProperty);
        }

        private void OnMaximumScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(ScaleXProperty);
        }

        private double CoerceMaximumScaleX(double value)
        {
            if (value < this.MinimumScaleX)
            {
                value = this.MinimumScaleX;
            }

            return value;
        }

        private object CoerceItemsMaximumX(double value)
        {
            var itemsMinimumX = this.ItemsMinimumX;
            if (!double.IsNaN(itemsMinimumX))
            {
                return Math.Max(itemsMinimumX, value);
            }
            else
            {
                return value;
            }
        }

        private static ShapeViewTransform CloneViewTransform(ShapeViewTransform viewTransform)
        {
            return new ShapeViewTransform()
            {
                Screen = new GeometryRectangle(viewTransform.Screen),
                World = new GeometryRectangle(viewTransform.World),
            };
        }

        private double TransformScreenXToWorldX(double screenX)
        {
            return _viewTransform.TransformScreenToWorld(new Vector2(screenX, 0)).X;
        }

        private double TransformWorldXToScreenX(double screenX)
        {
            return _viewTransform.TransformWorldToScreen(new Vector2(screenX, 0)).X;
        }

        private double GetValidMaximumX()
        {
            Assertion.Operation.False(double.IsNegativeInfinity(this.MaximumX));

            var result = double.IsNaN(this.MaximumX)
                ? this.ItemsMaximumX
                : this.MaximumX;

            Assertion.Operation.False(double.IsNegativeInfinity(result));
            return result;
        }

        private double GetValidMinimumX()
        {
            Assertion.Operation.False(double.IsPositiveInfinity(this.MinimumX));

            var result = double.IsNaN(this.MinimumX)
                ? this.ItemsMinimumX
                : this.MinimumX;

            Assertion.Operation.False(double.IsPositiveInfinity(result));
            return result;
        }

        private double GetValidWidth()
        {
            var maximumX = this.GetValidMaximumX();
            if (double.IsNaN(maximumX) || double.IsPositiveInfinity(maximumX))
            {
                return double.NaN;
            }

            var minimumX = this.GetValidMinimumX();
            if (double.IsNaN(minimumX) || double.IsNegativeInfinity(minimumX))
            {
                return double.NaN;
            }

            Assertion.Operation.True(minimumX <= maximumX);
            return maximumX - minimumX;
        }

        /// <summary>
        /// X 方向スケールの中心になるスクリーン座標を指定します。
        /// null を指定した場合は、スクリーン左端がスケール中心になります。
        /// </summary>
        /// <param name="screenScaleCenterX"></param>
        private void SetScaleCenterX(double? screenScaleCenterX)
        {
            _screenScaleCenterX = screenScaleCenterX;
        }

        /// <summary>
        /// 現在のスケール設定、アイテムの状態をつかって、ViewTransform を更新します。
        /// スケール中心が設定されているときは、スケール中心の位置が変わらないように OriginX が調整されます。
        /// </summary>
        private void UpdateView()
        {
            // スケール変更前の座標を保持
            var worldX = _viewTransform.World.X;
            var worldScaleCenterX = _screenScaleCenterX.HasValue ? this.TransformScreenXToWorldX(_screenScaleCenterX.Value) : 0.0;

            // スケールの変更（幅と高さの更新）
            _viewTransform.World.Width = _viewTransform.Screen.Width / _currentScaleX.ToValue();
            _viewTransform.World.Height = _viewTransform.Screen.Height / _currentScaleY;

            if (_screenScaleCenterX.HasValue)
            {
                // スケール変更の前後でスケール中心が変わらないように位置調整
                var updatedWorldScaleCenterX = this.TransformScreenXToWorldX(_screenScaleCenterX.Value);

                worldX = _viewTransform.World.X + (worldScaleCenterX - updatedWorldScaleCenterX);
            }

            this.SetCurrentValue(OriginXProperty, this.CoerceViewportLeft(worldX));
        }

        private void SetViewportLeft(double value)
        {
            this.SetScaleCenterX(null);
            this.SetCurrentValue(OriginXProperty, this.CoerceViewportLeft(value));
        }

        private double CoerceViewportLeft(double value)
        {
            var result = value;

            var maximumX = this.GetValidMaximumX();
            if (!double.IsNaN(maximumX))
            {
                result = Math.Min(maximumX - _viewTransform.World.Width, result);
            }

            var minimumX = this.GetValidMinimumX();
            if (!double.IsNaN(minimumX))
            {
                result = Math.Max(minimumX, result);
            }

            return result;
        }

        private double CoerceScaleX(double value)
        {
            if (value < this.MinimumScaleX)
            {
                value = this.MinimumScaleX;
            }

            if (value > this.MaximumScaleX)
            {
                value = this.MaximumScaleX;
            }

            _currentScaleX = LineChartScale.FromValue(value, false);
            return value;
        }

        private void SetScaleXOnPreviewMouseWheel(LineChartScale scaleX, double targetX)
        {
            switch (this.MouseWheelScaleCenterModeX)
            {
                case ScaleCenterModeX.Cursor:
                    this.SetScaleCenterX(targetX);
                    break;

                case ScaleCenterModeX.OriginX:
                    this.SetScaleCenterX(null);
                    break;

                default:
                    throw new NotImplementedException();
            }

            this.SetCurrentValue(ScaleXProperty, scaleX.ToValue());

            this.UpdateView();
        }

        private void ZoomIn()
        {
            this.SetScaleCenterX(null);
            this.SetCurrentValue(ScaleXProperty, _currentScaleX.ScaleUp().ToValue());
            this.UpdateView();
        }

        private void ZoomOut()
        {
            this.SetScaleCenterX(null);
            this.SetCurrentValue(ScaleXProperty, _currentScaleX.ScaleDown().ToValue());
            this.UpdateView();
        }

        private void SetScaleY(double scaleY)
        {
            if (scaleY == _currentScaleY)
            {
                return;
            }

            // とりあえず 1/32 倍 ～ 32 倍まで許可する
            // 必要に応じてカスタマイズ可能にする
            if (scaleY > 32.0)
            {
                return;
            }
            if (scaleY < 1.0 / 32.0)
            {
                return;
            }

            _currentScaleY = scaleY;
            this.SetCurrentValue(ScaleYProperty, scaleY);

            this.UpdateView();
        }

        private void BringXIntoView(double worldX)
        {
            if (_viewTransform.World.MaxX <= worldX)
            {
                // 画面上に CurrentX ラインを表示するために + 2 している
                this.SetViewportLeft(
                    _viewTransform.World.X + worldX - _viewTransform.World.MaxX + 2);
            }

            if (worldX < _viewTransform.World.MinX)
            {
                this.SetViewportLeft(
                    _viewTransform.World.X - (_viewTransform.World.MinX - worldX));
            }
        }

        private void StartAutoScroll(double delta)
        {
            if (_autoScrollTimer != null)
            {
                return;
            }

            _autoScrollTimer = new DispatcherTimer(DispatcherPriority.Input)
            {
                Interval = TimeSpan.FromMilliseconds(AutoScrollInterval),
            };
            _autoScrollTimer.Tick += (sender, e) =>
            {
                var screenX = delta <= 0 ? delta : this.ActualWidth + delta;
                this.SetCurrentValue(CurrentXProperty, this.TransformScreenXToWorldX(screenX));
            };

            _autoScrollTimer.Start();
        }

        private void StopAutoScroll()
        {
            if (_autoScrollTimer != null)
            {
                _autoScrollTimer.Stop();
                _autoScrollTimer = null;
            }
        }

        private double CoerceCurrentX(double x)
        {
            double result = x;

            var minimumX = this.GetValidMinimumX();
            if (!double.IsNaN(minimumX))
            {
                result = Math.Max(minimumX, result);
            }

            var maximumX = this.GetValidMaximumX();
            if (!double.IsNaN(maximumX))
            {
                result = Math.Min(maximumX, result);
            }

            return result;
        }

        private void OnMouseLeftButtonDrag(MouseEventArgs e)
        {
            // コントロール内なら、現在位置を更新
            // コントロール外なら、スクロールして現在位置を更新
            // UIElement.IsMouseOver では検知できないので、座標で判断
            double screenPositionX = e.MouseDevice.GetPosition(this).X;

            if (this.ActualWidth <= screenPositionX)
            {
                this.StartAutoScroll(AutoScrollPixels);
            }
            else if (screenPositionX < 0)
            {
                this.StartAutoScroll(-AutoScrollPixels);
            }
            else
            {
                this.StopAutoScroll();
                this.SetCurrentValue(CurrentXProperty, this.TransformScreenXToWorldX(screenPositionX));
            }
        }

        private void OnMouseRightButtonDrag(MouseEventArgs e)
        {
            if (_dragOrigin == null)
            {
                return;
            }

            var mousePosition = e.MouseDevice.GetPosition(this);

            this.SetViewportLeft(
                this.TransformScreenXToWorldX(_dragOrigin.Value.X - mousePosition.X));

            _dragOrigin = mousePosition;
        }

        private static PlotCanvasController Self(DependencyObject obj)
        {
            return (PlotCanvasController)obj;
        }
    }
}
