﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows.Primitives
{
    /// <summary>
    /// アイテムの配置に対してオフセットとスケールを設定できるようにしたキャンバスです。
    ///
    /// アイテムの配置は (Canvas.Left - OriginX) * ScaleX で計算されます。
    ///
    /// アイテムの矩形の右端の X 座標も指定したい場合には PlotCanvas.Left2 添付プロパティを使います。
    /// </summary>
    public class PlotCanvas : Canvas
    {
        // 依存プロパティ

        /// <summary>
        /// キャンバスの左端の X 座標を表す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty OriginXProperty =
            DependencyProperty.Register(
                nameof(OriginX),
                typeof(double),
                typeof(PlotCanvas),
                new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnOriginXChanged(e)));

        /// <summary>
        /// アイテムの X 座標に対するスケールを表す依存プロパティです。
        /// </summary>
        public static readonly DependencyProperty ScaleXProperty =
            DependencyProperty.Register(
                nameof(ScaleX),
                typeof(double),
                typeof(PlotCanvas),
                new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnScaleXChanged(e)));

        // 添付プロパティ

        /// <summary>
        /// アイテムの矩形の左端の X 座標を表す添付プロパティです。
        /// </summary>
        public static readonly DependencyProperty LeftXProperty =
            DependencyProperty.RegisterAttached(
                "LeftX",
                typeof(double),
                typeof(PlotCanvas),
                new FrameworkPropertyMetadata(double.NaN, OnLeftXChanged));

        public static double GetLeftX(UIElement target)
        {
            return (double)target.GetValue(LeftXProperty);
        }

        public static void SetLeftX(UIElement target, double value)
        {
            target.SetValue(LeftXProperty, value);
        }

        /// <summary>
        /// アイテムの矩形の右端の X 座標を表す添付プロパティです。
        /// </summary>
        public static readonly DependencyProperty RightXProperty =
            DependencyProperty.RegisterAttached(
                "RightX",
                typeof(double),
                typeof(PlotCanvas),
                new FrameworkPropertyMetadata(double.NaN, OnRightXChanged));

        public static double GetRightX(UIElement target)
        {
            return (double)target.GetValue(RightXProperty);
        }

        public static void SetRightX(UIElement target, double value)
        {
            target.SetValue(RightXProperty, value);
        }

        // ロジック

        static PlotCanvas()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(PlotCanvas),
                new FrameworkPropertyMetadata(typeof(PlotCanvas)));
        }

        /// <summary>
        /// OriginX 依存プロパティの値を取得または設定します。
        /// アイテムの位置は ( Canvas.Left - OriginX ) * ScaleX で計算されます。
        /// </summary>
        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        /// <summary>
        /// ScaleX 依存プロパティの値を取得または設定します。
        /// アイテムの位置は ( Canvas.Left - OriginX ) * ScaleX で計算されます。
        /// </summary>
        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        private double ValidatedOriginX
        {
            get { return GetValidatedOriginX(this.OriginX); }
        }

        private double ValidatedScaleX
        {
            get { return GetValidatedScaleX(this.ScaleX); }
        }

        private void OnOriginXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateArrange();
        }

        private void OnScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateMeasure();
        }

        private static void OnLeftXChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var canvas = VisualTreeHelper.GetParent(d) as PlotCanvas;
            if (canvas != null)
            {
                canvas.InvalidateArrange();
            }
        }

        private static void OnRightXChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var canvas = VisualTreeHelper.GetParent(d) as PlotCanvas;
            if (canvas != null)
            {
                canvas.InvalidateArrange();
            }
        }

        public static double GetValidatedOriginX(double value)
        {
            return double.IsNaN(value) ? 0.0 : value;
        }

        public static double GetValidatedScaleX(double value)
        {
            return double.IsNaN(value) ? 1.0 : Math.Max(double.Epsilon, value);
        }

        protected override Size MeasureOverride(Size constraint)
        {
            var originX = this.ValidatedOriginX;
            var scaleX = this.ValidatedScaleX;

            foreach (var child in this.InternalChildren.OfType<UIElement>())
            {
                var availableSize = new Size(double.PositiveInfinity, double.PositiveInfinity);
                var leftX = GetLeftX(child);
                if (!double.IsNaN(leftX))
                {
                    leftX = (leftX - originX) * scaleX;

                    var rightX = GetRightX(child);
                    if (!double.IsNaN(rightX))
                    {
                        rightX = (rightX - originX) * scaleX;
                        availableSize.Width = Math.Max(0, rightX - leftX);
                    }
                }

                child.Measure(availableSize);
            }

            return default(Size);
        }

        protected override Size ArrangeOverride(Size arrangeSize)
        {
            var originX = this.ValidatedOriginX;
            var scaleX = this.ValidatedScaleX;

            foreach (var child in this.InternalChildren.OfType<UIElement>())
            {
                double x = 0;
                double y = 0;
                double width = child.DesiredSize.Width;

                var leftX = GetLeftX(child);
                if (!double.IsNaN(leftX))
                {
                    leftX = (leftX - originX) * scaleX;

                    x = leftX;

                    var rightX = GetRightX(child);
                    if (!double.IsNaN(rightX))
                    {
                        rightX = (rightX - originX) * scaleX;

                        width = Math.Max(0, rightX - leftX);
                    }
                }
                else
                {
                    var left = Canvas.GetLeft(child);
                    if (!double.IsNaN(left))
                    {
                        x = left;
                    }
                    else
                    {
                        var right = Canvas.GetRight(child);
                        if (!double.IsNaN(right))
                        {
                            x = arrangeSize.Width - right - width;
                        }
                    }
                }

                var top = GetTop(child);
                var bottom = GetBottom(child);
                var height = 0d;

                if (!double.IsNaN(top))
                {
                    y = top;
                    height = arrangeSize.Height - top;
                }
                else if (!double.IsNaN(bottom))
                {
                    // TODO : 必要になったら実装する
                    throw new NotImplementedException();
                }
                else
                {
                    height = arrangeSize.Height;
                }

                x = Math.Ceiling(x);
                width = Math.Max(1d, width);

                child.Arrange(new Rect(new Point(x, y), new Size(width, height)));
            }

            return arrangeSize;
        }

        private static PlotCanvas Self(DependencyObject d)
        {
            return (PlotCanvas)d;
        }
    }
}
