﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Globalization;
using System.Linq;
using System.Windows;
using System.Windows.Data;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// 要素数を真偽値に変換します。
    /// </summary>
    [ValueConversion(typeof(IEnumerable), typeof(bool))]
    public sealed class ItemCount2BoolConverter : IValueConverter
    {
        /// <summary>
        /// オペレータの種別を示します。
        /// </summary>
        public enum OperatorType
        {
            Equal,
            Less,
            LessEqual,
            Greater,
            GreaterEqual,
        }

        /// <summary>
        /// オペレータを取得または設定します。
        /// デフォルト値は、Equal です。
        /// </summary>
        public OperatorType Operator { get; set; } = OperatorType.Equal;

        /// <summary>
        /// 比較の基準となるアイテム数を取得または設定します。
        /// デフォルト値は、0 です。
        /// </summary>
        public int ItemCount { get; set; }

        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            var items = value as IEnumerable;

            if (items == null)
            {
                return false;
            }

            var collection = items as ICollection;

            return this.Operate(collection != null ? collection.Count : items.Cast<object>().Count());
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return DependencyProperty.UnsetValue;
        }

        private bool Operate(int value)
        {
            switch (this.Operator)
            {
                case OperatorType.Equal:
                    return this.ItemCount == value;

                case OperatorType.Less:
                    return this.ItemCount > value;

                case OperatorType.LessEqual:
                    return this.ItemCount >= value;

                case OperatorType.Greater:
                    return this.ItemCount < value;

                case OperatorType.GreaterEqual:
                    return this.ItemCount <= value;
            }

            throw new NotSupportedException("サポートしていないオペレータです。");
        }
    }
}
