﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Interop;

namespace NintendoWare.Spy.Windows.Input
{
    public static class MouseTilt
    {
        /// <summary>
        /// 横方向に１単位スクロールする傾き量です。
        /// </summary>
        public const int Delta = 120;

        private static MouseTiltEventManager _eventManager;
        private static MouseTiltScrollController _scrollController;

        public delegate void MouseTiltEventHandler(object sender, MouseTiltEventArgs args);

        public static readonly RoutedEvent PreviewMouseTiltEvent =
            EventManager.RegisterRoutedEvent(
                "PreviewMouseTilt",
                RoutingStrategy.Tunnel,
                typeof(MouseTiltEventHandler),
                typeof(MouseTilt));

        public static void AddPreviewMouseTiltHandler(UIElement element, MouseTiltEventHandler handler)
        {
            element.AddHandler(PreviewMouseTiltEvent, handler);
        }

        public static void RemovePreviewMouseTiltHandler(UIElement element, MouseTiltEventHandler handler)
        {
            element.RemoveHandler(PreviewMouseTiltEvent, handler);
        }

        public static readonly RoutedEvent MouseTiltEvent =
            EventManager.RegisterRoutedEvent(
                "MouseTilt",
                RoutingStrategy.Bubble,
                typeof(MouseTiltEventHandler),
                typeof(MouseTilt));

        public static void AddMouseTiltHandler(UIElement element, MouseTiltEventHandler handler)
        {
            element.AddHandler(MouseTiltEvent, handler);
        }

        public static void RemoveMouseTiltHandler(UIElement element, MouseTiltEventHandler handler)
        {
            element.RemoveHandler(MouseTiltEvent, handler);
        }

        public static void Initialize()
        {
            if (_eventManager == null)
            {
                _eventManager = new MouseTiltEventManager();
                _eventManager.Initialize();
            }

            if (_scrollController == null)
            {
                _scrollController = new MouseTiltScrollController();
                _scrollController.Initialize();
            }
        }

        private class MouseTiltEventManager
        {
            public void Initialize()
            {
                EventManager.RegisterClassHandler(
                    typeof(Window),
                    FrameworkElement.LoadedEvent,
                    new RoutedEventHandler(Window_Loaded),
                    handledEventsToo: true);

                EventManager.RegisterClassHandler(
                    typeof(Window),
                    FrameworkElement.UnloadedEvent,
                    new RoutedEventHandler(Window_Unloaded),
                    handledEventsToo: true);
            }

            private void Window_Loaded(object sender, RoutedEventArgs args)
            {
                var window = (Window)sender;
                var source = PresentationSource.FromDependencyObject(window);
                (source as HwndSource)?.AddHook(Hook);
            }

            private void Window_Unloaded(object sender, RoutedEventArgs args)
            {
                var window = (Window)sender;
                var source = PresentationSource.FromDependencyObject(window);
                (source as HwndSource)?.RemoveHook(Hook);
            }

            private IntPtr Hook(IntPtr hwnd, int msg, IntPtr wParam, IntPtr lParam, ref bool handled)
            {
                short HIWORD(IntPtr value)
                {
                    return (short)((value.ToInt64() >> 16) & 0xffff);
                }

                switch (msg)
                {
                    case 0x020E: // WM_MOUSEHWHEEL
                        int tilt = HIWORD(wParam);
                        OnMouseTilt(tilt);
                        break;
                }

                return IntPtr.Zero;
            }

            private void OnMouseTilt(int tilt)
            {
                var target = Mouse.Captured ?? Mouse.DirectlyOver;

                if (target != null)
                {
                    var args = new MouseTiltEventArgs()
                    {
                        Tilt = -tilt, // 左への傾きを正とします。
                    };

                    args.RoutedEvent = MouseTilt.PreviewMouseTiltEvent;
                    target.RaiseEvent(args);

                    if (!args.Handled)
                    {
                        args.RoutedEvent = MouseTilt.MouseTiltEvent;
                        target.RaiseEvent(args);
                    }
                }
            }
        }

        private class MouseTiltScrollController
        {
            /// <summary>
            /// 蓄積された傾き量を保持します。
            /// </summary>
            private class AccumulatedTilt
            {
                public int Value { get; set; }
            }

            private readonly ConditionalWeakTable<object, AccumulatedTilt> _accumulatedTilts = new ConditionalWeakTable<object, AccumulatedTilt>();

            public void Initialize()
            {
                EventManager.RegisterClassHandler(
                    typeof(ScrollViewer),
                    MouseTiltEvent,
                    new MouseTiltEventHandler(OnScrollViewerMouseTilt));
            }

            private void OnScrollViewerMouseTilt(object sender, MouseTiltEventArgs args)
            {
                var scrollViewer = (ScrollViewer)sender;

                if (scrollViewer.HorizontalScrollBarVisibility == ScrollBarVisibility.Disabled)
                {
                    return;
                }

                var tilt = GetAccumulatedTiltValue(sender, args.Tilt);
                if (scrollViewer.FlowDirection == FlowDirection.RightToLeft)
                {
                    tilt = -tilt;
                }

                var horizontalOffset = scrollViewer.HorizontalOffset;

                if (tilt > 0)
                {
                    Enumerable.Repeat(0, tilt / Delta).ForEach(_ => scrollViewer.LineLeft());
                }
                else if (tilt < 0)
                {
                    Enumerable.Repeat(0, -tilt / Delta).ForEach(_ => scrollViewer.LineRight());
                }

                if (scrollViewer.HorizontalOffset != horizontalOffset)
                {
                    args.Handled = true;
                }
            }

            private int GetAccumulatedTiltValue(object sender, int tilt)
            {
                var lastTilt = _accumulatedTilts.GetOrCreateValue(sender);

                var result = lastTilt.Value + tilt;

                lastTilt.Value = result % Delta;

                return result;
            }
        }
    }
}
