﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;

namespace NintendoWare.Spy.Windows
{
    [TemplatePart(Name = TemplatePartHeaderColumnDefinitionName, Type = typeof(ColumnDefinition))]
    [TemplatePart(Name = TemplatePartHeaderThumbName, Type = typeof(Thumb))]
    [TemplatePart(Name = TemplatePartScrollViewer, Type = typeof(ScrollViewer))]
    public class HeaderedListBox : ItemsControl
    {
        private const string TemplatePartHeaderColumnDefinitionName = "HeaderColumnDefinition";
        private const string TemplatePartHeaderThumbName = "HeaderThumb";
        private const string TemplatePartScrollViewer = "PART_ScrollViewer";

        public static readonly DependencyProperty ItemHeaderTemplateProperty = DependencyProperty.Register(
            nameof(ItemHeaderTemplate),
            typeof(DataTemplate),
            typeof(HeaderedListBox),
            new FrameworkPropertyMetadata(null));

        public static readonly DependencyProperty ItemHeaderWidthProperty = DependencyProperty.Register(
            nameof(ItemHeaderWidth),
            typeof(double),
            typeof(HeaderedListBox),
            new FrameworkPropertyMetadata(
                120d,
                (sender, e) => ((HeaderedListBox)sender).OnItemHeaderWidthChanged(e),
                (d, value) => ((HeaderedListBox)d).CoerceItemHeaderWidth((double)value)));

        public static readonly DependencyProperty MaximumItemHeaderWidthProperty = DependencyProperty.Register(
            nameof(MaximumItemHeaderWidth),
            typeof(double),
            typeof(HeaderedListBox),
            new FrameworkPropertyMetadata(
                480d,
                (sender, e) => ((HeaderedListBox)sender).OnMaximumItemHeaderWidthChanged(e),
                (d, value) => ((HeaderedListBox)d).CoerceMaximumItemHeaderWidth((double)value)));

        public event EventHandler ItemHeaderWidthChanged;

        private ColumnDefinition _headerColumnDefinition;
        private Thumb _headerThumb;

        static HeaderedListBox()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(HeaderedListBox), new FrameworkPropertyMetadata(typeof(HeaderedListBox)));
        }

        public DataTemplate ItemHeaderTemplate
        {
            get { return (DataTemplate)GetValue(ItemHeaderTemplateProperty); }
            set { SetValue(ItemHeaderTemplateProperty, value); }
        }

        public double ItemHeaderWidth
        {
            get { return (double)GetValue(ItemHeaderWidthProperty); }
            set { SetValue(ItemHeaderWidthProperty, value); }
        }

        public double MaximumItemHeaderWidth
        {
            get { return (double)GetValue(MaximumItemHeaderWidthProperty); }
            set { SetValue(MaximumItemHeaderWidthProperty, value); }
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _headerColumnDefinition = this.Template.FindName(TemplatePartHeaderColumnDefinitionName, this) as ColumnDefinition;
            _headerThumb = this.Template.FindName(TemplatePartHeaderThumbName, this) as Thumb;

            if (_headerThumb != null)
            {
                _headerThumb.DragDelta += (sender, e) =>
                {
                    this.ItemHeaderWidth += e.HorizontalChange;
                };
            }
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.CoerceValue(HeaderedListBox.ItemHeaderWidthProperty);
        }

        protected override DependencyObject GetContainerForItemOverride()
        {
            return new HeaderedListBoxItem() { HeaderWidth = this.ItemHeaderWidth };
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            base.PrepareContainerForItemOverride(element, item);

            var itemContainer = (HeaderedListBoxItem)element;
            itemContainer.ContentTemplate = this.ItemTemplate;
            itemContainer.HeaderTemplate = this.ItemHeaderTemplate;
            itemContainer.Content = item;
        }

        private void OnItemHeaderWidthChanged(DependencyPropertyChangedEventArgs e)
        {
            foreach (var item in this.Items)
            {
                var container = (HeaderedListBoxItem)this.ItemContainerGenerator.ContainerFromItem(item);
                if (container != null)
                {
                    container.HeaderWidth = this.ItemHeaderWidth;
                }
            }

            if (this.ItemHeaderWidthChanged != null)
            {
                this.ItemHeaderWidthChanged(this, EventArgs.Empty);
            }
        }

        private void OnMaximumItemHeaderWidthChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(HeaderedListBox.ItemHeaderWidthProperty);
        }

        private double CoerceItemHeaderWidth(double baseValue)
        {
            if (_headerThumb == null)
            {
                return baseValue;
            }
            else
            {
                var value = Math.Min(this.ActualWidth, baseValue);
                value = Math.Min(value, this.MaximumItemHeaderWidth);
                value = Math.Max(value, _headerThumb.ActualWidth);

                _headerThumb.SetValue(Canvas.LeftProperty, baseValue - _headerThumb.ActualWidth);

                if (_headerColumnDefinition != null)
                {
                    _headerColumnDefinition.Width = new GridLength(value);
                }

                return value;
            }
        }

        private double CoerceMaximumItemHeaderWidth(double baseValue)
        {
            return baseValue < 0d ? 0d : baseValue;
        }
    }
}
