﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.InGameEditing.Communication;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace NintendoWare.Spy.Foundation.Communications.Htcs
{
    /// <summary>
    /// Htcs による HostIO 機能を提供します。
    /// </summary>
    public sealed class HostIOHtcs : IComEndPoint
    {
        public static readonly string SyncPortName = "NnSpy_Sync";
        public static readonly string DataPortName = "NnSpy_Data";

        private readonly object _connectLock = new object();

        private int _connectCount = 0;

        private ConnectionManager _connectionManager;

        private SynchronizationContext _syncContext;

        private readonly Func<TargetInfo, PortInfo, bool> _filterTargetPort;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public HostIOHtcs()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="filterTargetPort">接続対象のポートならば true を返します。</param>
        public HostIOHtcs(Func<TargetInfo, PortInfo, bool> filterTargetPort)
        {
            _filterTargetPort = filterTargetPort;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続の有無を取得します。
        /// </summary>
        public bool IsConnected
        {
            get
            {
                return _connectCount > 0 &&
                    _connectionManager != null;
            }
        }

        /// <summary>
        /// エンディアンを取得します。
        /// </summary>
        public bool IsLittleEndian
        {
            // TODO: SIGLO-18969 リトルエンディアン固定の修正
            get { return true; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="syncContext">同期コンテキストを指定します。</param>
        public void Connect(SynchronizationContext syncContext)
        {
            Ensure.Argument.NotNull(syncContext);
            Ensure.Argument.True(_syncContext == null || object.ReferenceEquals(_syncContext, syncContext));

            lock (_connectLock)
            {
                if (_connectCount == 0)
                {
                    _syncContext = syncContext;

                    ConnectionManager cm = null;
                    try
                    {
                        cm = new ConnectionManager();

                        cm.ConnectionInfoUpdated += (sender, arg) =>
                            {
                                DumpTargetInfos(arg.TargetInfos);
                            };

                        cm.Start();

                        DumpTargetInfos(cm.TargetInfos);
                    }
                    catch
                    {
                        try
                        {
                            if (cm != null)
                            {
                                cm.Dispose();
                            }
                        }
                        catch
                        {
                        }

                        throw;
                    }

                    _connectionManager = cm;
                }

                _connectCount++;
            }
        }

        private void DumpTargetInfos(IEnumerable<TargetInfo> targetInfos)
        {
            foreach (var targetInfo in targetInfos)
            {
                System.Diagnostics.Trace.WriteLine(string.Format("TargetInfo PeerType:{0} HtcsPeerName:{1}", targetInfo.PeerType, targetInfo.HtcsPeerName));
                foreach (var portInfo in targetInfo.PortInfos)
                {
                    bool isTarget = _filterTargetPort == null ? true : _filterTargetPort(targetInfo, portInfo);
                    System.Diagnostics.Trace.WriteLine(string.Format("  PortInfo PortName:{0} Number:{1} IsTarget:{2}", portInfo.PortName, portInfo.EndPoint.Port, isTarget));
                }
            }
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        public void Disconnect()
        {
            lock (_connectLock)
            {
                if (_connectCount == 0)
                {
                    return;
                }

                if (_connectCount == 1)
                {
                    if (_connectionManager != null)
                    {
                        _connectionManager.Dispose();
                        _connectionManager = null;
                    }
                }

                _connectCount--;
            }
        }

        public IComChannel OpenChannel(string portName)
        {
            lock (_connectLock)
            {
                if (_connectionManager == null)
                {
                    return null;
                }

                // TODO: SIGLO-19054 複数の接続先候補があるときどうするか
                // 現状は最初に見つかったポートに接続します。
                var portInfo = _connectionManager.TargetInfos
                    .SelectMany(t => t.PortInfos, (t, p) => new { TargetInfo = t, PortInfo = p })
                    .Where(it => _filterTargetPort == null || _filterTargetPort(it.TargetInfo, it.PortInfo))
                    .Select(it => it.PortInfo)
                    .FirstOrDefault(p => string.Equals(p.PortName, portName));

                if (portInfo == null)
                {
                    return null;
                }

                return HostIOHtcsChannel.Open(portInfo.EndPoint, portName);
            }
        }

        IComChannel IComEndPoint.OpenChannel(object port)
        {
            Ensure.Argument.True(port is string);
            return this.OpenChannel(port as string);
        }
    }
}
