﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;

namespace NintendoWare.Spy.Foundation.Binary
{
    internal class ObjectBinaryWriter
    {
        private const int BufferSize = 1024;

        private delegate void WriteObjectHandler(ObjectBinaryWriter self, object obj);

        private static Dictionary<Type, WriteObjectHandler> _handlers =
            new Dictionary<Type, WriteObjectHandler>();

        private readonly BinaryWriter _writer;

        //-----------------------------------------------------------------

        static ObjectBinaryWriter()
        {
            Initialize();
        }

        public ObjectBinaryWriter(BinaryWriter writer)
        {
            Ensure.Argument.NotNull(writer);
            _writer = writer;
        }

        //-----------------------------------------------------------------

        public BinaryWriter Writer
        {
            get
            {
                return _writer;
            }
        }

        public long Position
        {
            get { return _writer.BaseStream.Position; }
            set { _writer.BaseStream.Position = value; }
        }

        //-----------------------------------------------------------------

        public static bool IsTypeSupported(Type type)
        {
            return ObjectBinaryWriter.GetTargetType(type) != null;
        }

        public void Write(object obj)
        {
            Ensure.Argument.NotNull(obj);
            Ensure.Operation.NotNull(_writer);

            Type type = ObjectBinaryWriter.GetTargetType(obj.GetType());

            if (type == null)
            {
                throw new ArgumentException("unsupported type.");
            }

            _handlers[type](this, obj);
        }

        public void Flush()
        {
            Ensure.Operation.NotNull(_writer);
            _writer.Flush();
        }

        public void Align(int alignment)
        {
            _writer.Align(alignment);
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        private static void Initialize()
        {
            ObjectBinaryWriter._handlers.Add(typeof(byte), (ObjectBinaryWriter self, object obj) => self.Writer.Write((byte)obj));
            ObjectBinaryWriter._handlers.Add(typeof(Int16), (ObjectBinaryWriter self, object obj) => self.Writer.Write((Int16)obj));
            ObjectBinaryWriter._handlers.Add(typeof(Int32), (ObjectBinaryWriter self, object obj) => self.Writer.Write((Int32)obj));
            ObjectBinaryWriter._handlers.Add(typeof(Int64), (ObjectBinaryWriter self, object obj) => self.Writer.Write((Int64)obj));
            ObjectBinaryWriter._handlers.Add(typeof(UInt16), (ObjectBinaryWriter self, object obj) => self.Writer.Write((UInt16)obj));
            ObjectBinaryWriter._handlers.Add(typeof(UInt32), (ObjectBinaryWriter self, object obj) => self.Writer.Write((UInt32)obj));
            ObjectBinaryWriter._handlers.Add(typeof(UInt64), (ObjectBinaryWriter self, object obj) => self.Writer.Write((UInt64)obj));
            ObjectBinaryWriter._handlers.Add(typeof(float), (ObjectBinaryWriter self, object obj) => self.Writer.Write((float)obj));
            ObjectBinaryWriter._handlers.Add(typeof(bool), (ObjectBinaryWriter self, object obj) => self.Writer.Write((bool)obj));
            ObjectBinaryWriter._handlers.Add(typeof(string), (ObjectBinaryWriter self, object obj) => self.Writer.Write((obj as string).ToCharArray()));
            ObjectBinaryWriter._handlers.Add(typeof(char[]), (ObjectBinaryWriter self, object obj) => self.Writer.Write((char[])obj));
            ObjectBinaryWriter._handlers.Add(typeof(byte[]), (ObjectBinaryWriter self, object obj) => self.Writer.Write((byte[])obj));
            ObjectBinaryWriter._handlers.Add(typeof(Stream), WriteStream);
            ObjectBinaryWriter._handlers.Add(typeof(IBinarizable), WriteProperties);
            ObjectBinaryWriter._handlers.Add(typeof(IList), WriteList);
        }

        private static Type GetTargetType(Type type)
        {
            foreach (Type supportedType in _handlers.Keys)
            {
                if (type.IsSupported(supportedType))
                {
                    return supportedType;
                }
            }

            return null;
        }

        private static void WriteStream(ObjectBinaryWriter self, object obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Writer);

            Stream stream = obj as Stream;

            if (stream == null)
            {
                throw new InvalidDataException("[InternalError] obj must be Stream.");
            }

            byte[] buffer = new byte[BufferSize];

            while (true)
            {
                int readSize = stream.Read(buffer, 0, BufferSize);

                if (readSize <= 0)
                {
                    break;
                }

                self.Writer.Write(buffer, 0, readSize);
            }
        }

        private static void WriteProperties(ObjectBinaryWriter self, object obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Writer);

            var propertiesOwner = obj as IBinarizable;

            if (propertiesOwner == null)
            {
                throw new InvalidDataException("[InternalError] obj must be IPropertyEnumerable.");
            }

            foreach (var property in propertiesOwner.GetProperties())
            {
                Ensure.Operation.NotNull(property);
                self.Write(property.GetValue(obj, null));
            }
        }

        private static void WriteList(ObjectBinaryWriter self, object obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Writer);

            var list = (IList)obj;

            foreach (var item in list)
            {
                Ensure.Operation.NotNull(item);
                self.Write(item);
            }
        }
    }
}
