﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Binary;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Reflection;

namespace NintendoWare.Spy.Communication
{
    /// <summary>
    /// パケットヘッダです。
    /// </summary>
    [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
    internal class PacketHeader : IBinarizable
    {
        public const int SignatureSize = 4;

        private const uint ResultError = 0x80000000;

        private char[] _signature = new char[4];
        private UInt32 _id = (UInt32)PacketID.Invalid;
        private UInt32 _size = 0;
        private UInt32 _result;

        //-----------------------------------------------------------------

        public PacketHeader() { }

        public PacketHeader(string signature, uint id)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(signature);

            this.Signature = signature.ToCharArray();
            _id = id;
        }

        public PacketHeader(string signature, uint id, uint size)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(signature);

            this.Signature = signature.ToCharArray();
            _id = id;
            _size = size;
        }

        //-----------------------------------------------------------------

        public char[] Signature
        {
            get { return _signature; }
            set
            {
                Ensure.Argument.NotNull(value);
                Ensure.Argument.True(value.Length == 4);
                _signature = value;
            }
        }

        public UInt32 ID
        {
            get { return _id; }
            set { _id = value; }
        }

        public UInt32 Size
        {
            get { return _size; }
            set { _size = value; }
        }

        public UInt32 Result
        {
            get { return _result; }
            set { _result = value; }
        }

        /// <summary>
        /// Result プロパティが正常値の場合を示します。
        /// </summary>
        public bool IsResultSuccess
        {
            get { return (_result & ResultError) == 0; }
        }

        protected int SignatureSizeInternal
        {
            get { return 4; }
        }

        //-----------------------------------------------------------------

        public IEnumerable<PropertyInfo> GetProperties()
        {
            yield return this.GetProperty(target => target.Signature);
            yield return this.GetProperty(target => target.ID);
            yield return this.GetProperty(target => target.Size);
            yield return this.GetProperty(target => target.Result);

            foreach (var property in this.GetExtendedProperties())
            {
                yield return property;
            }
        }

        protected virtual IEnumerable<PropertyInfo> GetExtendedProperties()
        {
            yield break;
        }
    }
}
