﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Binary;
using System.Collections.Generic;
using System.Reflection;

namespace NintendoWare.Spy.Communication
{
    /// <summary>
    /// パケットのインターフェイスです。
    /// </summary>
    internal interface IPacket : IBinarizable
    {
        PacketHeader Header { get; }
        PacketBody Body { get; }
    }

    /// <summary>
    /// パケットの基本クラスです。
    /// </summary>
    /// <typeparam name="TPacketBody">パケットボディの型を指定します。</typeparam>
    internal class PacketBase<TPacketBody> : IPacket
        where TPacketBody : PacketBody, new()
    {
        private PacketHeader _header;
        private TPacketBody _body;

        //-----------------------------------------------------------------

        public PacketBase() :
            this(new PacketHeader(), new TPacketBody())
        {
        }

        public PacketBase(PacketHeader header) :
            this(header, new TPacketBody())
        {
        }

        public PacketBase(PacketHeader header, TPacketBody body)
        {
            Assertion.Argument.NotNull(header);
            Assertion.Argument.NotNull(body);
            _header = header;
            _body = body;

            _header.Size = _body.Size;

            _body.SizeChanged +=
                (sender, e) => _header.Size = _body.Size;
        }

        //-----------------------------------------------------------------

        public PacketHeader Header
        {
            get { return _header; }
        }

        public TPacketBody Body
        {
            get { return _body; }
        }

        PacketHeader IPacket.Header
        {
            get { return _header; }
        }

        PacketBody IPacket.Body
        {
            get { return _body; }
        }

        //-----------------------------------------------------------------

        public IEnumerable<PropertyInfo> GetProperties()
        {
            yield return this.GetProperty(target => target.Header);
            yield return this.GetProperty(target => target.Body);
        }
    }

    /// <summary>
    /// パケットクラスです。
    /// </summary>
    internal class Packet<TPacketBody> : PacketBase<TPacketBody>
         where TPacketBody : PacketBody, new()
    {
        public Packet()
        {
        }

        public Packet(string signature, uint id)
            : base(new PacketHeader(signature, id))
        {
        }
    }
}
