﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;

namespace NintendoWare.Spy.Collections
{
    /// <summary>
    /// 通知機能付きで読み取り専用の辞書です。
    /// </summary>
    /// <typeparam name="TKey">辞書内のキーの型です。</typeparam>
    /// <typeparam name="TValue">辞書内の値の型です。</typeparam>
    public sealed class ReadOnlyObservableDictionary<TKey, TValue> :
        ObservableObject,
        IDictionary<TKey, TValue>,
        IReadOnlyDictionary<TKey, TValue>,
        INotifyCollectionChanged
    {
        private const string ReadOnlyMessage = "ReadOnlyObservableDictionary: this dictionary is readonly.";

        private readonly IDictionary<TKey, TValue> _dictionary;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="dictionary">読み取り専用にラップしたい辞書です。</param>
        public ReadOnlyObservableDictionary(IDictionary<TKey, TValue> dictionary)
        {
            Ensure.Argument.NotNull(dictionary);

            _dictionary = dictionary;

            INotifyCollectionChanged collectionChanged = _dictionary as INotifyCollectionChanged;

            if (collectionChanged != null)
            {
                collectionChanged.CollectionChanged += this.OnCollectionChanged;
            }

            INotifyPropertyChanged propertyChanged = _dictionary as INotifyPropertyChanged;

            if (propertyChanged != null)
            {
                propertyChanged.PropertyChanged += this.OnDictionaryPropertyChanged;
            }
        }

        /// <summary>
        /// コレクション変更イベントです。
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        /// <summary>
        /// 格納されているキー/値ペアの数を取得します。
        /// </summary>
        public int Count
        {
            get { return _dictionary.Count; }
        }

        /// <summary>
        /// 格納されているキーのコレクションを取得します。
        /// </summary>
        public ICollection<TKey> Keys
        {
            get { return _dictionary.Keys; }
        }

        /// <summary>
        /// 格納されている値のコレクションを取得します。
        /// </summary>
        public ICollection<TValue> Values
        {
            get { return _dictionary.Values; }
        }

        /// <summary>
        /// 指定したキーの値を取得します。
        /// </summary>
        /// <param name="key">キーです。</param>
        /// <returns>指定したキーに関連づけられた値を返します。</returns>
        public TValue this[TKey key]
        {
            get { return _dictionary[key]; }
        }

        protected override void DisposeManagedInstance()
        {
            base.DisposeManagedInstance();

            if (_dictionary is INotifyCollectionChanged)
            {
                (_dictionary as INotifyCollectionChanged).CollectionChanged -= this.OnCollectionChanged;
            }

            if (_dictionary is INotifyPropertyChanged)
            {
                (_dictionary as INotifyPropertyChanged).PropertyChanged -= this.OnDictionaryPropertyChanged;
            }
        }

        /// <summary>
        /// 指定したキーが辞書に格納されているかどうかを判断します。
        /// </summary>
        /// <param name="key">検索されるキーです。</param>
        /// <returns>指定したキーを持つ要素が辞書に存在する場合は true、それ以外の場合は false を返します。</returns>
        public bool ContainsKey(TKey key)
        {
            return _dictionary.ContainsKey(key);
        }

        /// <summary>
        /// 指定したキーに関連付けられている値を取得します。
        /// </summary>
        /// <param name="key">取得する値のキーです。</param>
        /// <param name="value">指定したキーに関連付けられている値が格納されます。</param>
        /// <returns>存在する場合は true、それ以外の場合は false を返します。</returns>
        public bool TryGetValue(TKey key, out TValue value)
        {
            return _dictionary.TryGetValue(key, out value);
        }

        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.CollectionChanged?.Invoke(this, e);
        }

        private void OnDictionaryPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "Count")
            {
                this.OnPropertyChanged(e);
            }
        }

        bool ICollection<KeyValuePair<TKey, TValue>>.IsReadOnly
        {
            get { return true; }
        }

        bool ICollection<KeyValuePair<TKey, TValue>>.Contains(KeyValuePair<TKey, TValue> item)
        {
            return _dictionary.Contains(item);
        }

        IEnumerable<TKey> IReadOnlyDictionary<TKey, TValue>.Keys
        {
            get { return _dictionary.Keys; }
        }

        IEnumerable<TValue> IReadOnlyDictionary<TKey, TValue>.Values
        {
            get { return _dictionary.Values; }
        }

        TValue IDictionary<TKey, TValue>.this[TKey key]
        {
            get { return _dictionary[key]; }
            set { throw new NotSupportedException(ReadOnlyMessage); }
        }

        IEnumerator<KeyValuePair<TKey, TValue>> IEnumerable<KeyValuePair<TKey, TValue>>.GetEnumerator()
        {
            return _dictionary.GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return (_dictionary as IEnumerable).GetEnumerator();
        }

        void ICollection<KeyValuePair<TKey, TValue>>.CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex)
        {
            _dictionary.CopyTo(array, arrayIndex);
        }

        void IDictionary<TKey, TValue>.Add(TKey key, TValue value)
        {
            throw new NotSupportedException(ReadOnlyMessage);
        }

        bool IDictionary<TKey, TValue>.Remove(TKey key)
        {
            throw new NotSupportedException(ReadOnlyMessage);
        }

        void ICollection<KeyValuePair<TKey, TValue>>.Add(KeyValuePair<TKey, TValue> keyValuePair)
        {
            throw new NotSupportedException(ReadOnlyMessage);
        }

        bool ICollection<KeyValuePair<TKey, TValue>>.Remove(KeyValuePair<TKey, TValue> keyValuePair)
        {
            throw new NotSupportedException(ReadOnlyMessage);
        }

        void ICollection<KeyValuePair<TKey, TValue>>.Clear()
        {
            throw new NotSupportedException(ReadOnlyMessage);
        }
    }
}
