﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using System;
using System.IO;
using System.Linq;
using System.Text;

namespace NintendoWare.Spy.Binary
{
    internal class SpyDataBinaryReader : ObservableObject
    {
        private const uint SpyDataIDMax = 32;

        private static readonly Version ContainerVersionMinimum = SpyDataBinary.ContainerVersion_3_0_0_0;
        private static readonly Version ContainerVersionMaximum = SpyDataBinary.ContainerVersion_3_2_0_0;

        private static readonly ushort LittleEndianBom = 0xfeff;
        private static readonly ushort BigEndianBom = 0xfffe;

        //-----------------------------------------------------------------

        private BinaryReader _reader;
        private SpyDataBinaryHeader _header;

        //-----------------------------------------------------------------

        public bool IsOpened
        {
            get { return _reader != null; }
        }

        public SpyDataBinaryHeader Header
        {
            get { return _header; }
        }

        //-----------------------------------------------------------------

        public void Open(Stream stream)
        {
            Ensure.Argument.NotNull(stream);
            Ensure.Operation.Null(_reader);

            _reader = LittleEndianBinaryReader.Create(stream, Encoding.ASCII);
        }

        public void Close()
        {
            _header = null;
            this.Dispose();
        }

        public long GetCurrentPosition()
        {
            Ensure.Operation.True(this.IsOpened);
            return _reader.BaseStream.Position;
        }

        public void SetCurrentPosition(long value)
        {
            Ensure.Operation.True(this.IsOpened);
            _reader.BaseStream.Position = value;
        }

        public SpyDataBinaryHeader ReadHeader()
        {
            Ensure.Operation.True(this.IsOpened);

            if (_header != null)
            {
                return _header;
            }

            var result = new SpyDataBinaryHeader();

            var signature = _reader.ReadChars(4);
            Ensure.True(signature.Length == 4, () => new EndOfStreamException());
            Ensure.True(signature.SequenceEqual(SpyDataBinary.Signature), () => new InvalidDataException());

            var containerVersion = new Version(_reader.ReadByte(), _reader.ReadByte(), _reader.ReadByte(), _reader.ReadByte());
            Ensure.True(
                ContainerVersionMinimum <= containerVersion && containerVersion < ContainerVersionMaximum.IncrementMinor(),
                () => new InvalidDataException());

            result.ContainerVersion = containerVersion;

            var bom = _reader.ReadUInt16();

            if (bom == LittleEndianBom)
            {
                result.IsLittleEndian = true;
            }
            else
            {
                Ensure.True(bom == BigEndianBom, () => new InvalidDataException());
                result.IsLittleEndian = false;

                var newReader = BigEndianBinaryReader.Create(_reader.BaseStream, Encoding.ASCII);

                // newReader に引き継ぐので Close() しない
                _reader = null;
                _reader = newReader;
            }

            var nameLength = _reader.ReadUInt16();
            Ensure.True(nameLength > 0, () => new InvalidDataException());

            var dataVersion = _reader.ReadBytes(4);
            Ensure.True(dataVersion.Length == 4, () => new EndOfStreamException());
            result.DataVersion = new Version(dataVersion[0], dataVersion[1], dataVersion[2], dataVersion[3]);

            long dataBlockOffsetBase = _reader.BaseStream.Position;
            var dataBlockOffset = _reader.ReadUInt32();
            if (containerVersion >= SpyDataBinary.ContainerVersion_3_2_0_0)
            {
                result.TotalDataBlockLength = _reader.ReadInt64();
                Ensure.True(result.TotalDataBlockLength >= 0, () => new InvalidDataException(nameof(result.TotalDataBlockLength) + " is negative"));
            }
            else
            {
                result.TotalDataBlockLength = _reader.ReadUInt32();
            }

            var nameBytes = _reader.ReadBytes(nameLength);
            Ensure.True(nameBytes.Length == nameLength, () => new EndOfStreamException());
            var nameLengthPadded = (nameLength + 3) & ~3;
            _reader.BaseStream.Position += nameLengthPadded - nameLength;
            result.DataName = Encoding.ASCII.GetString(nameBytes);

            // 最初のブロック位置にシークする。
            _reader.BaseStream.Position = dataBlockOffsetBase + dataBlockOffset;

            _header = result;
            return result;
        }

        public SpyDataBinaryBlock ReadNextBlock()
        {
            Ensure.Operation.True(this.IsOpened);

            if (_header == null)
            {
                ReadHeader();
            }

            if (_reader.BaseStream.Position == _reader.BaseStream.Length)
            {
                return null;
            }

            var timestamp = _reader.ReadInt64();

            long dataBlockID = timestamp; // 古いフォーマットのときは timestamp で代用します。
            if (_header.ContainerVersion >= SpyDataBinary.ContainerVersion_3_1_0_0)
            {
                dataBlockID = _reader.ReadInt64();
            }

            var dataLength = _reader.ReadUInt32();
            Assertion.True(dataLength <= int.MaxValue, () => new InvalidDataException());

            var data = _reader.ReadBytes((int)dataLength);
            Ensure.True(dataLength == data.Length, () => new EndOfStreamException());

            return new SpyDataBinaryBlock()
            {
                ID = dataBlockID,
                Timestamp = timestamp,
                RawData = data,
            };
        }

        protected override void Dispose(bool disposing)
        {
            if (_reader != null)
            {
                _reader.Dispose();
                _reader = null;
            }

            base.Dispose(disposing);
        }
    }
}
