﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.Spy.Resources;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Input;
using System.Windows.Markup;

namespace NintendoWare.Spy.Windows
{
    [ContentProperty("Nodes")]
    public class TimelinePanelViewModel : ObservableObject
    {
        internal static class Names
        {
            internal static class Item
            {
                public const string AudioPerformance = "AudioPerformance";
                public const string AudioVoiceCount = "AudioVoiceCount";
            }

            internal static class Chart
            {
                public const string Sink = "Sink";
                public const string FinalMix = "FinalMix";
                public const string SubMix = "SubMix";
                public const string Voice = "Voice";
                public const string IsOverTime = "IsOverTime";
                public const string VoiceCount = "VoiceCount";
                public const string DropCount = "DropCount";
            }
        }

        private const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。
        private static readonly string[] AudioPerformanceChartOrder =
        {
            Names.Chart.Sink,
            Names.Chart.FinalMix,
            Names.Chart.SubMix,
            Names.Chart.Voice,
            Names.Chart.IsOverTime,
        };
        private static readonly Dictionary<string, string> ToolTips = new Dictionary<string, string>()
        {
            [ToChartId(Names.Item.AudioPerformance, Names.Chart.Sink)] = Messages.ToolTip_AudioPerformance_Sink,
            [ToChartId(Names.Item.AudioPerformance, Names.Chart.FinalMix)] = Messages.ToolTip_AudioPerformance_FinalMix,
            [ToChartId(Names.Item.AudioPerformance, Names.Chart.SubMix)] = Messages.ToolTip_AudioPerformance_SubMix,
            [ToChartId(Names.Item.AudioPerformance, Names.Chart.Voice)] = Messages.ToolTip_AudioPerformance_Voice,
            [ToChartId(Names.Item.AudioPerformance, Names.Chart.IsOverTime)] = Messages.ToolTip_AudioPerformance_IsOverTime,
            [ToChartId(Names.Item.AudioVoiceCount, Names.Chart.VoiceCount)] = Messages.ToolTip_AudioVoiceCount_VoiceCount,
            [ToChartId(Names.Item.AudioVoiceCount, Names.Chart.DropCount)] = Messages.ToolTip_AudioVoiceCount_DropCount,
        };

        private readonly object _observerOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;
        private FrameSyncSpyModel _frameSyncModel;

        private readonly ObservableKeyedList<string, TimelineItemViewModel> _plotNodes = new ObservableKeyedList<string, TimelineItemViewModel>(node => node.Name);

        private IList<TimelineItemViewModel> _filteredNodes;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private double _currentFrame;
        private double _minimumFrame = double.NaN;
        private double _maximumFrame = double.NaN;
        private double _scaleX = 0.001;
        private double _viewportLeft;
        private string _error = null;

        public TimelinePanelViewModel()
        {
            _filteredNodes = _plotNodes;
        }

        public TimelinePanelViewModel(SpyService spyService, SpyPlaybackService playbackService)
            : this()
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(playbackService);

            _spyService = spyService;
            _playbackService = playbackService;

            this.DismissErrorCommand = new DelegateCommand(
                param => this.Error = null);
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            _plotNodes.ForEach(n => n.Dispose());

            base.DisposeManagedInstance();
        }

        public ICommand DismissErrorCommand
        {
            get;
            private set;
        }

        public ObservableCollection<TimelineItemViewModel> Nodes
        {
            get { return _plotNodes; }
        }

        public IList<TimelineItemViewModel> FilteredNodes
        {
            get { return _filteredNodes; }
            private set { this.SetPropertyValue(ref _filteredNodes, value); }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set { this.SetCurrentFrameImpl(value, from: this); }
        }

        public double MinimumFrame
        {
            get { return _minimumFrame; }
            set { this.SetPropertyValue(ref _minimumFrame, value); }
        }

        public double MaximumFrame
        {
            get { return _maximumFrame; }
            set { this.SetPropertyValue(ref _maximumFrame, value); }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set
            {
                value = Math.Min(value, MaxScaleX);
                this.SetPropertyValue(ref _scaleX, value);
            }
        }

        public double ViewportLeft
        {
            get { return _viewportLeft; }
            set { this.SetPropertyValue(ref _viewportLeft, value); }
        }

        public string Error
        {
            get { return _error; }
            set { this.SetPropertyValue(ref _error, value); }
        }

        public FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel; }
            private set { this.SetPropertyValue(ref _frameSyncModel, value); }
        }

        public void SetCurrentFrameFromPresenter(double value)
        {
            this.SetCurrentFrameImpl(value, from: null);
        }

        private void SetCurrentFrameImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.MinimumFrame, this.MaximumFrame);
            }

            if (this.SetPropertyValue(ref _currentFrame, value, nameof(CurrentFrame)))
            {
                if (from == this)
                {
                    if (!double.IsNaN(value))
                    {
                        _playbackService?.SetCurrentTime(SpyGlobalTime.FromMicroSeconds((long)value));
                    }
                }
            }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            _plotModel = model;
            this.SyncPlotItems();

            this.ResetErrorIfNecessary();
        }

        public void SetFrameSyncModel(FrameSyncSpyModel model)
        {
            this.FrameSyncModel = model;
        }

        private void SyncPlotItems()
        {
            _plotNodes.ForEach(n => n.Dispose());
            _plotNodes.Clear();

            if (_plotModel == null)
            {
                return;
            }

            var audioPerformanceItem = this.CreateTimelineItemViewModel(Resources.Labels.ItemLabelAudioPerformance);

            // HACK : 本来は、ターゲットアプリから受け取ったマーカーを表示すべきだが、
            //        今は暫定で nn::audio パフォーマンス用のマーカーを決め打ち設定している
            audioPerformanceItem.HorizontalMarkers = new List<ILineChartHorizontalMarkerData>();
            audioPerformanceItem.HorizontalMarkers.Add(
                new TimelineHorizontalMarkerViewModel()
                {
                    Name = "2500 usec",
                    Value = 2500,
                    MinimumY = 0,
                    MaximumY = 6000,
                });
            audioPerformanceItem.HorizontalMarkers.Add(
                new TimelineHorizontalMarkerViewModel()
                {
                    Name = "5000 usec",
                    Value = 5000,
                    MinimumY = 0,
                    MaximumY = 6000,
                });

            _plotNodes.Add(audioPerformanceItem);

            // HACK : 本来は動的に追加すべきだが、とりあえず決め打ちでボイス数ノードを追加
            var audioVoiceCountNode = this.CreateTimelineItemViewModel(Resources.Labels.ItemLabelAudioVoiceCount);
            _plotNodes.Add(audioVoiceCountNode);

            if (_plotModel == null)
            {
                return;
            }

            CollectionChangedObservation.GetObserver(_observerOwner, _plotModel.Floats).AddHandlerForAddItems(
                (sender, e) =>
                {
                    e.Items.OfType<PlotSpyModel
                        .PlotFloat>()
                        .ForEach(i => this.CreatePerformanceData(i));
                });

            _plotModel.Floats
                .OfType<PlotSpyModel.PlotFloat>()
                .ForEach(i => this.CreatePerformanceData(i));

            PropertyChangedObservation.GetObserver(_observerOwner, _plotModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckPlotModelErrorUnexpectedDataVersion();
                });

            this.CheckPlotModelError();
        }

        private void CreatePerformanceData(PlotSpyModel.PlotFloat item)
        {
            void OnCreate(string itemName, TimelineChartItemViewModel chart)
            {
                if (ToolTips.TryGetValue(ToChartId(itemName, chart.Name), out var value))
                {
                    chart.ToolTip = value;
                }
            }

            if (item.Parent != null)
            {
                var name = item.Parent.FullName;
                if (name == "@nn/audio/PerformanceMetrics/VoiceSummary" ||
                    name == "@nn/audio/PerformanceMetrics/SubMixSummary" ||
                    name == "@nn/audio/PerformanceMetrics/SinkSummary" ||
                    name == "@nn/audio/PerformanceMetrics/FinalMix")
                {
                    if (item.Name == "StartTime" || item.Name == "ProcessingTime")
                    {
                        name = name.Replace("@nn/audio/PerformanceMetrics/", string.Empty);
                        name = name.Replace("Summary", string.Empty);

                        var audioPerformanceNode = _plotNodes[0];
                        audioPerformanceNode.CreateChartItemViewModel(
                            item,
                            name,
                            chartOrder: AudioPerformanceChartOrder,
                            onCreate: chart => OnCreate(Names.Item.AudioPerformance, chart));
                    }
                }
                else if (name == "@nn/audio/PerformanceMetrics")
                {
                    switch (item.Name)
                    {
                        case "VoiceCount":
                            {
                                // HACK : 決め打ちでノード取得
                                var audioVoiceCountNode = _plotNodes[1];
                                audioVoiceCountNode.CreateChartItemViewModelWithoutSummary(
                                    item,
                                    Names.Chart.VoiceCount,
                                    onCreate: chart => OnCreate(Names.Item.AudioVoiceCount, chart));
                            }
                            break;

                        case "VoiceDropCount":
                            {
                                // HACK : 決め打ちでノード取得
                                var audioVoiceCountNode = _plotNodes[1];
                                audioVoiceCountNode.CreateChartItemViewModelWithoutSummary(
                                    item,
                                    Names.Chart.DropCount,
                                    onCreate: chart => OnCreate(Names.Item.AudioVoiceCount, chart));
                            }
                            break;

                        case "IsRenderingTimeLimitExceeded":
                            {
                                void OnCreateIsOverTime(TimelineChartItemViewModel chart)
                                {
                                    OnCreate(Names.Item.AudioPerformance, chart);

                                    // 値が 0 の時の丸印が他のチャートと重なると見づらいので、見えないようにします。
                                    chart.MinimumY = 0.5;
                                }

                                // HACK : 決め打ちでノード取得
                                var audioPerformanceNode = _plotNodes[0];
                                audioPerformanceNode.CreateChartItemViewModelWithoutSummary(
                                    item,
                                    Names.Chart.IsOverTime,
                                    chartOrder: AudioPerformanceChartOrder,
                                    onCreate: OnCreateIsOverTime);
                            }
                            break;
                    }
                }
            }
        }

        private TimelineItemViewModel CreateTimelineItemViewModel(string name)
        {
            return new TimelineItemViewModel(_spyService, _playbackService, name);
        }

        private void AddErrorMessage(string errorMessage)
        {
            if (this.Error == null)
            {
                this.Error = errorMessage;
            }
            else
            {
                this.Error = errorMessage + "\n" + this.Error;
            }
        }

        private void CheckPlotModelErrorUnexpectedDataVersion()
        {
            if (_plotModel != null && _plotModel.ErrorUnexpectedDataVersion)
            {
                this.AddErrorMessage(Resources.Messages.UnexpectedPlotDataVersion);
            }
        }

        private void CheckPlotModelError()
        {
            this.CheckPlotModelErrorUnexpectedDataVersion();
        }

        private void ResetErrorIfNecessary()
        {
            if (_plotModel == null)
            {
                this.Error = null;
            }
        }

        private static string ToChartId(string itemName, string chartName)
        {
            return $"{itemName}.{chartName}";
        }
    }
}
