﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using System;
using System.Collections.Generic;
using System.Linq;
using ObserverOwnerKey = System.Tuple<NintendoWare.Spy.Windows.TimelineChartItemViewModel, string>;

namespace NintendoWare.Spy.Windows
{
    public class TimelineItemViewModel : ObservableObject
    {
        private readonly ObservableKeyedList<string, TimelineChartItemViewModel> _performanceDatas = new ObservableKeyedList<string, TimelineChartItemViewModel>(item => item.Name);
        private readonly Dictionary<ObserverOwnerKey, object> _observerOwners = new Dictionary<ObserverOwnerKey, object>();

        private double _scaleY = 0.9;
        private double _height;

        public TimelineItemViewModel(SpyService spyService, SpyPlaybackService spyPlaybackService, string name)
        {
            this.SpyService = spyService;
            this.SpyPlaybackService = spyPlaybackService;
            this.Name = name;
            this.Height = 120.0;
        }

        public IList<TimelineChartItemViewModel> Floats
        {
            get
            {
                return _performanceDatas;
            }
        }

        public string Name
        {
            get;
            private set;
        }

        public double Height
        {
            get { return _height; }
            set { this.SetPropertyValue(ref _height, value); }
        }

        public SpyService SpyService
        {
            get;
            private set;
        }

        public SpyPlaybackService SpyPlaybackService
        {
            get;
            private set;
        }

        public double ScaleY
        {
            get { return _scaleY; }
            set { this.SetPropertyValue(ref _scaleY, value); }
        }

        public IList<ILineChartHorizontalMarkerData> HorizontalMarkers { get; set; }

        public void CreateChartItemViewModelWithoutSummary(
            PlotSpyModel.PlotFloat item,
            string dataName,
            string[] chartOrder = null,
            Action<TimelineChartItemViewModel> onCreate = null)
        {
            TimelineChartItemViewModel itemData = null;

            if (_performanceDatas.ContainsKey(dataName) == true)
            {
                itemData = _performanceDatas.GetItem(dataName);
                CollectionChangedObservation.RemoveObservers(this.GetObserverOwner(itemData, item.Name));
            }
            else
            {
                itemData = new TimelineChartItemViewModel(
                    this.SpyService,
                    this.SpyPlaybackService,
                    item,
                    dataName,
                    item.Color,
                    LineChartDrawType.Line,
                    TimelineChartValue.ValueMode.Scalar);

                onCreate?.Invoke(itemData);

                if (chartOrder == null)
                {
                    _performanceDatas.Add(itemData);
                }
                else
                {
                    int getIndex(TimelineChartItemViewModel p)
                    {
                        return Array.IndexOf(chartOrder, p.Name);
                    }

                    var insertIndex = _performanceDatas
                        .TakeWhile(p => getIndex(p) < getIndex(itemData))
                        .Count();
                    _performanceDatas.Insert(insertIndex, itemData);
                }
            }

            CollectionChangedObservation.GetObserver(this.GetObserverOwner(itemData, item.Name), item.Values).AddHandlerForAddItems(
                (sender, e) => AddItems(itemData, e.Items.OfType<PlotSpyModel.PlotFloatValue>(), this.SetValue));

            AddItems(itemData, item.Values, this.SetValue);
        }

        public void CreateChartItemViewModel(
            PlotSpyModel.PlotFloat item,
            string dataName,
            string[] chartOrder = null,
            Action<TimelineChartItemViewModel> onCreate = null)
        {
            if (item.Name == "StartTime" || item.Name == "ProcessingTime")
            {
                TimelineChartItemViewModel itemData = null;

                if (_performanceDatas.ContainsKey(dataName) == true)
                {
                    itemData = _performanceDatas.GetItem(dataName);
                    CollectionChangedObservation.RemoveObservers(this.GetObserverOwner(itemData, item.Name));
                }
                else
                {
                    itemData = new TimelineChartItemViewModel(
                        this.SpyService,
                        this.SpyPlaybackService,
                        item,
                        dataName,
                        item.Color,
                        LineChartDrawType.Line,
                        TimelineChartValue.ValueMode.MinMax);

                    onCreate?.Invoke(itemData);

                    if (chartOrder == null)
                    {
                        _performanceDatas.Add(itemData);
                    }
                    else
                    {
                        int getIndex(TimelineChartItemViewModel p)
                        {
                            return Array.IndexOf(chartOrder, p.Name);
                        }

                        var insertIndex = _performanceDatas
                            .TakeWhile(p => getIndex(p) < getIndex(itemData))
                            .Count();
                        _performanceDatas.Insert(insertIndex, itemData);
                    }
                }

                switch (item.Name)
                {
                    case "StartTime":
                        CollectionChangedObservation.GetObserver(this.GetObserverOwner(itemData, item.Name), item.Values).AddHandlerForAddItems(
                            (sender, e) => AddItems(itemData, e.Items.OfType<PlotSpyModel.PlotFloatValue>(), this.SetStartTime));

                        AddItems(itemData, item.Values, this.SetStartTime);
                        break;

                    case "ProcessingTime":
                        CollectionChangedObservation.GetObserver(this.GetObserverOwner(itemData, item.Name), item.Values).AddHandlerForAddItems(
                            (sender, e) => AddItems(itemData, e.Items.OfType<PlotSpyModel.PlotFloatValue>(), this.SetProcessingTime));

                        AddItems(itemData, item.Values, this.SetProcessingTime);
                        break;
                }
            }
        }

        private object GetObserverOwner(TimelineChartItemViewModel viewModel, string itemName)
        {
            object owner = null;
            var key = new ObserverOwnerKey(viewModel, itemName);
            if (!_observerOwners.TryGetValue(key, out owner))
            {
                _observerOwners[key] = owner = new object();
            }
            return owner;
        }

        protected override void DisposeManagedInstance()
        {
            base.DisposeManagedInstance();

            _observerOwners.Values.ForEach(it => CollectionChangedObservation.RemoveObservers(it));
            _observerOwners.Clear();

            foreach (var itemData in _performanceDatas)
            {
                itemData.Dispose();
            }
            _performanceDatas.Clear();
        }

        private void AddItems(
            TimelineChartItemViewModel itemData,
            IEnumerable<PlotSpyModel.PlotFloatValue> items,
            Action<TimelineChartValue, PlotSpyModel.PlotFloatValue> setValueFunc)
        {
            foreach (var item in items)
            {
                if (itemData.Values.IsEmpty() || itemData.Values.Last().Time.Timestamp < item.Timestamp)
                {
                    // 新しいデータを受信したら Values に登録します。

                    var value = new TimelineChartValue(item);
                    itemData.Values.Add(value);
                    setValueFunc(value, item);
                }
                else if (itemData.Values.Last().Time.Timestamp == item.Timestamp)
                {
                    // 同じ時刻の複数のデータを集約します。
                    // 副作用として、同じ時刻のデータが複数あるときは最新のデータが採用されます。
                    setValueFunc(itemData.Values.Last(), item);
                }
                else
                {
                    // 過去にさかのぼってデータを集約します。
                    var index = BinarySearchUtility.BinarySearch(itemData.Values, item.Timestamp, it => it.Time.Timestamp);
                    if (index >= 0)
                    {
                        var value = itemData.Values[index];
                        setValueFunc(value, item);
                    }
                }
            }
        }

        private void SetStartTime(TimelineChartValue value, PlotSpyModel.PlotFloatValue item)
        {
            value.MinValue = item.Value;
            this.UpdateMaxValue(value);
        }

        private void SetProcessingTime(TimelineChartValue value, PlotSpyModel.PlotFloatValue item)
        {
            value.Value = item.Value;
            this.UpdateMaxValue(value);
        }

        private void UpdateMaxValue(TimelineChartValue value)
        {
            if (!double.IsNaN(value.MinValue) && !double.IsNaN(value.Value))
            {
                value.MaxValue = value.MinValue + value.Value; // StartTime + ProcessingTime
            }
        }

        private void SetValue(TimelineChartValue value, PlotSpyModel.PlotFloatValue item)
        {
            value.Value = item.Value;
        }
    }
}
