﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows
{
    public class SpyDebugPanelViewModel : ObservableObject
    {
        private const string FullNameDataBufferLength = "@nn/spy/DataBufferLength";
        private const string FullNameDataBufferUsage = "@nn/spy/DataBufferUsage";

        private readonly object _observerOwner = new object();

        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;

        private double _dataBufferLength = 0d;
        private double _dataBufferUsage = 0d;

        private readonly MergedRequestDispatcher _requestUpdateCurrentTime = new MergedRequestDispatcher();

        private readonly PlotSpyModelWatcher _plotModelWatcher = new PlotSpyModelWatcher()
        {
            Floats =
            {
                FullNameDataBufferLength,
                FullNameDataBufferUsage,
            },
        };

        public SpyDebugPanelViewModel(SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(playbackService);

            _playbackService = playbackService;

            PropertyChangedObservation.GetObserver(_observerOwner, _playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateCurrentTime());

            _plotModelWatcher.Changed += (sender, args) => this.RequestUpdateCurrentTime();
            _plotModelWatcher.Start();
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            _plotModelWatcher.Stop();

            base.DisposeManagedInstance();
        }

        public double DataBufferLength
        {
            get { return _dataBufferLength; }
            private set
            {
                if (this.SetPropertyValue(ref _dataBufferLength, value))
                {
                    this.NotifyPropertyChanged(nameof(DataBufferUseRate));
                }
            }
        }

        public double DataBufferUsage
        {
            get { return _dataBufferUsage; }
            private set
            {
                if (this.SetPropertyValue(ref _dataBufferUsage, value))
                {
                    this.NotifyPropertyChanged(nameof(DataBufferUseRate));
                }
            }
        }

        public double DataBufferUseRate
        {
            get
            {
                return this.DataBufferLength > 0
                    ? this.DataBufferUsage / this.DataBufferLength
                    : 0;
            }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            _plotModel = model;
            _plotModelWatcher.PlotSpyModel = model;

            this.UpdateCurrentTime();
        }

        private void RequestUpdateCurrentTime()
        {
            _requestUpdateCurrentTime.Request(UpdateCurrentTime, DispatcherPriority.Background);
        }

        private void UpdateCurrentTime()
        {
            var current = _playbackService.Current.Timestamp;

            this.DataBufferLength = _plotModel?.GetFloat(FullNameDataBufferLength)?.Values.FirstOrDefault()?.Value ?? 0d;

            this.DataBufferUsage = _plotModel?.GetFloat(FullNameDataBufferUsage)?.FindValue(current)?.Value ?? 0d;
        }
    }
}
