﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using System;
using System.Collections.Generic;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// プロットアイテムのビューモデルです。
    /// </summary>
    public class PlotFloatData : ObservableObject, ILineChartItemData
    {
        private readonly object _observerOwner = new object();
        private readonly MergedRequestDispatcher _requestNotifySamplesChanged = new MergedRequestDispatcher();
        private PlotSpyModel.PlotFloat _model;
        private bool _isVisible = true;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="model">プロットアイテムモデルを指定します。</param>
        public PlotFloatData(PlotSpyModel.PlotFloat model)
        {
            Assertion.Argument.NotNull(model);

            _model = model;

            this.Name = model.Name;
            this.Color = model.Color;
            this.MinimumY = model.Minimum;
            this.MaximumY = model.Maximum;
            this.Interpolation = ConvertToLineChartInterpolation(model.InterpolationMode);

            // ファイルからの読み出し時にイベントが大量に発生するのを防ぎます。
            CollectionChangedObservation.GetObserver(_observerOwner, model.Values).AddHandler(
                (sender, e) => this.RequestNotifySamplesChanged());
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            base.DisposeManagedInstance();
        }

        public event EventHandler SamplesChanged;

        public string Name { get; }

        public bool IsVisible
        {
            get { return _isVisible; }
            set { this.SetPropertyValue(ref _isVisible, value); }
        }

        public LineChartDrawType DrawType
        {
            get { return LineChartDrawType.Line; }
        }

        public Color Color { get; }

        public LineChartInterpolation Interpolation { get; }

        public double MinimumX { get; set; }

        public double MaximumX { get; set; }

        public double MinimumY { get; }

        public double MaximumY { get; }

        public double CenterY
        {
            get { return this.MinimumY + ((this.MaximumY - this.MinimumY) / 2.0); }
        }

        public IEnumerable<LineChartSample> EnumerateSamples(double beginX, double endX, double scale)
        {
            var values = _model.Values;

            if (values.Count == 0)
            {
                yield break;
            }

            var currentSampleIndex = this.FindSampleSmallestIndex(beginX);

            if (currentSampleIndex >= values.Count)
            {
                yield break;
            }

            if (currentSampleIndex < 0)
            {
                // beginX がサンプル前方に範囲外の場合、先頭サンプルから処理します
                currentSampleIndex = 0;
            }

            if (scale <= 1)
            {
                // ズームインの場合は範囲内のすべてのアイテムを yield します。

                for (; currentSampleIndex < values.Count; currentSampleIndex++)
                {
                    var x = (double)values[currentSampleIndex].Time.GetMicroSeconds();
                    var y = values[currentSampleIndex].Value;

                    yield return new LineChartSample(new Range1(x, x), new Range1(y, y));

                    if (x > endX)
                    {
                        break;
                    }
                }
            }
            else
            {
                // ズームアウトの場合は 画面の X 軸方向のピクセルが同じサンプルをまとめて min, max を求めます。

                var currentXBegin = (double)values[currentSampleIndex].Time.GetMicroSeconds();
                var currentXEnd = scale > 1.0
                    ? currentXBegin / scale * scale + scale
                    : currentXBegin + 1;

                Range1 rangeX = new Range1(currentXBegin, currentXBegin);
                Range1 rangeY = new Range1(
                    values[currentSampleIndex].Value,
                    values[currentSampleIndex].Value);

                while (true)
                {
                    // 指定範囲を超えたら、サンプルを返して、ループ終了
                    // スクリーン領域外（右側）のサンプルも描画対象にする
                    if (endX < currentXBegin)
                    {
                        yield return new LineChartSample(rangeX, rangeY);
                        break;
                    }

                    currentSampleIndex++;

                    // すべての plotValue を確認したら、サンプルを返してループ終了
                    if (values.Count <= currentSampleIndex)
                    {
                        yield return new LineChartSample(rangeX, rangeY);
                        break;
                    }

                    var nextXBegin = (double)values[currentSampleIndex].Time.GetMicroSeconds();

                    // 次のサンプルが、同一ピクセルに含めるべきなら、Range に含める
                    // 次のピクセルに含めるべきサンプルなら、今のサンプルを yield return して、次のサンプル取得を待つ
                    if (nextXBegin < currentXEnd)
                    {
                        rangeX.Max = currentXBegin;
                        rangeY.Min = Math.Min(rangeY.Min, values[currentSampleIndex].Value);
                        rangeY.Max = Math.Max(rangeY.Max, values[currentSampleIndex].Value);
                    }
                    else
                    {
                        yield return new LineChartSample(rangeX, rangeY);

                        currentXBegin = nextXBegin;
                        currentXEnd = scale > 1.0
                            ? currentXBegin / scale * scale + scale
                            : currentXBegin + 1;

                        rangeX = new Range1(currentXBegin, currentXBegin);
                        rangeY = new Range1(
                            values[currentSampleIndex].Value,
                            values[currentSampleIndex].Value);
                    }
                }
            }
        }

        public double Evaluate(double x)
        {
            return _model?.FindValue(SpyGlobalTime.FromMicroSeconds((long)x))?.Value ?? double.NaN;
        }

        /// <summary>
        /// 指定フレームに近いサンプルインデックスを検索します。
        /// 指定フレームに複数のサンプルがあるときは最初のサンプルを検索します。
        /// </summary>
        /// <param name="x">現在の時間単位におけるフレームを指定します。</param>
        /// <returns>
        /// <list type="bullet">
        /// <item>指定フレームに最も近いサンプルのインデックスを返します。</item>
        /// <item>指定フレーム &lt; サンプル先頭の場合、-1 を返します。</item>
        /// <item>指定フレーム &gt; サンプル末尾の場合、サンプル数（リストの要素数）を返します。</item>
        /// </list>
        /// </returns>
        private int FindSampleSmallestIndex(double x)
        {
            if (_model.Values.Count == 0)
            {
                return 0;
            }

            // 二分探索して一致した場合
            // ・・・そのインデックスを返す
            // 一致しなかった場合
            // ・・・範囲内なら、BinarySearch() が返した補数を解決して、指定フレーム以下の最も近いインデックスを返す
            // ・・・範囲外なら、そのままサンプル数（リストの要素数）を返す
            var result = BinarySearchUtility.BinarySearch(_model.Values, x, value => (double)value.Time.GetMicroSeconds(), BinarySearchUtility.Options.SmallestIndex);

            if (result >= 0)
            {
                return result;
            }

            result = ~result;

            return result < _model.Values.Count ? result - 1 : result;
        }

        private void RequestNotifySamplesChanged()
        {
            if (this.SamplesChanged != null)
            {
                _requestNotifySamplesChanged.Request(this.NotifySamplesChanged);
            }
        }

        private void NotifySamplesChanged()
        {
            this.SamplesChanged?.Invoke(this, EventArgs.Empty);
        }

        private static LineChartInterpolation ConvertToLineChartInterpolation(PlotSpyModel.PlotFloatInterpolationMode interpolation)
        {
            switch (interpolation)
            {
                case PlotSpyModel.PlotFloatInterpolationMode.Linear:
                    return LineChartInterpolation.Linear;

                default:
                    return LineChartInterpolation.None;
            }
        }
    }
}
