﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NAudio.CoreAudioApi;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Framework.Windows;
using NintendoWare.Spy.Settings;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows;

namespace NintendoWare.Spy.Windows
{
    public class OptionWindowPresenter : SimpleWindowPresenter
    {
        private OptionWindowViewModel _viewModel;
        private OptionWindow _window;

        //-----------------------------------------------------------------

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public override IEnumerable<Type> DependentServiceTypes
        {
            get
            {
                yield return typeof(SpyService);
                yield return typeof(SpyPlaybackService);
                yield return typeof(SettingsService);
            }
        }

        // HACK : ★とりあえず。たぶん Presenter に親子ノード参照を追加して、そこから取得するべき。
        public Window OwnerWindow { get; set; }

        public Action ResetPanelLayout { get; set; }

        //-----------------------------------------------------------------

        protected override void OnInitialize()
        {
            base.OnInitialize();

            var enumerator = new MMDeviceEnumerator();
            var endPoints = enumerator.EnumerateAudioEndPoints(DataFlow.Render, DeviceState.Active | DeviceState.Unplugged);

            _viewModel = new OptionWindowViewModel()
            {
                PlaybackDeviceName = this.GetApplicationSettings().PlaybackDeviceName,
                UpdateInterval = (int)this.GetApplicationSettings().UpdateInterval.TotalMilliseconds,
                TimeUnit = this.GetApplicationSettings().TimeUnit,
                WorkDirectoryPath = this.GetSpyService().WorkDirectoryPath,
            };

            if (string.IsNullOrEmpty(_viewModel.PlaybackDeviceName))
            {
                var endPoint = endPoints.FirstOrDefault();

                if (endPoint != null)
                {
                    _viewModel.PlaybackDeviceName = endPoint.FriendlyName;
                }
            }

            _viewModel.Setup(endPoints.Select(endPoint => endPoint.FriendlyName));

            this.SetupCommand();
        }

        protected override void OnUninitialize()
        {
            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        /// <summary>
        /// ウィンドウを生成します。
        /// </summary>
        /// <returns>生成したウィンドウを返します。</returns>
        protected override Window CreateWindow()
        {
            _window = new OptionWindow()
            {
                DataContext = _viewModel,
                Owner = this.OwnerWindow,
                WindowStartupLocation = WindowStartupLocation.CenterOwner,
            };

            return _window;
        }

        [DebuggerStepThrough]
        private SettingsService GetSettingsService()
        {
            return this.ServiceProvider.GetService<SettingsService>();
        }

        [DebuggerStepThrough]
        private SpyPlaybackService GetPlaybackService()
        {
            return this.ServiceProvider.GetService<SpyPlaybackService>();
        }

        [DebuggerStepThrough]
        private ApplicationSettings GetApplicationSettings()
        {
            return this.GetSettingsService().GetApplicationSettings();
        }

        [DebuggerStepThrough]
        private ConnectionSettings GetConnectionSettings()
        {
            return this.GetSettingsService().GetConnectionSettings();
        }

        [DebuggerStepThrough]
        private SpyService GetSpyService()
        {
            return this.ServiceProvider.GetService<SpyService>();
        }

        private void SetupCommand()
        {
            _viewModel.OKCommand = new DelegateCommand(
                parameter => this.Apply(),
                parameter => _viewModel == null ? false : !_viewModel.HasError);
            _viewModel.CancelCommand = new DelegateCommand(parameter => this.Cancel());
            _viewModel.OpenWorkDirectoryPathCommand = new DelegateCommand(parameter => { try { System.Diagnostics.Process.Start(_viewModel.WorkDirectoryPath); } catch { } });
            _viewModel.ResetPanelLayoutCommand = new DelegateCommand(p =>
                {
                    Assertion.Argument.NotNull(this.ResetPanelLayout);

                    if (MessageBox.Show(
                            _window,
                            Resources.Messages.ConfirmResetPanelLayout,
                            Resources.Labels.ResetPanelLayoutMessageBoxTitle,
                            MessageBoxButton.YesNo,
                            MessageBoxImage.Warning) == MessageBoxResult.Yes)
                    {
                        this.ResetPanelLayout();
                    }
                });
        }

        private void Apply()
        {
            _window.Close();

            ApplicationSettings applicationSettings = this.GetApplicationSettings();
            if (applicationSettings.PlaybackDeviceName != _viewModel.PlaybackDeviceName)
            {
                applicationSettings.PlaybackDeviceName = _viewModel.PlaybackDeviceName;
                this.GetPlaybackService().ResetDevice();
            }

            applicationSettings.UpdateInterval = TimeSpan.FromMilliseconds(_viewModel.UpdateInterval);
            applicationSettings.TimeUnit = _viewModel.TimeUnit;
        }

        private void Cancel()
        {
            _window.Close();
        }
    }
}
