﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System.Collections.Generic;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    public class MeterPerformanceData : ObservableObject
    {
        private readonly SpyService _spyService;
        private readonly List<PlotSpyModel.PlotFloatValue> _values = new List<PlotSpyModel.PlotFloatValue>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="spyService">SpyService を指定します。</param>
        /// <param name="name">名前を指定します。</param>
        /// <param name="color">カラーを指定します。</param>
        public MeterPerformanceData(SpyService spyService, string name, Color color)
        {
            Assertion.Argument.NotNull(spyService);

            _spyService = spyService;
            this.Name = name;
            this.Color = color;
        }

        public string Name
        {
            get;
            private set;
        }

        public IList<PlotSpyModel.PlotFloatValue> Values
        {
            get { return _values; }
        }

        public Color Color
        {
            get;
            protected set;
        }

        public PlotSpyModel.PlotFloatValue GetPerformanceValueData(double x)
        {
            var valueData = this.GetValueData(x);

            if (valueData == null && _spyService.State == SpyService.ServiceState.Running)
            {
                if (_values.Count > 0 && _values[_values.Count - 1].Timestamp.MicroSeconds < x)
                {
                    valueData = _values[_values.Count - 1];
                }
            }

            return valueData;
        }

        private PlotSpyModel.PlotFloatValue GetValueData(double x)
        {
            if (_values.Count == 0)
            {
                return null;
            }

            if ((double)_values[_values.Count - 1].Timestamp.MicroSeconds < x)
            {
                return null;
            }

            var currentSampleIndex = this.FindSampleBiggestIndex(x);

            if (currentSampleIndex < 0)
            {
                return null;
            }

            if (currentSampleIndex >= _values.Count)
            {
                return null;
            }

            return _values[currentSampleIndex];
        }

        /// <summary>
        /// 指定フレームに近いサンプルインデックスを検索します。
        /// 同一フレームに複数のサンプルがあるときは最新のサンプルを検索します。
        /// </summary>
        /// <param name="x">現在の時間単位におけるフレームを指定します。</param>
        /// <returns>
        /// <list type="bullet">
        /// <item>指定フレームに最も近いサンプルのインデックスを返します。</item>
        /// <item>指定フレーム &lt; サンプル先頭の場合、-1 を返します。</item>
        /// <item>指定フレーム &gt; サンプル末尾の場合、サンプル数（リストの要素数）を返します。</item>
        /// </list>
        /// </returns>
        private int FindSampleBiggestIndex(double x)
        {
            if (_values.Count == 0)
            {
                return 0;
            }

            // 二分探索して一致した場合
            // ・・・そのインデックスを返す
            // 一致しなかった場合
            // ・・・範囲内なら、BinarySearch() が返した補数を解決して、指定フレーム以下の最も近いインデックスを返す
            // ・・・範囲外なら、そのままサンプル数（リストの要素数）を返す
            var result = BinarySearchUtility.BinarySearch(_values, x, value => (double)value.Timestamp.MicroSeconds, BinarySearchUtility.Options.BiggestIndex);

            if (result >= 0)
            {
                return result;
            }

            result = ~result;

            return result < _values.Count ? result - 1 : result;
        }
    }
}
