﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    public class MeterControlViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private readonly object _plotObserverOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;

        private readonly ObservableKeyedList<string, MeterPerformanceNodeData> _plotNodes = new ObservableKeyedList<string, MeterPerformanceNodeData>(node => node.Name);

        private string _error = null;

        private string _title;
        private double _actualWidth = double.NaN;
        private double _viewportX = 0.0;
        private double _scaleX = 1;
        private double _minimumScaleX = 1.0;
        private double _maximumX = 5500.0;
        private double _displayMaxPerformanceValue;
        private double _currentPerformanceValue;
        private readonly ObservableCollection<MeterControlItemViewModel> _sources = new ObservableCollection<MeterControlItemViewModel>();

        private readonly Dictionary<string, PerformanceDataPair> _performanceDataPairDictionary = new Dictionary<string, PerformanceDataPair>();

        public MeterControlViewModel()
        {
        }

        public MeterControlViewModel(SpyService spyService, SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(playbackService);

            _spyService = spyService;
            _playbackService = playbackService;

            PropertyChangedObservation.GetObserver(_observerOwner, _playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateSources());
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);
            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);

            _plotNodes.ForEach(it => it.Dispose());
            _sources.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public string Title
        {
            get
            {
                return _title;
            }
            set
            {
                this.SetPropertyValue(ref _title, value);
            }
        }

        public double ActualWidth
        {
            get
            {
                return _actualWidth;
            }
            set
            {
                if (this.SetPropertyValue(ref _actualWidth, value))
                {
                    this.UpdateScaleX();
                    this.UpdateMinimumScaleX();
                }
            }
        }

        public double ViewportX
        {
            get
            {
                return _viewportX;
            }
            set
            {
                if (this.SetPropertyValue(ref _viewportX, value) == true)
                {
                    this.UpdateMinimumScaleX();
                }
            }
        }

        public double ScaleX
        {
            get
            {
                return _scaleX;
            }
            set
            {
                if (this.SetPropertyValue(ref _scaleX, value) == true)
                {
                    this.UpdateDisplayMaxPerformanceValue();
                }
            }
        }

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value) == true)
                {
                    this.UpdateMinimumScaleX();
                }
            }
        }

        public double MinimumScaleX
        {
            get { return _minimumScaleX; }
            set { this.SetPropertyValue(ref _minimumScaleX, value); }
        }

        public double DisplayMaxPerformanceValue
        {
            get
            {
                return _displayMaxPerformanceValue;
            }
            set
            {
                if (this.SetPropertyValue(ref _displayMaxPerformanceValue, value))
                {
                    this.UpdateScaleX();
                }
            }
        }

        public double CurrentPerformanceValue
        {
            get
            {
                return _currentPerformanceValue;
            }
            set
            {
                this.SetPropertyValue(ref _currentPerformanceValue, value);
            }
        }

        public ObservableCollection<MeterControlItemViewModel> Sources
        {
            get
            {
                return _sources;
            }
        }

        private void UpdateScaleX()
        {
            if (!double.IsNaN(this.ActualWidth) && this.DisplayMaxPerformanceValue > 0)
            {
                this.ScaleX = this.ActualWidth / this.DisplayMaxPerformanceValue;
            }
        }

        private void UpdateDisplayMaxPerformanceValue()
        {
            if (double.IsNaN(this.ActualWidth) == false)
            {
                this.DisplayMaxPerformanceValue = this.ActualWidth / this.ScaleX;
            }
        }

        private void UpdateMinimumScaleX()
        {
            if (double.IsNaN(this.ActualWidth) == false)
            {
                var visibleMaximumX = this.ViewportX + this.ActualWidth / this.ScaleX;
                if (visibleMaximumX > this.MaximumX &&
                    this.ViewportX < this.MaximumX)
                {
                    this.MinimumScaleX = this.ActualWidth / (this.MaximumX - this.ViewportX);
                }
            }
        }

        public ObservableCollection<MeterPerformanceNodeData> Nodes
        {
            get { return _plotNodes; }
        }

        public string Error
        {
            get { return _error; }
            set { this.SetPropertyValue(ref _error, value); }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);

            _plotModel = model;
            this.SyncPlotItems();
            this.UpdateSources();

            this.ResetErrorIfNecessary();
        }

        private void SyncPlotItems()
        {
            _performanceDataPairDictionary.Clear();
            _plotNodes.ForEach(n => n.Dispose());
            _plotNodes.Clear();
            _sources.ForEach(it => it.Visibility = Visibility.Collapsed);

            if (_plotModel == null)
            {
                return;
            }

            _plotNodes.Add(this.CreatePerformanceNodeData(Resources.Labels.ItemLabelAudioPerformance));

            CollectionChangedObservation.GetObserver(_plotObserverOwner, _plotModel.Floats).AddHandlerForAddItems(
                (sender, e) =>
                {
                    e.Items.OfType<PlotSpyModel.PlotFloat>().ForEach(i => this.CreatePerformanceData(i));
                });

            _plotModel.Floats
                .OfType<PlotSpyModel.PlotFloat>()
                .ForEach(i => this.CreatePerformanceData(i));

            PropertyChangedObservation.GetObserver(_plotObserverOwner, _plotModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckPlotModelErrorUnexpectedDataVersion();
                });

            this.CheckPlotModelError();
        }

        private void CreatePerformanceData(PlotSpyModel.PlotFloat item)
        {
            if (item.Parent != null)
            {
                var name = item.Parent.FullName;

                if (Regex.IsMatch(name, "^@nn/audio/PerformanceMetrics/Voice/\\d+$") == true ||
                    Regex.IsMatch(name, "^@nn/audio/PerformanceMetrics/SubMix/\\d+$") == true ||
                    Regex.IsMatch(name, "^@nn/audio/PerformanceMetrics/Sink/\\d+$") == true ||
                    name == "@nn/audio/PerformanceMetrics/FinalMix")
                {
                    name = name.Replace("@nn/audio/PerformanceMetrics", string.Empty);

                    var cardinalNodeData = _plotNodes[0];
                    var data = cardinalNodeData.CreatePerformanceData(item, item.Name + name);

                    if (_performanceDataPairDictionary.ContainsKey(name) == false)
                    {
                        _performanceDataPairDictionary.Add(name, new PerformanceDataPair());
                    }

                    if (item.Name == "StartTime")
                    {
                        _performanceDataPairDictionary[name].StartTimeData = data;

                        Color drakColor = new Color()
                        {
                            A = data.Color.A,
                            R = (byte)(data.Color.R * 0.9),
                            G = (byte)(data.Color.G * 0.9),
                            B = (byte)(data.Color.B * 0.9),
                        };

                        Color lightColor = new Color()
                        {
                            A = data.Color.A,
                            R = (byte)Math.Min(data.Color.R * 1.1, byte.MaxValue),
                            G = (byte)Math.Min(data.Color.G * 1.1, byte.MaxValue),
                            B = (byte)Math.Min(data.Color.B * 1.1, byte.MaxValue),
                        };

                        LinearGradientBrush brush = new LinearGradientBrush(drakColor, lightColor, 0);
                        brush.Freeze();
                        _performanceDataPairDictionary[name].Fill = brush;
                    }
                    else if (item.Name == "ProcessingTime")
                    {
                        _performanceDataPairDictionary[name].ProcessingTimeData = data;
                    }
                }
            }
        }

        private MeterPerformanceNodeData CreatePerformanceNodeData(string name)
        {
            return new MeterPerformanceNodeData(_spyService, name);
        }

        private void AddErrorMessage(string errorMessage)
        {
            if (this.Error == null)
            {
                this.Error = errorMessage;
            }
            else
            {
                this.Error = errorMessage + "\n" + this.Error;
            }
        }

        private void CheckPlotModelErrorUnexpectedDataVersion()
        {
            if (_plotModel != null && _plotModel.ErrorUnexpectedDataVersion)
            {
                this.AddErrorMessage(Resources.Messages.UnexpectedPlotDataVersion);
            }
        }

        private void CheckPlotModelError()
        {
            this.CheckPlotModelErrorUnexpectedDataVersion();
        }

        private void ResetErrorIfNecessary()
        {
            if (_plotModel == null)
            {
                this.Error = null;
            }
        }

        private void UpdateSources()
        {
            if (_plotNodes.Count <= 0)
            {
                return;
            }

            double currentFrame = _playbackService.Current.GetMicroSeconds();

            var items = _performanceDataPairDictionary.Values
                .Where(it => it.StartTimeData != null && it.ProcessingTimeData != null)
                .Select(it => new
                {
                    Pair = it,
                    StartTime = it.StartTimeData.GetPerformanceValueData(currentFrame),
                    ProcessingTime = it.ProcessingTimeData.GetPerformanceValueData(currentFrame),
                })
                .Where(it => it.StartTime != null && it.ProcessingTime != null)
                .Where(it => !(it.StartTime.Value == 0 && it.ProcessingTime.Value == 0))
                .OrderBy(it => it.StartTime.Value)
                .ToList();

            this.CurrentPerformanceValue = items.IsEmpty() ? 0 : items.Max(it => it.StartTime.Value + it.ProcessingTime.Value);

            // VM を再利用します。
            items.ForEach((it, i) =>
            {
                if (i < _sources.Count)
                {
                    var vm = _sources[i];
                    vm.Visibility = Visibility.Visible;
                    vm.Label = it.Pair.Label;
                    vm.Fill = it.Pair.Fill;
                    vm.Left = it.StartTime.Value;
                    vm.Right = it.StartTime.Value + it.ProcessingTime.Value;
                    vm.Value = it.ProcessingTime.Value;
                }
                else
                {
                    var vm = new MeterControlItemViewModel()
                    {
                        Visibility = Visibility.Visible,
                        Label = it.Pair.Label,
                        Fill = it.Pair.Fill,
                        Left = it.StartTime.Value,
                        Right = it.StartTime.Value + it.ProcessingTime.Value,
                        Value = it.ProcessingTime.Value,
                    };
                    _sources.Add(vm);
                }
            });

            // 再利用されなかった VM は Visibility を Collapsed にし非表示にします。
            _sources
                .Skip(items.Count)
                .TakeWhile(it => it.Visibility != Visibility.Collapsed)
                .ForEach(it => it.Visibility = Visibility.Collapsed);
        }

        private class PerformanceDataPair
        {
            public Brush Fill { get; set; }
            public MeterPerformanceData StartTimeData { get; set; }
            public MeterPerformanceData ProcessingTimeData { get; set; }

            public string Label
            {
                get
                {
                    if (this.StartTimeData == null)
                    {
                        return string.Empty;
                    }

                    return this.StartTimeData.Name.Replace("StartTime/", string.Empty);
                }
            }
        }
    }
}
