﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.Win32;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.Spy.Commands;
using NintendoWare.Spy.Foundation;
using NintendoWare.Spy.Foundation.Commands;
using NintendoWare.Spy.Foundation.Platform;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Framework.Windows;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Resources;
using NintendoWare.Spy.Settings;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    internal sealed class MainWindowPresenter : SimpleWindowPresenter
    {
        private MainWindowViewModel _viewModel;

        private readonly Dictionary<SpyPanelPlugin, PanelPresenter> _panelPresenters =
            new Dictionary<SpyPanelPlugin, PanelPresenter>();

        private readonly GlobalTimelinePresenter _globalTimelinePresenter = new GlobalTimelinePresenter();
        private readonly Lazy<OptionWindowPresenter> _optionWindowPresenter = new Lazy<OptionWindowPresenter>();

        private readonly List<ConnectionInfoViewModel> _connectionInfoViewModels = new List<ConnectionInfoViewModel>();

        private MainWindow _window;

        private readonly object _observerOwner = new object();

        private readonly Brush _normalStatusBarBrush = new SolidColorBrush(Color.FromArgb(0xff, 0x00, 0x80, 0xc0));
        private readonly Brush _connectedStatusBarBrush = new SolidColorBrush(Color.FromArgb(0xff, 0x50, 0xc0, 0x10));

        //-----------------------------------------------------------------

        public MainWindowPresenter()
        {
            _normalStatusBarBrush.Freeze();
            _connectedStatusBarBrush.Freeze();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public override IEnumerable<Type> DependentServiceTypes
        {
            get
            {
                yield return typeof(ApplicationInfoService);
                yield return typeof(PluginsService);
                yield return typeof(SpyService);
                yield return typeof(SpyPlaybackService);
                yield return typeof(SpyTemporaryDataService);
                yield return typeof(SettingsService);
                yield return typeof(TeamSettingsService);
            }
        }

        private ApplicationInfoService ApplicationInfoService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<ApplicationInfoService>(); }
        }

        private PluginsService PluginsService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<PluginsService>(); }
        }

        private SettingsService SettingsService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<SettingsService>(); }
        }

        private SpyService SpyService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<SpyService>(); }
        }

        private SpyPlaybackService PlaybackService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<SpyPlaybackService>(); }
        }

        private SpyTemporaryDataService SpyTemporaryDataService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<SpyTemporaryDataService>(); }
        }

        private TeamSettingsService TeamSettingsService
        {
            [DebuggerStepThrough]
            get { return this.ServiceProvider.GetService<TeamSettingsService>(); }
        }

        //-----------------------------------------------------------------

        protected override void OnInitialize()
        {
            base.OnInitialize();

            _viewModel = new MainWindowViewModel(this.GetCommandService());

            this.InitializeConnectionInfos();
            this.InitializeMainWindowViewModel();

            this.InitializeGlobalTimelinePresenter();

            this.UpdateSpyDataIDFlags();
        }

        protected override void OnUninitialize()
        {
            this.SpyService.StateChanged -= this.UpdateConnectionIcon;
            this.SettingsService.Applied -= this.UpdateConnectionSettings;

            PropertyChangedObservation.RemoveObservers(_observerOwner);

            // 再生中はリソースが解放されないのでアプリケーションが停止できません。
            this.PlaybackService.Stop();

            this.UnsetupSettings();

            this.UninitializeGlobalTimelinePresenter();

            foreach (var panelPresenter in _panelPresenters.Values)
            {
                panelPresenter.Uninitialize();
            }

            _panelPresenters.Clear();

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        protected override void OnBindCommands(CommandService commandService)
        {
            base.OnBindCommands(commandService);

            commandService.BindCommand(
                SpyCommands.UpdateSpyDataIDFlags,
                (command, args) => this.SpyService != null ?
                    CommandStatus.Enabled : CommandStatus.Unsupported,
                (command, args) =>
                {
                    this.UpdateSpyDataIDFlags();
                    return CommandResult.CreateSucceeded();
                });

            commandService.BindCommand<AudioSaveRecordWaveDataDialogCommandArgs>(
                SpyAudioCommands.SaveRecordWaveDataDialog,
                (command, args) => AudioSaveRecordWaveDataDialogCommandHandler.CanExecute(args),
                (command, args) => AudioSaveRecordWaveDataDialogCommandHandler.Execute(args));
        }

        /// <summary>
        /// ウィンドウを生成します。
        /// </summary>
        /// <returns>生成したウィンドウを返します。</returns>
        protected override Window CreateWindow()
        {
            _window = new MainWindow()
            {
                DataContext = _viewModel,
            };

            var tabItems = this.PluginsService
                .RibbonTabPlugins
                .Select(it => it.CreateTab(this.ServiceProvider))
                .Where(tabItem => tabItem != null)
                .ToList();

            foreach (var tab in tabItems)
            {
                tab.SetBinding(Fluent.RibbonTabItem.IsEnabledProperty, new Binding("IsNotLoading")
                {
                    Source = _viewModel,
                });
            }

            _window.AddRibbonTabItems(tabItems);

            Application.Current.MainWindow = _window;

            this.UpdateConnectionIcon();

            _window.Loaded += (sender, e) =>
            {
                string message = this.TeamSettingsService.GetWarningMessage();
                if (message != null)
                {
                    MessageBox.Show(_window, message, string.Empty, MessageBoxButton.OK, MessageBoxImage.Warning);
                }

                message = (Application.Current as App).GetWarningMessage();
                if (message != null)
                {
                    MessageBox.Show(_window, message, string.Empty, MessageBoxButton.OK, MessageBoxImage.Warning);
                }
            };

            _window.MessageReceived += (sender, e) =>
            {
                this.OnMessageReceived(e);
            };

            return _window;
        }

        private void OnMessageReceived(MainWindow.MessageReceivedEventArgs args)
        {
            Debug.WriteLine($"MainWindowPresenter.OnMessageReceived {args.Message}");

            args.Result = CommandMessages.Result.Success;

            switch (args.Message)
            {
                case CommandMessages.Connect.CommandName:
                    this.ProcessConnectCommand(args);
                    break;

                case CommandMessages.Disconnect.CommandName:
                    this.ProcessDisconnectCommand(args);
                    break;

                default:
                    args.Result = CommandMessages.Result.UnexpectedCommand;
                    break;
            }
        }

        private void ProcessConnectCommand(MainWindow.MessageReceivedEventArgs args)
        {
            if (_viewModel?.SelectedConnectionInfo == null)
            {
                args.Result = CommandMessages.Connect.Result.TargetIsNotSelected;
                return;
            }

            var targetPlatformName = this.GetTargetPlatformName();
            if (this.IsTargetManagerRequired(targetPlatformName) && !this.IsTargetManagerRunning())
            {
                args.Result = CommandMessages.Connect.Result.NotRunningTargetManager;
                return;
            }

            _viewModel.IsConnecting = true;
        }

        private void ProcessDisconnectCommand(MainWindow.MessageReceivedEventArgs args)
        {
            if (_viewModel != null && _viewModel.IsConnecting)
            {
                _viewModel.IsConnecting = false;
            }
        }

        protected override void PreShowWindow()
        {
            this.SetupSettings();
        }

        private void InitializeMainWindowViewModel()
        {
            var appInfo = this.ApplicationInfoService.ApplicationInfo;

            FileVersionInfo vertionInfo = FileVersionInfo.GetVersionInfo(Assembly.GetEntryAssembly().Location);
            _viewModel.Title = string.Format(
                "{0} {1}.{2}.{3}.{4}",
                appInfo.Name,
                appInfo.Version.Major,
                appInfo.Version.Minor,
                appInfo.Version.Build,
                appInfo.Version.Revision);

            // ツールパネルは遅延生成するため、ViewModel から生成タイミングの通知を受けます。
            _viewModel.RegisterPanelPlugins(
                this.PluginsService.PanelPlugins,
                panelPlugin => this.CreatePanelPresenter(panelPlugin).Content);

            // SetupPropertyDependencies() より先に InitializeViewModelTargetPlatform() しないと
            // InitializeViewModelTargetPlatform() 内の SelectedConnectionInfo 設定時に IsConnecting が上書きされてしまう
            this.InitializeViewModelTargetPlatform();

            this.SetupPropertyDependencies();
            this.SetupCommands();
            this.SetupActions();

            this.SpyService.StateChanged += this.UpdateConnectionIcon;
            this.SettingsService.Applied += this.UpdateConnectionSettings;

            // 再生状態が変更されたら、コマンド状態を再評価する
            PropertyChangedObservation.GetObserver(_observerOwner, this.PlaybackService)
                .AddHandler(
                    target => target.State,
                    (target, args) =>
                    {
                        CommandManager.InvalidateRequerySuggested();
                        _viewModel.IsPlaying = this.PlaybackService.State == SpyPlaybackService.PlaybackState.Playing;
                    });

            _viewModel.SetStatusConnection();
        }

        private void InitializeViewModelTargetPlatform()
        {
            Ensure.Operation.NotNull(this.SettingsService);

            _viewModel.ConnectionInfos = CollectionViewSource.GetDefaultView(_connectionInfoViewModels);

            var connectionSetting = this.SettingsService.GetConnectionSettings();

            _viewModel.SelectedConnectionInfo = _connectionInfoViewModels.FirstOrDefault(
                item => item.TargetPlatformName == connectionSetting.TargetPlatformName);

            if (_viewModel.SelectedConnectionInfo == null)
            {
                connectionSetting.TargetPlatformName = ConnectionSettings.DefaultTargetPlatformName;
                connectionSetting.IsConnecting = false;

                _viewModel.SelectedConnectionInfo = _connectionInfoViewModels.FirstOrDefault(
                    item => item.TargetPlatformName == connectionSetting.TargetPlatformName);
            }

            _viewModel.IsConnecting = connectionSetting.IsConnecting;

            this.SpyTemporaryDataService.GetSpyConnectionInfo().SelectedConnectionInfo = _viewModel.SelectedConnectionInfo.ConnectionInfo;
        }

        private void InitializeConnectionInfos()
        {
            var connectionInfos = new List<ConnectionInfo>();

            foreach (var connectionPlugin in this.PluginsService.ConnectionPlugins)
            {
                var connectionInfo = new ConnectionInfo(connectionPlugin.TargetPlatformName, connectionPlugin.IsForNintendoSDK);
                connectionInfos.Add(connectionInfo);

                _connectionInfoViewModels.Add(
                    new ConnectionInfoViewModel(
                        connectionInfo,
                        connectionPlugin.SmallImageUri,
                        connectionPlugin.LargeImageUri,
                        connectionPlugin.Text,
                        selectedViewModel => _viewModel.SelectedConnectionInfo = selectedViewModel));
            }

            // 接続情報を一時データに登録
            this.SpyTemporaryDataService.GetSpyConnectionInfo().ConnectionInfos = connectionInfos;
        }

        private void InitializeGlobalTimelinePresenter()
        {
            _globalTimelinePresenter.Initialize(this.ServiceProvider, null);
            _viewModel.GlobalTimelineElement = _globalTimelinePresenter.Content;
        }

        private void UninitializeGlobalTimelinePresenter()
        {
            _globalTimelinePresenter.Uninitialize();
        }

        private void SetupSettings()
        {
            if (string.IsNullOrEmpty(this.SettingsService.GetWindowsSettings().MainWindowSettings.PanelLayouts))
            {
                this.DeserializeDockingLayouts(Resources.Files.DefaultPanelStates);
            }
            else
            {
                this.DeserializeDockingLayouts(
                    this.SettingsService.GetWindowsSettings().MainWindowSettings.PanelLayouts);
            }
        }

        private void UnsetupSettings()
        {
            var mainWindwoSettings = this.SettingsService.GetWindowsSettings().MainWindowSettings;

            mainWindwoSettings.PanelLayouts = this.SerializeDockingLayoutsData();

            mainWindwoSettings.Position.Left = _viewModel.Position.Left;
            mainWindwoSettings.Position.Top = _viewModel.Position.Top;
            mainWindwoSettings.Position.Width = _viewModel.Position.NormalWidth;
            mainWindwoSettings.Position.Height = _viewModel.Position.NormalHeight;

            // 最小化状態だったときは次回はノーマル状態で起動します。
            if (_viewModel.Position.WindowState == WindowState.Minimized)
            {
                mainWindwoSettings.Position.WindowState = WindowState.Normal;
            }
            else
            {
                mainWindwoSettings.Position.WindowState = _viewModel.Position.WindowState;
            }
        }

        private void DeserializeDockingLayouts(string data)
        {
            Assertion.Argument.NotNull(data);

            if (_window == null)
            {
                return;
            }

            _window.RequestDeserializeDockingLayouts(
                data,
                contentID =>
                {
                    var result = _viewModel.GetPanelViewModel(contentID);
                    if (result != null)
                    {
                        result.IsVisible = true;
                    }
                    return result;
                });
        }

        private string SerializeDockingLayoutsData()
        {
            var result = new StringBuilder();

            using (var writer = new StringWriter(result))
            {
                this.SerializeDockingLayouts(writer);
            }

            return result.ToString();
        }

        private void SerializeDockingLayouts(TextWriter writer)
        {
            Assertion.Argument.NotNull(writer);

            if (_window == null)
            {
                return;
            }

            _window.SerializeDockingLayouts(writer);
        }

        private PanelPresenter CreatePanelPresenter(SpyPanelPlugin panelPlugin)
        {
            Assertion.Argument.NotNull(panelPlugin);

            _panelPresenters.TryGetValue(panelPlugin, out PanelPresenter result);

            if (result != null)
            {
                return result;
            }

            result = panelPlugin.CreatePresenter();
            Ensure.Operation.NotNull(result);

            result.Initialize(this.ServiceProvider, this.GetCommandService());
            _panelPresenters.Add(panelPlugin, result);

            return result;
        }

        private void SetupPropertyDependencies()
        {
            // 接続設定の同期
            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.SelectedConnectionInfo,
                    (target, args) => this.ApplyConnectionSettings());

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.IsConnecting,
                    (target, args) =>
                    {
                        this.ApplyConnectionSettings();
                        CommandManager.InvalidateRequerySuggested();
                    });

            // 自動スクロール設定の同期
            _viewModel.IsAutoScrollEnabled = this.PlaybackService.IsAutoScrollEnabled;

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.IsAutoScrollEnabled,
                    (target, args) => this.PlaybackService.IsAutoScrollEnabled = target.IsAutoScrollEnabled);

            PropertyChangedObservation.GetObserver(_observerOwner, this.PlaybackService)
                .AddHandler(
                    target => target.IsAutoScrollEnabled,
                    (target, args) => _viewModel.IsAutoScrollEnabled = target.IsAutoScrollEnabled);

            // ウィンドウ状態の同期
            _viewModel.Position.SetModel(
                this.SettingsService.GetWindowsSettings().MainWindowSettings.Position);

            // 再生対象
            PropertyChangedObservation.GetObserver(_observerOwner, this.PlaybackService)
                .AddHandler(
                    target => target.CurrentTargetName,
                    (target, args) =>
                    {
                        _viewModel.CurrentPlaybackTargetName = target.CurrentTargetName;
                        CommandManager.InvalidateRequerySuggested();
                    });

            PropertyChangedObservation.GetObserver(_observerOwner, this.PlaybackService)
                .AddHandler(
                    target => target.WaveSourceOutputInfos,
                    (target, args) =>
                    {
                        _viewModel.WaveSourceOutputNames = target.WaveSourceOutputInfos.Select(info => info.Name).ToArray();

                        if (string.IsNullOrEmpty(_viewModel.CurrentSaveWaveDataTargetName))
                        {
                            _viewModel.CurrentSaveWaveDataTargetName = _viewModel.WaveSourceOutputNames.FirstOrDefault();
                        }
                    });

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.CurrentPlaybackTargetName,
                    (target, args) => this.PlaybackService.CurrentTargetName = target.CurrentPlaybackTargetName);
        }

        private void SetupCommands()
        {
            bool hasWaveformData() => !string.IsNullOrEmpty(this.PlaybackService.CurrentTargetName);

            // 接続
            _viewModel.ConnectCommand = new DelegateCommand(
                parameter => _viewModel.IsConnecting = !_viewModel.IsConnecting,
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        _viewModel.SelectedConnectionInfo != null;
                });

            // 開く
            _viewModel.OpenFileCommand = new DelegateCommand(
                parameter => this.SelectAndOpenSpyDataFiles(),
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        !_viewModel.IsConnecting &&
                        !this.SpyService.HasRecordedData;
                });

            // 読込の中止
            _viewModel.StopOpenFileCommand = new DelegateCommand(
                parameter => this.SpyService.StopLoadFiles(),
                parameter => _viewModel.IsLoading);

            // 閉じる
            _viewModel.CloseAllFilesCommand = new DelegateCommand(
                parameter => this.SpyService.ClearAllAndDeleteOldDataDirectories(),
                parameter =>
                {
                    return
                        !_viewModel.IsConnecting &&
                        !_viewModel.IsLoading &&
                        this.SpyService.HasData;
                });

            // 保存
            _viewModel.RecordCommand = new CommandServiceCommand(
                this.GetCommandService(),
                SpyCommands.RecordAllFiles,
                parameter => new RecordAllFilesCommandArgs()
                {
                    IsConnecting = _viewModel.IsConnecting,
                });

            // 再生
            _viewModel.PlayCommand = new DelegateCommand(
                parameter =>
                {
                    this.Play();
                },
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        !_viewModel.IsConnecting &&
                        hasWaveformData() &&
                        this.PlaybackService.State != SpyPlaybackService.PlaybackState.Playing;
                });

            _viewModel.StopCommand = new DelegateCommand(
                parameter => this.PlaybackService.Stop(),
                parameter => this.PlaybackService.State != SpyPlaybackService.PlaybackState.Stopped);

            _viewModel.PlayOrStopCommand = new DelegateCommand(
                parameter =>
                {
                    if (this.PlaybackService.State == SpyPlaybackService.PlaybackState.Playing)
                    {
                        this.PlaybackService.Stop();
                    }
                    else
                    {
                        this.Play();
                    }
                },
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        !_viewModel.IsConnecting &&
                        hasWaveformData();
                });

            _viewModel.SeekToBeginCommand = new DelegateCommand(
                parameter =>
                {
                    this.PlaybackService.SetCurrentTime(this.PlaybackService.Begin.Timestamp);
                },
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        hasWaveformData();
                });

            _viewModel.SeekToEndCommand = new DelegateCommand(
                parameter =>
                {
                    this.PlaybackService.SetCurrentTime(this.PlaybackService.End.Timestamp);
                },
                parameter =>
                {
                    return
                        !_viewModel.IsLoading &&
                        hasWaveformData();
                });

            _viewModel.OptionCommand = new DelegateCommand(
                parameter =>
                {
                    _optionWindowPresenter.Value.Initialize(this.ServiceProvider, null);
                    _optionWindowPresenter.Value.OwnerWindow = _window;
                    _optionWindowPresenter.Value.ResetPanelLayout = delegate ()
                        {
                            this.DeserializeDockingLayouts(Resources.Files.DefaultPanelStates);
                        };

                    _optionWindowPresenter.Value.Run();

                    _optionWindowPresenter.Value.Uninitialize();
                },
                parameter => !_viewModel.IsLoading);

            _viewModel.SaveWaveDataCommand = new CommandServiceCommand(
                this.GetCommandService(),
                SpyAudioCommands.SaveRecordWaveDataDialog,
                parameter => new AudioSaveRecordWaveDataDialogCommandArgs()
                {
                    WaveSourceOutputInfo = this.PlaybackService.WaveSourceOutputInfos
                        .FirstOrDefault(info => info.Name == _viewModel.CurrentSaveWaveDataTargetName),
                    SpyService = this.SpyService,
                    SpyPlaybackService = this.PlaybackService,
                    ConnectionSettings = this.SettingsService.GetConnectionSettings(),
                });
        }

        private void Play()
        {
            try
            {
                this.PlaybackService.RewindCurrentIfNecessary();
                this.PlaybackService.Play();
            }
            catch (SpyPlaybackService.PlaybackDeviceNotFoundException)
            {
                MessageBox.Show(_window, Messages.WarningPlaybackDeviceNotFound, Labels.WindowTitleError, MessageBoxButton.OK, MessageBoxImage.Error);
            }
            catch (SpyPlaybackService.PlaybackDeviceNotPluggedException)
            {
                MessageBox.Show(_window, Messages.WarningPlaybackDeviceNotPlugged, Labels.WindowTitleError, MessageBoxButton.OK, MessageBoxImage.Error);
            }
            catch (Exception exception)
            {
                MessageBox.Show(_window, exception.ToString(), Labels.WindowTitleError);
            }
        }

        private void SetupActions()
        {
            _viewModel.DragOverAction = e =>
            {
                var dragArgs = (DragEventArgs)e;

                if (this.SettingsService.GetConnectionSettings().IsConnecting ||
                    this.SpyService.HasRecordedData)
                {
                    dragArgs.Effects = DragDropEffects.None;
                    dragArgs.Handled = true;
                    return;
                }
            };

            _viewModel.DropAction = e =>
            {
                var filePaths = ((DragEventArgs)e).Data.GetData(DataFormats.FileDrop) as string[];

                if (filePaths == null)
                {
                    return;
                }

                if (filePaths.Length == 1 && Directory.Exists(filePaths[0]))
                {
                    this.OpenSpyDataDirectory(filePaths[0]);
                }
                else
                {
                    this.OpenSpyDataFiles(filePaths);
                }
            };
        }

        private string GetTargetPlatformName()
        {
            return _viewModel?.SelectedConnectionInfo?.TargetPlatformName ?? PlatformWin.Name;
        }

        private bool IsTargetManagerRequired(string targetPlatformName)
        {
            // HACK: TargetManagerを利用する Win, NXをPlatform名から判断して処理を行っています。
            if (targetPlatformName == "Win(NW4F)" || targetPlatformName == "Cafe")
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        private bool IsTargetManagerRunning()
        {
            return Process.GetProcessesByName("NintendoTargetManager").Count() > 0;
        }

        /// <summary>
        /// 接続設定を反映します。
        /// </summary>
        private void ApplyConnectionSettings()
        {
            Ensure.Operation.NotNull(this.SettingsService);

            string targetPlatformName = this.GetTargetPlatformName();

            var connectionSettings = this.SettingsService.GetConnectionSettings();

            if (connectionSettings.TargetPlatformName == targetPlatformName &&
                connectionSettings.IsConnecting == _viewModel.IsConnecting)
            {
                return;
            }

            if (_viewModel.IsConnecting == true && this.IsTargetManagerRequired(targetPlatformName))
            {
                if (!this.IsTargetManagerRunning())
                {
                    MessageBox.Show(_window, Resources.Messages.NotRunningTargetManager, Resources.Labels.WindowTitleError, MessageBoxButton.OK, MessageBoxImage.Error);
                }
            }

            this.SettingsService.GetConnectionSettings().TargetPlatformName = targetPlatformName;
            this.SettingsService.GetConnectionSettings().IsConnecting = _viewModel.IsConnecting;
            this.SettingsService.Apply();

            this.SpyTemporaryDataService.GetSpyConnectionInfo().SelectedConnectionInfo = _viewModel.SelectedConnectionInfo.ConnectionInfo;
        }

        /// <summary>
        /// 接続設定を更新します。
        /// </summary>
        private void UpdateConnectionSettings(object sender, EventArgs e)
        {
            this.InitializeViewModelTargetPlatform();
        }

        /// <summary>
        /// SpyDataID フラグを更新します。
        /// </summary>
        private void UpdateSpyDataIDFlags()
        {
            // TODO : ★コマンドハンドラ化して、MainWindowViewMOdel からコマンド発行するか
            HashSet<string> spyDataNames = new HashSet<string>();

            foreach (var panelViewModel in _viewModel.PanelViewModels)
            {
                if (!panelViewModel.IsVisible)
                {
                    continue;
                }

                foreach (var spyDataID in panelViewModel.RequiredSpyDataNames)
                {
                    spyDataNames.Add(spyDataID);
                }
            }

            this.SpyService.RequestSpyDatas(spyDataNames);
        }

        /// <summary>
        /// 接続状態の変更イベントを受けて、アイコンを更新します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="args">イベントデータを指定します。</param>
        private void UpdateConnectionIcon(object sender, EventArgs args)
        {
            this.UpdateConnectionIcon();
        }

        private void UpdateConnectionIcon()
        {
            if (_viewModel.SelectedConnectionInfo != null)
            {
                _viewModel.SelectedConnectionInfo.IsConnected = this.SpyService.State == SpyService.ServiceState.Running;
            }

            _window.UpdateStatusBarResources(
                _viewModel.SelectedConnectionInfo.IsConnected
                ? _connectedStatusBarBrush
                : _normalStatusBarBrush);
        }

        private void SelectAndOpenSpyDataFiles()
        {
            var dialog = new OpenFileDialog()
            {
                Title = Resources.Labels.SpyDataFileDialogTitle,
                Multiselect = true,
                Filter = string.Format(
                    "{0}(*{1})|*{1}",
                    Resources.Labels.SpyDataFileDescription,
                    ConstConfig.SpyDataFileExtention),
                CustomPlaces =
                {
                    new FileDialogCustomPlace(this.SpyService.WorkDirectoryPath),
                },
            };

            if (this.SpyService.HasLoadedData)
            {
                dialog.InitialDirectory = this.SpyService.LoadedDataDirectoryPath;
            }
            else
            {
                dialog.InitialDirectory = this.SpyService.WorkDirectoryPath;
            }

            bool? result = dialog.ShowDialog(_window);

            if (!result.HasValue || !result.Value)
            {
                return;
            }

            if (this.SpyService.IsRecordingDirectory(Path.GetDirectoryName(dialog.FileName)))
            {
                MessageBox.Show(_window, Resources.Messages.WarningCannotReadRecordingData, string.Empty, MessageBoxButton.OK, MessageBoxImage.Warning);
                return;
            }

            this.OpenSpyDataFiles(dialog.FileNames);
        }

        private void OpenSpyDataDirectory(string directoryPath)
        {
            OpenSpyDataFiles(Directory.EnumerateFiles(directoryPath));
        }

        private async void OpenSpyDataFiles(IEnumerable<string> filePaths)
        {
            var validFilePaths = filePaths
                .Where(it => Path.GetExtension(it) == ConstConfig.SpyDataFileExtention)
                .Where(it => File.Exists(it))
                .Select(it => Path.GetFullPath(it))
                .Distinct()
                .ToList();

            if (validFilePaths.IsEmpty())
            {
                return;
            }

            var dirNames = validFilePaths
                .Select(it => Path.GetDirectoryName(it))
                .Distinct();

            if (dirNames.Count() > 1)
            {
                MessageBox.Show(
                    _window,
                    Resources.Messages.FileOpenErrorMultipleDirectory,
                    Resources.Labels.WindowTitleError,
                    MessageBoxButton.OK,
                    MessageBoxImage.Error);

                return;
            }

            var spyService = this.SpyService;

            if (spyService.HasLoadedData && dirNames.First() != spyService.LoadedDataDirectoryPath)
            {
                var message = string.Format(
                    Resources.Messages.FileOpenErrorForInvalidPaths,
                    spyService.LoadedDataDirectoryPath);

                MessageBox.Show(_window, message, Resources.Labels.WindowTitleError, MessageBoxButton.OK, MessageBoxImage.Error);

                return;
            }

            var playbackService = this.PlaybackService;
            var isAutoScrollEnabled = playbackService.IsAutoScrollEnabled;

            playbackService.IsAutoScrollEnabled = false;
            playbackService.SetCurrentTime(SpyGlobalTime.Zero);

            _viewModel.StatusLoading.Progress = 0;
            _viewModel.SetStatusLoading();

            CommandManager.InvalidateRequerySuggested();

            var progress = new Progress<int>(p => _viewModel.StatusLoading.Progress = p);

            try
            {
                await spyService.LoadFiles(validFilePaths, progress);
            }
            catch (Exception ex)
            {
                MessageBox.Show(_window, ex.ToString(), Resources.Labels.WindowTitleError, MessageBoxButton.OK, MessageBoxImage.Error);
            }

            playbackService.SetCurrentTime(playbackService.End.Timestamp);
            playbackService.IsAutoScrollEnabled = isAutoScrollEnabled;

            _viewModel.SetStatusConnection();

            CommandManager.InvalidateRequerySuggested();
        }
    }
}
