﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows.Interop;
using System.Windows.Media;
using System.Xml;
using Xceed.Wpf.AvalonDock.Layout.Serialization;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public sealed partial class MainWindow
    {
        public class MessageReceivedEventArgs
        {
            public string Message { get; }

            /// <summary>
            /// メッセージの送信元に返される値です。
            /// </summary>
            public int Result { get; set; }

            public MessageReceivedEventArgs(string message)
            {
                this.Message = message;
            }
        }

        private const string StatusBarInnerBorderBrushKey = "StatusBarInnerBorderBrush";
        private const string StatusBarInnerBackgroundBrushKey = "StatusBarInnerBackgroundBrush";

        private string _dockingManagerLayoutData = null;
        private Func<string, object> _activatePanelFunc = null;

        public event EventHandler<MessageReceivedEventArgs> MessageReceived;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainWindow()
        {
            this.InitializeComponent();

            this.dockingManager.Loaded += (sender, e) =>
            {
                this.DeserializeDockingLayouts();
            };

            this.Loaded += (sender, e) =>
            {
                var source = HwndSource.FromHwnd(new WindowInteropHelper(this).Handle);
                source.AddHook(new HwndSourceHook(WndProc));
            };
        }

        private IntPtr WndProc(IntPtr hWnd, int msg, IntPtr wParam, IntPtr lParam, ref bool handled)
        {
            switch (msg)
            {
                case Win32.WM.WM_COPYDATA:
                    {
                        handled = true;

                        var message = IpcUtility.ReceiveStringMessage(hWnd, msg, wParam, lParam);
                        if (message == null)
                        {
                            return IntPtr.Zero;
                        }

                        return new IntPtr(this.OnMessageReceived(message));
                    }
            }

            return IntPtr.Zero;
        }

        private int OnMessageReceived(string message)
        {
            var args = new MessageReceivedEventArgs(message);

            this.MessageReceived?.Invoke(this, args);

            return args.Result;
        }

        public void RequestDeserializeDockingLayouts(string data, Func<string, object> activatePanel)
        {
            _dockingManagerLayoutData = data;
            _activatePanelFunc = activatePanel;

            this.Dispatcher.InvokeAsync(this.DeserializeDockingLayouts);
        }

        public void SerializeDockingLayouts(TextWriter writer)
        {
            Ensure.Argument.NotNull(writer);

            var serializer = new XmlLayoutSerializer(this.dockingManager);
            serializer.Serialize(writer);
        }

        public void AddRibbonTabItems(IEnumerable<Fluent.RibbonTabItem> tabItems)
        {
            tabItems.ForEach(tabItem => this.ribbon.Tabs.Add(tabItem));
        }

        public void UpdateStatusBarResources(Brush brush)
        {
            this.statusBar.Resources.BeginInit();

            this.statusBar.Resources[StatusBarInnerBorderBrushKey] = brush;
            this.statusBar.Resources[StatusBarInnerBackgroundBrushKey] = brush;

            this.statusBar.Resources.EndInit();
        }

        private void DeserializeDockingLayouts()
        {
            // AvalonDock がロードされるまで遅延する（レイアウトの復元に失敗するため）
            if (!this.dockingManager.IsLoaded)
            {
                return;
            }

            if (string.IsNullOrEmpty(_dockingManagerLayoutData) || _activatePanelFunc == null)
            {
                return;
            }

            var serializer = new XmlLayoutSerializer(this.dockingManager);

            serializer.LayoutSerializationCallback += (sender, e) =>
            {
                e.Content = _activatePanelFunc(e.Model.ContentId);
            };

            var settings = new XmlReaderSettings()
            {
                IgnoreWhitespace = true,
                IgnoreComments = true,
            };

            using (var reader = new StringReader(_dockingManagerLayoutData))
            {
                using (var xmlReader = XmlReader.Create(reader, settings))
                {
                    serializer.Deserialize(xmlReader);
                }
            }

            this.dockingManager.UpdateLayout();

            // レイアウトが実際にWindowに反映されるのを待って
            // Windowがオンスクリーンかどうか調べます。
            this.Dispatcher.InvokeAsync(ExposeOffScreenWindow);

            _dockingManagerLayoutData = null;
            _activatePanelFunc = null;
        }

        /// <summary>
        /// メインウィンドウ、およびフローティングウィンドウがスクリーン内に表示されるように位置を調整します。
        /// </summary>
        private void ExposeOffScreenWindow()
        {
            ExposeOffScreenWindow(this);

            foreach (var window in this.dockingManager.FloatingWindows)
            {
                ExposeOffScreenWindow(window);
            }
        }

        /// <summary>
        /// ウィンドウがスクリーン内に表示されるように位置を調整します。
        /// </summary>
        private void ExposeOffScreenWindow(System.Windows.Window window)
        {
            // ウィンドウのタイトルバーの部分がいずれかのスクリーン内にあるかを調べます。
            var captionBounds = new System.Drawing.Rectangle()
            {
                Location = new System.Drawing.Point((int)window.Left, (int)window.Top),
                Width = (int)window.ActualWidth,
                Height = System.Windows.Forms.SystemInformation.CaptionHeight,
            };

            // どのスクリーン内にもなかった場合はプライマリスクリーンの原点位置に移動させます。
            if (!IsOnScreen(captionBounds))
            {
                window.Left = 0;
                window.Top = 0;
            }
        }

        /// <summary>
        /// 矩形がいずれかのスクリーン内にあるかを調べます。
        /// </summary>
        private bool IsOnScreen(System.Drawing.Rectangle bounds)
        {
            // WPFではマルチスクリーンの情報が公開されていないのでFormsを使います。
            var screens = System.Windows.Forms.Screen.AllScreens;

            bool bVisible = false;
            foreach (var screen in screens)
            {
                bVisible = screen.WorkingArea.IntersectsWith(bounds);
                if (bVisible)
                {
                    break;
                }
            }

            return bVisible;
        }
    }
}
