﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using System;

namespace NintendoWare.Spy.Windows
{
    internal sealed class GlobalTimelineViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private SpyPlaybackService _model;

        private bool _isAutoScrollEnabled;
        private double _begin;
        private double _end;
        private double _current;
        private bool _isDragging;
        private int _tokenGrab;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;

        //-----------------------------------------------------------------

        public double Begin
        {
            get { return _begin; }
            set { this.SetPropertyValue(ref _begin, value); }
        }

        public double End
        {
            get { return _end; }
            set { this.SetPropertyValue(ref _end, value); }
        }

        public double Current
        {
            get { return _current; }
            set { this.SetCurrentImpl(value, from: this); }
        }

        public bool IsDragging
        {
            get
            {
                return _isDragging;
            }
            set
            {
                if (_isDragging != value)
                {
                    _isDragging = value;

                    if (_isDragging)
                    {
                        if (_model != null)
                        {
                            _tokenGrab = _model.GetGrabForCurrent();
                        }
                    }
                    else
                    {
                        if (_model != null)
                        {
                            _model.ReleaseGrabForCurrent(_tokenGrab);
                        }
                        _tokenGrab = SpyPlaybackService.TokenGrabInvalid;
                    }
                }
            }
        }

        public bool IsAutoScrollEnabled
        {
            get { return _isAutoScrollEnabled; }

            set
            {
                if (_isAutoScrollEnabled == value)
                {
                    return;
                }

                _isAutoScrollEnabled = value;

                if (_model != null)
                {
                    _model.IsAutoScrollEnabled = value;
                }

                this.NotifyPropertyChanged();
            }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set
            {
                if (this.SetPropertyValue(ref _timeUnit, value))
                {
                    if (_model != null)
                    {
                        this.Begin = ToDouble(_model.Begin);
                        this.End = ToDouble(_model.End);
                        this.Current = ToDouble(_model.Current);
                    }
                }
            }
        }

        //-----------------------------------------------------------------

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            base.DisposeManagedInstance();
        }

        public void SetModel(SpyPlaybackService model)
        {
            if (_model == model)
            {
                return;
            }

            if (_model != null)
            {
                PropertyChangedObservation.RemoveObservers(_observerOwner);
            }

            _model = model;
            _tokenGrab = SpyPlaybackService.TokenGrabInvalid;

            if (model != null)
            {
                // プロパティ同期
                this.Begin = ToDouble(model.Begin);
                this.End = ToDouble(model.End);
                this.Current = ToDouble(model.Current);
                this.IsAutoScrollEnabled = model.IsAutoScrollEnabled;

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Begin,
                        (target, args) => this.Begin = ToDouble(target.Begin));

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.End,
                        (target, args) => this.End = ToDouble(target.End));

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Current,
                        (target, args) => this.SetCurrentImpl(ToDouble(target.Current), from: null));

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.IsAutoScrollEnabled,
                        (target, args) => this.IsAutoScrollEnabled = target.IsAutoScrollEnabled);
            }
            else
            {
                this.Begin = 0.0d;
                this.End = 0.0d;
                this.Current = 0.0d;
                this.IsAutoScrollEnabled = false;
            }
        }

        private void SetCurrentImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.Begin, this.End);
            }

            if (this.SetPropertyValue(ref _current, value, nameof(this.Current)))
            {
                if (from == this)
                {
                    if (!double.IsNaN(value))
                    {
                        _model?.SetCurrentTime((long)value, this.TimeUnit, _tokenGrab);
                    }
                }
            }
        }

        private double ToDouble(SpyTime time)
        {
            return time.SelectFrameValue(this.TimeUnit);
        }
    }
}
