﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using System.Threading;
using System.Windows.Input;
using System.Windows.Markup;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows
{
    [ContentProperty(nameof(VisibleItems))]
    public sealed class ApiCallSpyViewModel : ObservableObject
    {
        private ApiCallSpyModel _model;
#if false // TODO: NW-104
        private SoundDataInfoSpyModel _soundDataInfoModel;
#endif

        private ApiCallLineViewModel _selectedItem;

        private SpyTime _currentTime = SpyTime.InvalidValue;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private ApiCallLineViewModel _itemLinedAbove = null;
        private ApiCallLineViewModel _itemLinedBelow = null;

        private string _includeFilter = string.Empty;
        private string _includeFilterLower = string.Empty;

        private readonly MergedRequestDispatcher _updateLogLinesDispatcher = new MergedRequestDispatcher();
        private readonly MergedRequestDispatcher _refreshVisibleLogLinesDispatcher = new MergedRequestDispatcher();

        //-----------------------------------------------------------------

        /// <summary>
        /// モデル、ビューモデル変換時に利用する同期コンテキストを取得または設定します。
        /// </summary>
        public SynchronizationContext SyncContext { get; set; }

        /// <summary>
        /// すべてのログアイテムのコレクションです。
        /// </summary>
        public ObservableCollection<ApiCallLineViewModel> Items { get; } = new ObservableCollection<ApiCallLineViewModel>();

        /// <summary>
        /// 表示されるログアイテムのコレクションです。
        /// </summary>
        public ObservableCollection<ApiCallLineViewModel> VisibleItems { get; } = new ObservableCollection<ApiCallLineViewModel>();

        /// <summary>
        /// カレント時間に一致するアイテムのコレクションです。
        /// </summary>
        public ObservableCollection<ApiCallLineViewModel> CurrentItems { get; } = new ObservableCollection<ApiCallLineViewModel>();

        /// <summary>
        /// 選択されたアイテムです。
        /// </summary>
        public ApiCallLineViewModel SelectedItem
        {
            get { return _selectedItem; }
            set { this.SetPropertyValue(ref _selectedItem, value); }
        }

        /// <summary>
        /// カレント時間です。
        /// </summary>
        public SpyTime CurrentTime
        {
            get { return _currentTime; }

            set
            {
                if (this.SetPropertyValue(ref _currentTime, value))
                {
                    this.RefreshCurrentItems();
                }
            }
        }

        /// <summary>
        /// 時間の表示単位です。
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }

            set
            {
                if (this.SetPropertyValue(ref _timeUnit, value))
                {
                    this.RefreshVisibleItemsState();
                    this.RefreshCurrentItems();
                }
            }
        }

        /// <summary>
        /// カレント時間の次のアイテム。
        /// アイテムの上にラインが引かれます。
        /// カレントフレームに一致するアイテムがある場合は null になります。
        /// </summary>
        public ApiCallLineViewModel ItemLinedAbove
        {
            get { return _itemLinedAbove; }

            private set
            {
                if (_itemLinedAbove == value)
                {
                    return;
                }

                if (_itemLinedAbove != null)
                {
                    _itemLinedAbove.IsAboveLineVisible = false;
                }

                _itemLinedAbove = value;

                if (_itemLinedAbove != null)
                {
                    _itemLinedAbove.IsAboveLineVisible = true;
                }

                this.NotifyPropertyChanged();
            }
        }

        /// <summary>
        /// カレントフレームの前のアイテム。
        /// アイテムの下にラインが引かれます。
        /// カレントフレームに一致するアイテムがある場合は null になります。
        /// </summary>
        public ApiCallLineViewModel ItemLinedBelow
        {
            get { return _itemLinedBelow; }

            private set
            {
                if (_itemLinedBelow == value)
                {
                    return;
                }

                if (_itemLinedBelow != null)
                {
                    _itemLinedBelow.IsBelowLineVisible = false;
                }

                _itemLinedBelow = value;

                if (_itemLinedBelow != null)
                {
                    _itemLinedBelow.IsBelowLineVisible = true;
                }

                this.NotifyPropertyChanged();
            }
        }

        /// <summary>
        /// この文字列を含むアイテムが表示されます。
        /// </summary>
        public string IncludeFilter
        {
            get { return _includeFilter; }
            set
            {
                if (this.SetPropertyValue(ref _includeFilter, value))
                {
                    // 以後に到着するアイテムにはただちにフィルタを適用します。
                    _includeFilterLower = value?.ToLower();

                    // 余裕ができたらリスト全体をリフレッシュします。
                    _refreshVisibleLogLinesDispatcher.Request(RefreshVisibleItems, DispatcherPriority.Background);
                }
            }
        }

        /// <summary>
        /// クリップボードにログをコピーします。
        /// </summary>
        public ICommand CopyAllApiCallLogsCommand { get; set; }

        /// <summary>
        /// 新しい表示アイテムが追加されたときに通知します。
        /// </summary>
        public event EventHandler VisibleItemsAdded;

        //-----------------------------------------------------------------

        public ApiCallSpyViewModel()
        {
        }

        protected override void DisposeManagedInstance()
        {
            this.Items.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        /// <summary>
        /// モデルを設定します。
        /// </summary>
        /// <param name="model">関連付けるモデルを指定します。</param>
        public void SetModel(ApiCallSpyModel model)
        {
            if (_model == model)
            {
                return;
            }

            if (_model != null)
            {
                var col = _model.LogLines as INotifyCollectionChanged;
                if (col != null)
                {
                    col.CollectionChanged -= this.OnLogLinesCollectionChanged;
                }
            }

            if (model != null)
            {
                var col = model.LogLines as INotifyCollectionChanged;
                if (col != null)
                {
                    col.CollectionChanged += this.OnLogLinesCollectionChanged;
                }
            }

            _model = model;

            this.SyncLogLines();
        }

#if false // TODO: NW-104
        public void SetSoundDataInfoSpyModel(SoundDataInfoSpyModel model)
        {
            if (_soundDataInfoModel == model)
            {
                return;
            }

            _soundDataInfoModel = model;

            this.RefreshLogLineLabels();
        }
#endif

        public bool FilterLogLine(ApiCallLineViewModel logLine)
        {
            if (!string.IsNullOrEmpty(_includeFilterLower))
            {
                if (logLine.Text.ToLower().Contains(_includeFilterLower))
                {
                    return true;
                }

                return false;
            }

            return true;
        }

        //-----------------------------------------------------------------

        private void RefreshVisibleItems()
        {
            this.VisibleItems.Clear();
            this.ClearCurrentItems(clearItemLined: false);

            ApiCallLineViewModel itemLinedAbove = null;
            ApiCallLineViewModel itemLinedBelow = null;

            foreach (var item in this.Items)
            {
                if (this.FilterLogLine(item))
                {
                    this.AddToVisibleItems(item, ref itemLinedAbove, ref itemLinedBelow);
                }
            }

            this.ItemLinedAbove = itemLinedAbove;
            this.ItemLinedBelow = itemLinedBelow;
        }

        private void RefreshVisibleItemsState()
        {
            if (this.VisibleItems.IsEmpty())
            {
                return;
            }

            // NOTE: 先頭のアイテムは常に IsFirstAtTime == true なので再計算は不要。

            foreach (var pair in this.VisibleItems.SuccessingPair())
            {
                var prev = pair.Item1;
                var curr = pair.Item2;

                curr.IsFirstAtTime = SpyTime.Compare(prev.Time, curr.Time, this.TimeUnit) != 0;
            }
        }

        private void SyncLogLines()
        {
            this.Items.ForEach(it => it.Dispose());
            this.Items.Clear();
            this.VisibleItems.Clear();
            this.ClearCurrentItems();

            this.UpdateLogLines();
        }

        private void OnLogLinesCollectionChanged(object sender, NotifyCollectionChangedEventArgs args)
        {
            _updateLogLinesDispatcher.Request(UpdateLogLines, DispatcherPriority.Background);
        }

        /// <summary>
        /// 新しいログを取り込みます。
        /// </summary>
        private void UpdateLogLines()
        {
            if (_model == null)
            {
                return;
            }

            int count = _model.LogLines.Count;
            if (count == this.Items.Count)
            {
                return;
            }

            ApiCallLineViewModel itemLinedAbove = this.ItemLinedAbove;
            ApiCallLineViewModel itemLinedBelow = this.ItemLinedBelow;

            for (int i = this.Items.Count; i < count; ++i)
            {
                var item = new ApiCallLineViewModel(_model.LogLines[i]);

#if false // TODO: NW-104
                item.ResolveLabel(_soundDataInfoModel);
#endif

                if (this.FilterLogLine(item))
                {
                    this.AddToVisibleItems(item, ref itemLinedAbove, ref itemLinedBelow);
                }

                this.Items.Add(item);
            }

            this.ItemLinedAbove = itemLinedAbove;
            this.ItemLinedBelow = itemLinedBelow;

            this.VisibleItemsAdded?.Invoke(this, EventArgs.Empty);
        }

        private void AddToVisibleItems(
            ApiCallLineViewModel item,
            ref ApiCallLineViewModel itemLinedAbove,
            ref ApiCallLineViewModel itemLinedBelow)
        {
            var last = this.VisibleItems.LastOrDefault();

            item.IsFirstAtTime = last == null ? true : (SpyTime.Compare(last.Time, item.Time, this.TimeUnit) != 0);

            var compResult = SpyTime.Compare(item.Time, this.CurrentTime, this.TimeUnit);
            if (compResult == 0)
            {
                // アイテムがカレント時間に一致の場合。
                item.IsCurrent = true;
                this.CurrentItems.Add(item);

                itemLinedAbove = null;
                itemLinedBelow = null;
            }
            else if (compResult < 0)
            {
                // アイテムがカレント時間より前の場合。
                itemLinedBelow = item;
            }
            else if (compResult > 0)
            {
                // アイテムがカレント時間より後の場合。
                if (itemLinedAbove == null)
                {
                    itemLinedAbove = item;
                }
            }

            this.VisibleItems.Add(item);
        }

#if false // TODO: NW-104
        private void RefreshLogLineLabels()
        {
            if (_soundDataInfoModel == null)
            {
                return;
            }

            foreach (var item in this.Items)
            {
                item.ResolveLabel(_soundDataInfoModel);
            }
        }
#endif

        private void RefreshCurrentItems()
        {
            if (this.VisibleItems.Count == 0)
            {
                return;
            }

            if (!this.CurrentTime.Timestamp.IsValid)
            {
                this.ClearCurrentItems();
                return;
            }

            var index = BinarySearchUtility.BinarySearch(this.VisibleItems, this.CurrentTime, i => i.Time, new SpyTimeTimeUnitComparer(this.TimeUnit), BinarySearchUtility.Options.SmallestIndex);
            if (index < 0)
            {
                // カレント時間に一致するアイテムが無い場合。
                // ItemLinedAbove と ItemLinedBelow のどちらかを設定します。

                this.ClearCurrentItems(clearItemLined: false);

                ApiCallLineViewModel itemLinedAbove = null;
                ApiCallLineViewModel itemLinedBelow = null;

                index = ~index; // CurrentTime を超えた最初のアイテムのインデックス。
                if (index < this.VisibleItems.Count)
                {
                    itemLinedAbove = this.VisibleItems[index];
                }
                else
                {
                    itemLinedBelow = this.VisibleItems.Last();
                }

                this.ItemLinedAbove = itemLinedAbove;
                this.ItemLinedBelow = itemLinedBelow;
            }
            else
            {
                // カレント時間に一致するアイテムがある場合。
                // 一致するすべてのアイテムを currentItems に登録します。

                this.ClearCurrentItems();

                for (; index < this.VisibleItems.Count; ++index)
                {
                    var item = this.VisibleItems[index];

                    if (SpyTime.Compare(this.CurrentTime, item.Time, this.TimeUnit) != 0)
                    {
                        break;
                    }

                    item.IsCurrent = true;
                    this.CurrentItems.Add(item);
                }
            }
        }

        private void ClearCurrentItems(bool clearItemLined = true)
        {
            if (clearItemLined)
            {
                this.ItemLinedAbove = null;
                this.ItemLinedBelow = null;
            }

            if (!this.CurrentItems.IsEmpty())
            {
                foreach (var line in this.CurrentItems)
                {
                    line.IsCurrent = false;
                }

                this.CurrentItems.Clear();
            }
        }
    }
}
