﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using Nintendo.ToolFoundation.Metadata;
using NintendoWare.Spy.Foundation;
using NintendoWare.Spy.Foundation.Communications;
using NintendoWare.Spy.Framework;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Resources;
using NintendoWare.Spy.Settings;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows;

namespace NintendoWare.Spy
{
    /// <summary>
    /// App.xaml の相互作用ロジック
    /// </summary>
    public partial class App : Application
    {
        private const string PluginsDirectoryName = "Plugins";
        private CommandLineOptionSettings _commandLineOptionSettings = null;

        private string _warningMessage = null; // 重複プラグインの警告メッセージ暫定対処

        //-----------------------------------------------------------------

        private SpyFramework _framework;

        //-----------------------------------------------------------------

        protected override void OnStartup(StartupEventArgs e)
        {
            InitializeCurrentUICulture(Thread.CurrentThread);

            var helpMessages = new StringBuilder();
            var errorMessages = new StringBuilder();

            try
            {
                CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = Messages.CommandLine_ApplicationDescription,
                    HelpWriter = (help) => helpMessages.AppendLine(help),
                    ErrorWriter = (error) => errorMessages.AppendLine(error),
                };

                if (new CommandLineParser(commandLineParserSettings).ParseArgs(e.Args, out _commandLineOptionSettings) == false)
                {
                    this.ShowCommandLineParserMessages(helpMessages.ToString(), errorMessages.ToString());
                    this.Shutdown();
                    return;
                }
            }
            catch (ArgumentException)
            {
                this.ShowCommandLineParserMessages(helpMessages.ToString(), errorMessages.ToString());
                this.Shutdown(-1);
                return;
            }

            if (_commandLineOptionSettings.Command != null)
            {
                try
                {
                    var command = _commandLineOptionSettings.Command;
                    switch (command)
                    {
                        case CommandMessages.Connect.CommandName:
                            this.SendConnectCommandMessage();
                            this.Shutdown(0);
                            return;

                        case CommandMessages.Disconnect.CommandName:
                            this.SendDisconnectCommandMessage();
                            this.Shutdown(0);
                            return;

                        default:
                            throw new AppException(string.Format(Messages.UnexpectedCommand, command));
                    }
                }
                catch (AppException ex)
                {
                    Console.Error.WriteLine(string.Format(Messages.ConsoleErrorMessage, ex.Message));
                    this.Shutdown(-1);
                    return;
                }
            }

            new SplashScreen("Resources/Images/Splash.png").Show(autoClose: true);

            base.OnStartup(e);

            this.DispatcherUnhandledException += (sender, args) =>
            {
                if (!System.Diagnostics.Debugger.IsAttached)
                {
                    MessageBox.Show(
                        args.Exception.ToString(),
                        this.GetApplicationName(),
                        MessageBoxButton.OK,
                        MessageBoxImage.Error,
                        MessageBoxResult.OK,
                        MessageBoxOptions.DefaultDesktopOnly);
                }
            };

            _framework = SpyFramework.CreateInstance();

            var settings = new ApplicationFrameworkSettings()
            {
                CreateRootServiceProvider = this.CreateRootServiceProvider,
                CreateApplicationPresenterFunc = () => new RootPresenter(),
            };

            _framework.Initialize(settings);
            _framework.Run();
        }

        private void SendConnectCommandMessage()
        {
            var result = this.SendCommandMessage(CommandMessages.Connect.CommandName);

            if (result <= 0)
            {
                switch (result)
                {
                    case CommandMessages.Connect.Result.TargetIsNotSelected:
                        throw new AppException(Messages.TargetIsNotSelected);

                    case CommandMessages.Connect.Result.NotRunningTargetManager:
                        throw new AppException(Messages.NotRunningTargetManager);

                    default:
                        throw new AppException(string.Format(Messages.UnexpectedCommandResult, result));
                }
            }
        }

        private void SendDisconnectCommandMessage()
        {
            var result = this.SendCommandMessage(CommandMessages.Disconnect.CommandName);

            if (result <= 0)
            {
                throw new AppException(string.Format(Messages.UnexpectedCommandResult, result));
            }
        }

        private int SendCommandMessage(string command)
        {
            var processName = Assembly.GetEntryAssembly().GetName().Name;
            var processId = Process.GetCurrentProcess().Id;

            var processes = Process.GetProcessesByName(processName)
                .Concat(Process.GetProcessesByName(processName + ".vshost"))
                .Where(it => it.Id != processId)
                .Where(it => it.MainWindowHandle != IntPtr.Zero)
                .Take(2)
                .ToArray();

            if (processes.IsEmpty())
            {
                throw new AppException(Messages.CommandReceiverNotFound);
            }

            if (processes.Count() > 1)
            {
                throw new AppException(Messages.CommandReceiverFoundMultiple);
            }

            try
            {
                var result = IpcUtility.SendStringMessage(processes.First().MainWindowHandle, command);

                if (result == CommandMessages.Result.UnexpectedCommand)
                {
                    throw new AppException(Messages.UnexpectedCommandByReceiver);
                }

                return result;
            }
            catch (IpcUtility.IpcUtilityException e)
            {
                throw new AppException(string.Format(Messages.CommandTransmissionFailed, e.Message));
            }
        }

        private string GetApplicationName()
        {
            return AssemblyHelper.GetAssemblyTitle(Assembly.GetEntryAssembly());
        }

        private void ShowCommandLineParserMessages(string help, string error)
        {
            var title = this.GetApplicationName();

            if (!string.IsNullOrEmpty(help))
            {
                MessageBox.Show(
                    help,
                    title,
                    MessageBoxButton.OK,
                    MessageBoxImage.Information,
                    MessageBoxResult.OK,
                    MessageBoxOptions.DefaultDesktopOnly);

                Console.Out.Write(help);
            }

            if (!string.IsNullOrEmpty(error))
            {
                MessageBox.Show(
                    error,
                    title,
                    MessageBoxButton.OK,
                    MessageBoxImage.Error,
                    MessageBoxResult.OK,
                    MessageBoxOptions.DefaultDesktopOnly);

                Console.Error.Write(error);
            }
        }

        protected override void OnExit(ExitEventArgs e)
        {
            if (_framework != null)
            {
                _framework.Uninitialize();
                _framework.Dispose();
                _framework = null;
            }

            base.OnExit(e);
        }

        private ServiceProvider CreateRootServiceProvider()
        {
            Assertion.Operation.NotNull(_commandLineOptionSettings);

            var result = new ServiceProvider();

            var teamSettingsService = this.CreateTeamSettingsService();
            if (_commandLineOptionSettings.TeamSettingsFile != null)
            {
                teamSettingsService.Load(_commandLineOptionSettings.TeamSettingsFile);
            }
            var pluginsService = this.CreatePluginsService(this.GetPluginAssemblies(teamSettingsService.PluginDirectoryPaths));
            var hostIOService = this.CreateHostIOService(pluginsService);
            var spyService = this.CreateSpyService(pluginsService, hostIOService);
            var settingsService = this.CreateSettingsService();

            result.InstallService(typeof(TeamSettingsService), teamSettingsService);
            result.InstallService(typeof(PluginsService), pluginsService);
            result.InstallService(typeof(ApplicationInfoService), this.CreateApplicationInfoService());
            result.InstallService(typeof(HostIOService), hostIOService);
            result.InstallService(typeof(SpyService), spyService);
            result.InstallService(typeof(SpyPlaybackService), this.CreateSpyPlaybackService(spyService, settingsService));
            result.InstallService(typeof(SpyTemporaryDataService), this.CreateSpyTemporaryDataService());
            result.InstallService(typeof(SettingsService), settingsService);

            return result;
        }

        /// <summary>
        /// プラグインアセンブリの列挙子を取得します。
        /// </summary>
        /// <returns>プラグインアセンブリの列挙子を返します。</returns>
        private IEnumerable<Assembly> GetPluginAssemblies(IEnumerable<string> additionalPaths)
        {
            // NOTE :
            // ファイルパスからアセンブリロードする際には、Assembly.Load() を利用する。
            // Assembly.LoadFile(), Assembly.LoadFrom() を使用すると、サテライトアセンブリの
            // 参照先パスとして "(アセンブリの場所)/(ロケールID)/" (例:"Plugins/en/") が利用できない。
            // .NET の仕様としてそれが正しいのかどうかは確認できていないが、
            // MEF の AssemblyCatalog にアセンブリロードさせた場合は、Assembly.Load() と同じ挙動に
            // なるので、そちらに合わせておくのが安全と判断。

            // このアセンブリ
            yield return Assembly.GetEntryAssembly();

            // SpyService を含むアセンブリ（SpyFoundation）
            yield return typeof(SpyService).Assembly;

            // Plugins/*.dll
            var pluginsDirectoryPath = Path.Combine(
                Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                PluginsDirectoryName);

            // 重複しているプラグインとしていないプラグインで処理を分ける
            StringBuilder builder = new StringBuilder();
            foreach (var asm in
                new[] { pluginsDirectoryPath }
                .Concat(additionalPaths)
                .Where(d => Directory.Exists(d))
                .GroupBy(d => d.ToLower())
                .SelectMany(d => Directory.GetFiles(d.First(), "*.dll"))
                .Select(f => AssemblyName.GetAssemblyName(f))
                .GroupBy(a => a.Name))
            {
                if (asm.Count() == 1)
                {
                    // 重複していないプラグインのみ読み込む
                    yield return Assembly.Load(asm.First());
                }
                else
                {
                    // 重複しているプラグインを警告メッセージに追加
                    asm.ForEach(a => builder.AppendFormat("\"{0}\"\n", Regex.Replace(a.CodeBase, "^file:///", string.Empty, RegexOptions.IgnoreCase)));
                }
            }

            if (builder.Length > 0)
            {
                _warningMessage = string.Format(Messages.WarningDuplicatedPluginFile, builder.ToString());
            }
        }

        // 重複プラグインの警告メッセージ暫定対処
        public string GetWarningMessage()
        {
            return _warningMessage;
        }

        private PluginsService CreatePluginsService(IEnumerable<Assembly> assemblies)
        {
            var result = new PluginsService();
            result.Initialize(assemblies);

            return result;
        }

        private ApplicationInfoService CreateApplicationInfoService()
        {
            var result = new ApplicationInfoService();
            result.Initialize(Assembly.GetEntryAssembly());

            return result;
        }

        private HostIOService CreateHostIOService(PluginsService pluginsService)
        {
            var result = new HostIOService();
            result.Initialize(pluginsService.HostIOPlugins);

            return result;
        }

        private SpyService CreateSpyService(PluginsService pluginsService, HostIOService hostIOService)
        {
            var result = new SpyService();
            result.Initialize(pluginsService.ModelPlugins, hostIOService);

            return result;
        }

        private SpyPlaybackService CreateSpyPlaybackService(SpyService spyService, SettingsService settingsService)
        {
            var result = new SpyPlaybackService();
            result.Initialize(spyService, settingsService);

            return result;
        }

        private SpyTemporaryDataService CreateSpyTemporaryDataService()
        {
            return new SpyTemporaryDataService();
        }

        private SettingsService CreateSettingsService()
        {
            return new SettingsService();
        }

        private TeamSettingsService CreateTeamSettingsService()
        {
            return new TeamSettingsService();
        }

        private void InitializeCurrentUICulture(Thread thread)
        {
            string satelliteAssemblyLocation =
                Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "en");
            var oneOfSatelliteAssemblies = Path.Combine(satelliteAssemblyLocation, "Spy.resources.dll");

            if (!File.Exists(oneOfSatelliteAssemblies))
            {
                thread.CurrentCulture = new CultureInfo("ja-JP");
                thread.CurrentUICulture = new CultureInfo("ja-JP");
            }
            else
            {
                thread.CurrentCulture = new CultureInfo("en-US");
                thread.CurrentUICulture = new CultureInfo("en-US");
            }
        }

        private class AppException : ApplicationException
        {
            public AppException(string message)
                : base(message)
            { }
        }

        private class CommandLineOptionSettings
        {
            [CommandLineOption("team-settings-file", ValueName = "team-settings-file", Description = nameof(Messages.CommandLine_UsageMessage_Option_TeamSettingsFile), DescriptionConverterName = nameof(LocalizeDescription))]
            public string TeamSettingsFile { get; set; }

            [CommandLineOption("command", ValueName = "command", Description = nameof(Messages.CommandLine_UsageMessage_Value_Command), DescriptionConverterName = nameof(LocalizeDescription))]
            public string Command { get; set; }

            /// <summary>
            /// ローカライズしたオプションの説明を取得します。
            /// </summary>
            /// <param name="description">Descriptionプロパティの値</param>
            /// <param name="valueName">引数名またはオプション名</param>
            /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
            public static string LocalizeDescription(string description, string valueName)
            {
                return Messages.ResourceManager.GetString(description, Messages.Culture);
            }
        }
    }
}
