﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System.Linq;

namespace NintendoWare.SpySample.Windows
{
    public class SampleAPanelViewModel : ObservableObject
    {
        private readonly object observerOwner = new object();
        private readonly object plotObserverOwner = new object();

        private readonly SpyPlaybackService playbackService;

        private PlotSpyModel plotModel;

        private PlotSpyModel.PlotFloat dataBufferLengthPlotFloat;
        private PlotSpyModel.PlotFloat dataBufferUsagePlotFloat;

        private double dataBufferLength = 0d;
        private double dataBufferUsage = 0d;

        public SampleAPanelViewModel(SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(playbackService);

            this.playbackService = playbackService;

            PropertyChangedObservation.GetObserver(this.observerOwner, this.playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateCurrentTime());
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(this.observerOwner);
            CollectionChangedObservation.RemoveObservers(this.plotObserverOwner);

            base.DisposeManagedInstance();
        }

        public double DataBufferLength
        {
            get { return this.dataBufferLength; }
            private set
            {
                if (this.SetPropertyValue(ref this.dataBufferLength, value))
                {
                    this.NotifyPropertyChanged(nameof(DataBufferUseRate));
                }
            }
        }

        public double DataBufferUsage
        {
            get { return this.dataBufferUsage; }
            private set
            {
                if (this.SetPropertyValue(ref this.dataBufferUsage, value))
                {
                    this.NotifyPropertyChanged(nameof(DataBufferUseRate));
                }
            }
        }

        public double DataBufferUseRate
        {
            get
            {
                return this.DataBufferLength > 0
                    ? this.DataBufferUsage / this.DataBufferLength
                    : 0;
            }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (this.plotModel == model)
            {
                return;
            }

            this.plotModel = model;
            this.SyncPlotItems();
            this.UpdateCurrentTime();
        }

        private void SyncPlotItems()
        {
            CollectionChangedObservation.RemoveObservers(this.plotObserverOwner);

            if (this.plotModel == null)
            {
                this.dataBufferLengthPlotFloat = null;
                this.dataBufferUsagePlotFloat = null;

                return;
            }

            if (!this.FindDataBufferPlotFloats(this.plotModel))
            {
                CollectionChangedObservation.GetObserver(this.plotObserverOwner, this.plotModel.Floats).AddHandlerForAddItems(
                    (sender, e) => this.FindDataBufferPlotFloats(this.plotModel));
            }
        }

        private bool FindDataBufferPlotFloats(PlotSpyModel model)
        {
            if (this.dataBufferLengthPlotFloat == null)
            {
                this.dataBufferLengthPlotFloat = model.GetFloat("@nn/spy/DataBufferLength");
            }

            if (this.dataBufferUsagePlotFloat == null)
            {
                this.dataBufferUsagePlotFloat = model.GetFloat("@nn/spy/DataBufferUsage");
            }

            return this.dataBufferLengthPlotFloat != null && this.dataBufferUsagePlotFloat != null;
        }

        private void UpdateCurrentTime()
        {
            var current = this.playbackService.Current.Timestamp;

            this.DataBufferLength = this.dataBufferLengthPlotFloat?.Values.FirstOrDefault()?.Value ?? 0d;

            this.DataBufferUsage = this.dataBufferUsagePlotFloat?.FindValue(current)?.Value ?? 0d;
        }
    }
}
