﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;

namespace NintendoWare.SpySample.Windows.Controls
{
    [TemplatePart(Name = PartCanvas, Type = typeof(Canvas))]
    public class FrequencyGrid : Control
    {
        private const string PartCanvas = "PartCanvas";
        private const int MaximumGridCount = 500;

        public static readonly DependencyProperty OrientationProperty =
            DependencyProperty.Register(
                nameof(Orientation),
                typeof(Orientation),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(
                    Orientation.Horizontal,
                    (d, e) => Self(d).OnOrientationChanged(e)));

        public static readonly DependencyProperty ScaleProperty =
            DependencyProperty.Register(
                nameof(Scale),
                typeof(double),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(
                    1.0,
                    (d, e) => Self(d).OnScaleChanged(e)),
                v => IsValidFiniteDouble(v, d => d > 0));

        public static readonly DependencyProperty MaximumProperty =
            DependencyProperty.Register(
                nameof(Maximum),
                typeof(double),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(
                    100.0,
                    (d, e) => Self(d).OnMaximumChanged(e),
                    (d, v) => Self(d).CoerceMaximum((double)v)),
                IsValidFiniteDouble);

        public static readonly DependencyProperty MinimumProperty =
            DependencyProperty.Register(
                nameof(Minimum),
                typeof(double),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(
                    0.0,
                    (d, e) => Self(d).OnMinimumChanged(e)),
                IsValidFiniteDouble);

        public static readonly DependencyProperty GridFrequencyProperty =
            DependencyProperty.Register(
                nameof(GridFrequency),
                typeof(double),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(
                    20.0,
                    (d, e) => Self(d).OnGridFrequencyChanged(e)),
                v => IsValidFiniteDouble(v, d => d >= 1));

        public static readonly DependencyProperty LabelTemplateProperty =
            DependencyProperty.Register(
                nameof(LabelTemplate),
                typeof(DataTemplate),
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(null));

        private Canvas _canvas;
        private int _gridCount;
        private readonly MergedRequestDispatcher _requestDispatcher = new MergedRequestDispatcher();

        public Orientation Orientation
        {
            get { return (Orientation)this.GetValue(OrientationProperty); }
            set { this.SetValue(OrientationProperty, value); }
        }

        public double Scale
        {
            get { return (double)this.GetValue(ScaleProperty); }
            set { this.SetValue(ScaleProperty, value); }
        }

        public double Maximum
        {
            get { return (double)this.GetValue(MaximumProperty); }
            set { this.SetValue(MaximumProperty, value); }
        }

        public double Minimum
        {
            get { return (double)this.GetValue(MinimumProperty); }
            set { this.SetValue(MinimumProperty, value); }
        }

        /// <summary>
        /// 保証すべきグリッド間隔(ピクセル)です。
        /// 実際のグリッド間隔はこの値以上になります。
        /// </summary>
        public double GridFrequency
        {
            get { return (double)this.GetValue(GridFrequencyProperty); }
            set { this.SetValue(GridFrequencyProperty, value); }
        }

        public DataTemplate LabelTemplate
        {
            get { return (DataTemplate)this.GetValue(LabelTemplateProperty); }
            set { this.SetValue(LabelTemplateProperty, value); }
        }

        static FrequencyGrid()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(FrequencyGrid),
                new FrameworkPropertyMetadata(typeof(FrequencyGrid)));
        }

        public FrequencyGrid()
        {
        }

        private void OnOrientationChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGridItems();
        }

        private void OnScaleChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGridItems();
        }

        private void OnMaximumChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGridItems();
        }

        private object CoerceMaximum(double v)
        {
            return Math.Max(this.Minimum, v);
        }

        private void OnMinimumChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumProperty);

            this.RequestUpdateGridItems();
        }

        private void OnGridFrequencyChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGridItems();
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _canvas = this.GetTemplateChild(PartCanvas) as Canvas;

            if (_canvas != null)
            {
                _canvas.Children.Clear();
                _gridCount = 0;
            }
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.RequestUpdateGridItems();
        }

        private void RequestUpdateGridItems()
        {
            _requestDispatcher.Request(this.UpdateGridItems);
        }

        private void UpdateGridItems()
        {
            if (_canvas == null)
            {
                return;
            }

            var screenSize = this.Orientation == Orientation.Horizontal ? _canvas.ActualHeight : _canvas.ActualWidth;
            if (screenSize < double.Epsilon)
            {
                this.HideUnusedGridItems(0);
                return;
            }

            var center = this.Maximum - this.Minimum;
            var maximum = (this.Maximum - center) / this.Scale + center;
            var minimum = (this.Minimum - center) / this.Scale + center;

            var gridDistance = CalcGridDistance(screenSize);

            var start = GetLowerMultipleOf(minimum, gridDistance);
            var end = GetUpperMultipleOf(maximum, gridDistance);

            if (end <= start)
            {
                this.HideUnusedGridItems(0);
                return;
            }

            var toScreen = screenSize / (maximum - minimum);
            var gridCount = Math.Min(MaximumGridCount, (int)((end - start) / gridDistance) + 1);

            for (int i = 0; i < gridCount; i++)
            {
                Assertion.Operation.True(i <= _canvas.Children.Count);

                FrequencyGridItem item = null;
                if (i < _canvas.Children.Count)
                {
                    item = (FrequencyGridItem)_canvas.Children[i];
                    item.Orientation = this.Orientation;
                    item.Visibility = Visibility.Visible;
                }
                else
                {
                    item = new FrequencyGridItem()
                    {
                        Orientation = this.Orientation,
                    };
                    item.SetBinding(FrequencyGridItem.ContentTemplateProperty, new Binding(nameof(LabelTemplate)) { Source = this });
                    item.SetBinding(FrequencyGridItem.StrokeProperty, new Binding(nameof(Foreground)) { Source = this });
                    _canvas.Children.Add(item);
                }

                var gridValue = start + i * gridDistance;

                item.Content = gridValue;

                if (this.Orientation == Orientation.Horizontal)
                {
                    item.Width = _canvas.ActualWidth;
                    Canvas.SetBottom(item, (gridValue - minimum) * toScreen);
                }
                else
                {
                    item.Height = _canvas.ActualHeight;
                    Canvas.SetLeft(item, (gridValue - minimum) * toScreen);
                }
            }

            this.HideUnusedGridItems(gridCount);
        }

        private void HideUnusedGridItems(int usedGridCount)
        {
            // 未使用のグリッドのうち、数の多すぎる部分を削除します。
            var maxReservedGridCount = Math.Max(usedGridCount * 2, 100);
            while (maxReservedGridCount < _canvas.Children.Count)
            {
                _canvas.Children.RemoveAt(_canvas.Children.Count - 1);
            }

            // 未使用のグリッドを非表示にします。
            _canvas.Children
                .Cast<FrequencyGridItem>()
                .Take(_gridCount)
                .Skip(usedGridCount)
                .ForEach(it => it.Visibility = Visibility.Hidden);

            _gridCount = usedGridCount;
        }

        /// <summary>
        /// 目盛間隔を求めます。
        /// </summary>
        private double CalcGridDistance(double screenRange)
        {
            Assertion.Argument.True(screenRange > 0);

            // グリッド間隔として保証する最低の幅 (グリッド座標)
            var distance = (this.Maximum - this.Minimum) * this.GridFrequency / screenRange / this.Scale;

            // distance 以下の 10 のべき乗のうち、最も大きい値を求めます。
            var distanceBase = Math.Pow(10.0, Math.Floor(Math.Log10(distance)));

            if (distance <= distanceBase * 1)
            {
                return distanceBase * 1;
            }
            else if (distance <= distanceBase * 2)
            {
                return distanceBase * 2;
            }
            else if (distance <= distanceBase * 5)
            {
                return distanceBase * 5;
            }
            else
            {
                return distanceBase * 10;
            }
        }

        private static double GetUpperMultipleOf(double value, double round)
        {
            return Math.Ceiling(value / round) * round;
        }

        private static double GetLowerMultipleOf(double value, double round)
        {
            return Math.Floor(value / round) * round;
        }

        private static bool IsValidFiniteDouble(object v)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidFiniteDouble(object v, Func<double, bool> predicate)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d) && predicate(d);
            }
            else
            {
                return false;
            }
        }

        private static FrequencyGrid Self(DependencyObject d)
        {
            return (FrequencyGrid)d;
        }
    }
}
