﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.Spy;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public sealed class TimelineItemViewModel : ObservableObject
    {
        private readonly ObservableCollection<TimelineItemStateViewModel> _statesList = new ObservableCollection<TimelineItemStateViewModel>();

        //public delegate void AppendedStateEventHandler( TimelineItemStateViewModel state );
        //public event AppendedStateEventHandler AppendedState = null;

        private string _name = string.Empty;
        private uint _soundId;
        private uint _playerId;
        private string _playerName = string.Empty;
        private bool _isActive;
        private SpyTime _startTime = SpyTime.Zero;
        private SpyTime _endTime = SpyTime.Zero;

        /// <summary>
        /// アイテムの名前
        /// </summary>
        public string Name
        {
            get { return _name; }
            set
            {
                this.SetPropertyValue(ref _name, value);
            }
        }

        /// <summary>
        /// サウンド（アイテム）ID
        /// </summary>
        public uint SoundId
        {
            get { return _soundId; }
            set
            {
                this.SetPropertyValue(ref _soundId, value);
            }
        }

        /// <summary>
        /// 管理上のインスタンスID
        /// </summary>
        public uint InstanceId { get; set; }

        /// <summary>
        /// プレイヤーID
        /// </summary>
        public uint PlayerId
        {
            get { return _playerId; }
            set
            {
                this.SetPropertyValue(ref _playerId, value);
            }
        }

        public string PlayerName
        {
            get { return _playerName; }
            set
            {
                this.SetPropertyValue(ref _playerName, value);
            }
        }

        /// <summary>
        /// まだ生きてるか
        /// </summary>
        public bool IsActive
        {
            get { return _isActive; }
            set
            {
                if (_isActive != value)
                {
                    _isActive = value;
                    NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// 各ステータス
        /// </summary>
        public IList<TimelineItemStateViewModel> States
        {
            get { return _statesList; }
        }

        public ObservableCollection<TimelineItemStateViewModel> States_
        {
            get { return _statesList; }
        }

        /// <summary>
        /// 現在アクティブなステート
        /// </summary>
        public TimelineItemStateViewModel ActiveState
        {
            get
            {
                if (IsActive)
                {
                    return States[States.Count - 1];
                }
                return null;
            }
        }

        /// <summary>
        /// 開始時刻
        /// </summary>
        public SpyTime StartTime
        {
            get { return _startTime; }
            private set
            {
                this.SetPropertyValue(ref _startTime, value);
            }
        }

        /// <summary>
        /// 終了時刻
        /// </summary>
        public SpyTime EndTime
        {
            get { return _endTime; }
            private set
            {
                this.SetPropertyValue(ref _endTime, value);
            }
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public TimelineItemViewModel()
        {
            IsActive = false;
        }

        protected override void DisposeManagedInstance()
        {
            _statesList.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        /// <summary>
        /// アクティブなステートを更新する
        /// </summary>
        /// <param name="time"></param>
        public void Update(SpyTime time)
        {
            ActiveState.Update(time);
            this.EndTime = time;
        }

        /// <summary>
        /// ステートを切り替える。
        /// これまでアクティブだったステートを終了させて追加。
        /// </summary>
        /// <param name="state"></param>
        public void AppendState(TimelineItemStateViewModel state)
        {
            if (States.Count == 0)
            {
                this.StartTime = state.StartTime;
            }
            else
            {
                ActiveState.Close(state.StartTime);
            }

            States.Add(state);
            IsActive = true;

            this.EndTime = state.EndTime;
        }

        /// <summary>
        /// このアイテムを終了させる
        /// </summary>
        /// <param name="time"></param>
        public void Close(SpyTime time)
        {
            ActiveState.Close(time);
            IsActive = false;

            this.EndTime = time;
        }
    }

    /// <summary>
    /// サウンドのステータスのビューモデル
    /// </summary>
    public class TimelineItemStateViewModel : ObservableObject
    {
        /// <summary>
        /// ステートタイプ
        /// </summary>
        public enum StateType
        {
            Preapre = 0,
            Playing,
            Pause
        }

        private SpyTime _startTime = SpyTime.InvalidValue;
        private SpyTime _endTime = SpyTime.InvalidValue;

        //public event EventHandler OnEndFrameChanged = null;

        /// <summary>
        /// 親インスタンス
        /// </summary>
        public TimelineItemViewModel Parent { get; private set; }

        /// <summary>
        /// サウンドアイテムの状況
        /// </summary>
        public StateType State { get; private set; }

        /// <summary>
        /// イベント開始時刻
        /// </summary>
        public SpyTime StartTime
        {
            get { return _startTime; }
            private set
            {
                this.SetPropertyValue(ref _startTime, value);
            }
        }

        /// <summary>
        /// イベント終了時刻
        /// </summary>
        public SpyTime EndTime
        {
            get { return _endTime; }
            private set
            {
                this.SetPropertyValue(ref _endTime, value);
            }
        }

        /// <summary>
        /// アクティブかどうか
        /// </summary>
        public bool IsActive { get; private set; }

        /// <summary>
        /// コンストラクタ。イベント開始
        /// </summary>
        public TimelineItemStateViewModel(TimelineItemViewModel parent, SoundStatusInfoSpyModel.SoundInfo info)
        {
            Parent = parent;
            State = GetStateType(info);
            StartTime = info.BelongingFrame;
            EndTime = info.BelongingFrame;
        }

        /// <summary>
        /// ステートの更新。
        /// </summary>
        /// <param name="time"></param>
        public void Update(SpyTime time)
        {
            IsActive = true;
            EndTime = time;
        }

        /// <summary>
        /// このステートが終了するときに呼んでもらう
        /// </summary>
        /// <param name="time"></param>
        public void Close(SpyTime time)
        {
            IsActive = false;
            EndTime = time;
        }

        /// <summary>
        /// ステートタイプを取得
        /// </summary>
        /// <param name="info"></param>
        /// <returns></returns>
        public static StateType GetStateType(SoundStatusInfoSpyModel.SoundInfo info)
        {
            if (info.StatusBitFlag == SoundStatusInfoSpyModel.SoundInfo.StatusPrepared)
            {
                return StateType.Preapre; // プリペア中だけ。
            }
            else if ((info.StatusBitFlag & SoundStatusInfoSpyModel.SoundInfo.StatusPause) > 0)
            {
                return StateType.Pause; // ポーズがかかってたらこれを優先
            }
            else
            {
                return StateType.Playing; // それ以外で生きてたらすべて再生中とみなす
            }
        }
    }
}
