﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.Spy;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows.Data;
using System.Windows.Markup;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    [ContentProperty("Items")]
    public class StartSoundHistoryPanelViewModel : ObservableObject
    {
        private readonly object _lockObject = new object();
        private SoundStatusInfoSpyModel _soundStatusInfo;
        private SoundDataInfoSpyModel _soundDataInfo;
        private readonly ObservableCollection<StartSoundHistoryItemViewModel> _items = new ObservableCollection<StartSoundHistoryItemViewModel>();
        private readonly Dictionary<uint, StartSoundHistoryItemViewModel> _activeItems = new Dictionary<uint, StartSoundHistoryItemViewModel>();
        private SpyGlobalTime _lastTimestamp;
        private ICollectionView _viewItems;
        private StartSoundHistoryItemViewModel _selectedItem;
        private readonly StringFilter _nameFilter = new StringFilter();
        private string _nameFilterString = string.Empty;
        private SpyTimeUnit _timeUnit;
        private bool _showColumnPlayerName = true;
        private bool _showColumnVolume = true;
        private bool _showColumnPriority = true;
        private bool _showColumnSoundType = true;

        public StartSoundHistoryPanelViewModel()
        {
        }

        protected override void DisposeManagedInstance()
        {
            _items.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public SoundStatusInfoSpyModel SoundStatusInfo
        {
            get { return _soundStatusInfo; }
        }

        public SoundDataInfoSpyModel SoundDataInfo
        {
            get { return _soundDataInfo; }
        }

        public ObservableCollection<StartSoundHistoryItemViewModel> Items
        {
            get
            {
                return _items;
            }
        }

        public ICollectionView ViewItems
        {
            get
            {
                if (_viewItems == null)
                {
                    lock (_lockObject)
                    {
                        if (_viewItems == null)
                        {
                            this.SetupViewItems(this.Items);
                        }
                    }
                }

                return _viewItems;
            }
        }

        public StartSoundHistoryItemViewModel SelectedItem
        {
            get
            {
                return _selectedItem;
            }
            set
            {
                this.SetPropertyValue(ref _selectedItem, value);
            }
        }

        public string NameFilter
        {
            get
            {
                return _nameFilterString;
            }
            set
            {
                if (this.SetPropertyValue(ref _nameFilterString, value))
                {
                    _nameFilter.Filter = _nameFilterString;
                    this.UpdateItemsVisibility();
                }
            }
        }

        public SpyTimeUnit TimeUnit
        {
            get
            {
                return _timeUnit;
            }

            set
            {
                this.SetPropertyValue(ref _timeUnit, value);
            }
        }

        public bool ShowColumnPlayerName
        {
            get
            {
                return _showColumnPlayerName;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPlayerName, value);
            }
        }

        public bool ShowColumnVolume
        {
            get
            {
                return _showColumnVolume;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnVolume, value);
            }
        }

        public bool ShowColumnPriority
        {
            get
            {
                return _showColumnPriority;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPriority, value);
            }
        }

        public bool ShowColumnSoundType
        {
            get
            {
                return _showColumnSoundType;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnSoundType, value);
            }
        }

        private void UpdateItemsVisibility()
        {
            foreach (var item in this.Items)
            {
                item.IsVisible = this.ItemFilter(item);
            }
        }

        private bool ItemFilter(StartSoundHistoryItemViewModel item)
        {
            bool match = true;

            // グループ間はAND, グループ内はOR

            if (match == true)
            {
                // 名前フィルタ
                match = false;

                if (!match && _nameFilter.Execute(item.SoundName))
                {
                    match = true;
                }

                if (!match && _nameFilter.Execute(item.PlayerName))
                {
                    match = true;
                }
            }

#if false // フィルタできるプロパティが追加されたときのためのサンプルコード
            if (match == true)
            {
                // 種別グループ
                match = false;

                if (!match && _showOneshot && item.PlayKind == "Oneshot")
                {
                    match = true;
                }

                if (!match && _showLoop && item.PlayKind == "Loop")
                {
                    match = true;
                }
            }
#endif

            return match;
        }

        /// <summary>
        /// モデルをプロパティと関連付けます。
        /// </summary>
        internal void AttachModel(SoundDataInfoSpyModel soundDataInfo)
        {
            _soundDataInfo = soundDataInfo;

            foreach (var item in Items)
            {
                UpdateItemInfo(item);
                item.IsVisible = ItemFilter(item);
            }
        }

        /// <summary>
        /// モデルをプロパティと関連付けます。
        /// </summary>
        internal void AttachModel(SoundStatusInfoSpyModel soundStatusInfo)
        {
            if (_soundStatusInfo != null)
            {
                _soundStatusInfo.SoundInfoChanged -= HandleSoundStatusInfoChanged;
            }

            _soundStatusInfo = soundStatusInfo;
            this.Clear();

            if (_soundStatusInfo != null)
            {
                _soundStatusInfo.SoundInfoChanged += HandleSoundStatusInfoChanged;
                _soundStatusInfo.ReadAllSoundStatusInfos();
            }
        }

        private void Clear()
        {
            _items.ForEach(it => it.Dispose());
            _items.Clear();
            _activeItems.Clear();
            _lastTimestamp = SpyGlobalTime.Zero;
        }

        private void HandleSoundStatusInfoChanged(object sender, SoundStatusInfoSpyModel.SoundInfoChangedEventArgs args)
        {
            if (args.Timestamp <= _lastTimestamp)
            {
                this.Clear();
            }

            _lastTimestamp = args.Timestamp;

            // infosに新しく現れたアイテムは再生を開始したことを表します。
            // infosから消えたアイテムは停止したことを表します。

            var stoppedSounds = _activeItems.Where(item => args.SoundInfos.FirstOrDefault(info => info.InstanceId == item.Key) == null).ToArray();
            foreach (var item in stoppedSounds)
            {
                _activeItems.Remove(item.Key);
            }

            var newSounds = args.SoundInfos.Where(info => !_activeItems.ContainsKey(info.InstanceId));
            foreach (var info in newSounds)
            {
                var item = new StartSoundHistoryItemViewModel()
                {
                    SoundId = info.SoundId,
                    Time = new SpyTime(args.Timestamp, args.BelongingFrame),
                };

                UpdateItemInfo(item);

                item.IsVisible = this.ItemFilter(item);

                _activeItems.Add(info.InstanceId, item);
                this.Items.Add(item);
            }
        }

        private void UpdateItemInfo(StartSoundHistoryItemViewModel item)
        {
            var soundId = item.SoundId;

            var soundInfo = _soundDataInfo != null ? _soundDataInfo.GetSoundInfo(soundId) : null;
            if (soundInfo != null)
            {
                item.SoundName = soundInfo.Label;
                item.Volume = soundInfo.Volume;
                item.Priority = soundInfo.PlayerPriority;
                item.SoundType = soundInfo.SoundType.ToString();
            }
            else
            {
                item.SoundName = soundId.ToString();
            }

            var playerInfo = soundInfo != null ? _soundDataInfo.GetPlayerInfo(soundInfo.PlayerId) : null;
            if (playerInfo != null)
            {
                item.PlayerName = playerInfo.Label;
            }
        }

        /// <summary>
        /// Itemsプロパティにグループおよびデフォルトのソートを設定します。
        /// </summary>
        /// <param name="source"></param>
        private void SetupViewItems(object source)
        {
            var viewItems = CollectionViewSource.GetDefaultView(source);

            if (viewItems != null)
            {
                viewItems.SortDescriptions.Add(new SortDescription("Time.Timestamp", ListSortDirection.Descending));
                viewItems.GroupDescriptions.Add(new PropertyGroupDescription("Time.Timestamp"));
            }

            _viewItems = viewItems;
        }
    }
}
