﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.NwSoundSpyPlugin.Plugins;
using NintendoWare.Spy;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using System.Windows.Controls;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    internal class StartSoundHistoryPanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private StartSoundHistoryPanel _view;
        private StartSoundHistoryPanelViewModel _viewModel;
        private SpyTimeUnit _timeUnit;

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => this.UpdateGlobalTimelineCurrent());

            this.UpdateGlobalTimelineCurrent();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        protected override Control CreateContent()
        {
            _view = new StartSoundHistoryPanel();
            _viewModel = new StartSoundHistoryPanelViewModel();
            _view.DataContext = _viewModel;

            this.ApplySettings(null);

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.SelectedItem,
                    (target, args) =>
                    {
                        if (target.SelectedItem != null)
                        {
                            if (!this.GetPlaybackService().ShouldAutoScroll)
                            {
                                this.GetPlaybackService().SetCurrentTime(target.SelectedItem.Time.Timestamp);
                            }
                        }
                    });

            return _view;
        }

        private void UpdateGlobalTimelineCurrent()
        {
            var current = GetPlaybackService().Current;

            if (_viewModel == null || _viewModel.Items == null || _viewModel.Items.Count == 0)
            {
                return;
            }

            var comparer = new SpyTimeTimeUnitComparer(_timeUnit);

            using (var enumerator = _viewModel.Items.GetEnumerator())
            {
                StartSoundHistoryItemViewModel follower = null;
                if (enumerator.MoveNext())
                {
                    follower = enumerator.Current;
                }

                foreach (StartSoundHistoryItemViewModel item in _viewModel.Items)
                {
                    // follower は item よりも Time の大きい最初の要素です。
                    // 末尾に達すると null になります。
                    if (follower != null)
                    {
                        while (comparer.Compare(follower.Time, item.Time) <= 0)
                        {
                            if (enumerator.MoveNext())
                            {
                                follower = enumerator.Current;
                            }
                            else
                            {
                                follower = null;
                                break;
                            }
                        }
                    }

                    if (follower != null)
                    {
                        item.IsCurrent = comparer.Compare(item.Time, current) <= 0 && comparer.Compare(current, follower.Time) < 0;
                    }
                    else
                    {
                        item.IsCurrent = comparer.Compare(item.Time, current) <= 0;
                    }
                }
            }

            // ハイライト状態の最初の要素が表示されるようにスクロールさせます。
            // ビューはソートされているので最初の要素はViewItemsで判断します。
            if (_view != null && _viewModel.ViewItems != null)
            {
                foreach (StartSoundHistoryItemViewModel item in _viewModel.ViewItems)
                {
                    if (item.IsCurrent)
                    {
                        _view.List.ScrollIntoView(item);
                        break;
                    }
                }
            }
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            switch (dataName)
            {
                case SoundDataInfoSpyModelPlugin.SpyDataName:
                    if (_viewModel.SoundDataInfo != model)
                    {
                        _viewModel.AttachModel(model as SoundDataInfoSpyModel);
                    }
                    break;

                case SoundStatusInfoSpyModelPlugin.SpyDataName:
                    if (_viewModel.SoundStatusInfo != model)
                    {
                        _viewModel.AttachModel(model as SoundStatusInfoSpyModel);
                        this.UpdateGlobalTimelineCurrent();
                    }
                    break;
            }
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (settings != null)
            {
                _timeUnit = settings.TimeUnit;
            }

            if (_viewModel != null)
            {
                _viewModel.TimeUnit = _timeUnit;
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => ApplySettings(target));

            base.OnSettingsApplied(service);
        }
    }
}
