﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    /// <summary>
    /// サンプルの区間 ( Left <= x <= Right ) を表します。
    /// </summary>
    public struct SampleRange
    {
        /// <summary>
        /// 区間の左端です。
        /// </summary>
        public long Left;

        /// <summary>
        /// 区間の右端です。
        /// </summary>
        public long Right;

        /// <summary>
        /// 有効な区間であれば true を返します。
        /// </summary>
        public bool IsValid { get { return this.Left <= this.Right; } }

        /// <summary>
        /// 無効な区間を表します。
        /// </summary>
        public static SampleRange Invalid = new SampleRange(long.MaxValue, long.MinValue);

        /// <summary>
        /// 指定した区間に初期化します。
        /// </summary>
        public SampleRange(long left, long right)
        {
            this.Left = left;
            this.Right = right;
        }

        /// <summary>
        /// rangeをコピーして初期化します。
        /// </summary>
        public SampleRange(SampleRange range)
        {
            this.Left = range.Left;
            this.Right = range.Right;
        }

        /// <summary>
        /// 区間が重なる、あるいは隣接している場合に true を返します。
        /// </summary>
        public bool IsContinuous(SampleRange range)
        {
            var isect = this.GetIntersection(range);
            return isect.Left <= isect.Right + 1;
        }

        /// <summary>
        /// 現在の区間と指定した区間の両方を包含する最少の区間を求めます。
        /// </summary>
        public void Join(SampleRange range)
        {
            if (!range.IsValid)
            {
            }
            else if (!this.IsValid)
            {
                this = range;
            }
            else
            {
                this.Left = Math.Min(this.Left, range.Left);
                this.Right = Math.Max(this.Right, range.Right);
            }
        }

        /// <summary>
        /// range と重なった区間を返します。
        /// 区間が重ならない場合は戻り値の IsValid プロパティが false になります。
        /// </summary>
        public SampleRange GetIntersection(SampleRange range)
        {
            var left = Math.Max(this.Left, range.Left);
            var right = Math.Min(this.Right, range.Right);
            return new SampleRange(left, right);
        }

        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }

            var value = (SampleRange)obj;
            return this.Left == value.Left && this.Right == value.Right;
        }

        public override int GetHashCode()
        {
            return this.Left.GetHashCode() ^ this.Right.GetHashCode();
        }
    }

    /// <summary>
    /// サンプル区間のコレクションです。
    /// </summary>
    public class SampleRangeCollection : ICollection<SampleRange>
    {
        private readonly List<SampleRange> _collection = new List<SampleRange>();

        /// <summary>
        /// 区間を加え、可能であれば連結します。
        /// </summary>
        public void AddRange(long left, long right)
        {
            this.AddRange(new SampleRange(left, right));
        }

        /// <summary>
        /// 区間を加え、可能であれば連結します。
        /// </summary>
        public void AddRange(SampleRange range)
        {
            List<SampleRange> intersectItems = new List<SampleRange>();

            foreach (var item in _collection)
            {
                if (item.IsContinuous(range))
                {
                    intersectItems.Add(item);
                }
            }

            foreach (var item in intersectItems)
            {
                _collection.Remove(item);
                range.Join(item);
            }

            _collection.Add(range);
        }

        /// <summary>
        /// コレクションに含まれる区間を加え、可能であれば連結します。
        /// </summary>
        public void AddRange(SampleRangeCollection collection)
        {
            foreach (var item in collection)
            {
                this.AddRange(item);
            }
        }

        /// <summary>
        /// コレクションに含まれる区間からrangeと重なる部分を削除します。
        /// </summary>
        public bool RemoveRange(SampleRange range)
        {
            if (!range.IsValid)
            {
                return false;
            }

            var addItems = new List<SampleRange>();
            var removeItems = new List<SampleRange>();

            var retval = false;
            foreach (var item in _collection)
            {
                var intersect = item.GetIntersection(range);
                if (intersect.IsValid)
                {
                    if (item.Left < intersect.Left)
                    {
                        addItems.Add(new SampleRange(item.Left, intersect.Left));
                    }

                    if (intersect.Right < item.Right)
                    {
                        addItems.Add(new SampleRange(intersect.Right, item.Right));
                    }

                    retval = true;
                    removeItems.Add(item);
                }
            }

            foreach (var item in removeItems)
            {
                _collection.Remove(range);
            }

            foreach (var item in addItems)
            {
                _collection.Add(range);
            }

            return retval;
        }

        /// <summary>
        /// rangeと重なる部分の区間のみをコレクションに残します。
        /// </summary>
        public void LimitRange(SampleRange range)
        {
            if (!range.IsValid)
            {
                _collection.Clear();
                return;
            }

            var newItems = new List<SampleRange>();

            foreach (var item in _collection)
            {
                var intersection = item.GetIntersection(range);
                if (intersection.IsValid)
                {
                    newItems.Add(intersection);
                }
            }

            _collection.Clear();
            foreach (var item in newItems)
            {
                _collection.Add(item);
            }
        }

        /// <summary>
        /// コレクションに"含まれない"区間のうち、rangeと重なる部分のコレクションを返します。
        /// </summary>
        public SampleRangeCollection InvertRange(SampleRange range)
        {
            var result = new SampleRangeCollection();

            var left = range.Left;
            foreach (var item in _collection.OrderBy(e => e.Left))
            {
                if (range.Right < item.Left)
                {
                    break;
                }

                if (left < item.Left)
                {
                    result.Add(new SampleRange(left, item.Left - 1));
                    left = item.Right + 1;
                }
                else if (left <= item.Right)
                {
                    left = item.Right + 1;
                }
            }

            if (left <= range.Right)
            {
                result.Add(new SampleRange(left, range.Right));
            }

            return result;
        }

        public void Add(SampleRange range)
        {
            _collection.Add(range);
        }

        public bool Remove(SampleRange item)
        {
            return _collection.Remove(item);
        }

        public void Clear()
        {
            _collection.Clear();
        }

        public bool Contains(SampleRange item)
        {
            return _collection.Contains(item);
        }

        public void CopyTo(SampleRange[] array, int arrayIndex)
        {
            _collection.CopyTo(array, arrayIndex);
        }

        public int Count
        {
            get { return _collection.Count; }
        }

        public bool IsReadOnly
        {
            get { return false; }
        }

        public IEnumerator<SampleRange> GetEnumerator()
        {
            return _collection.GetEnumerator();
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return _collection.GetEnumerator();
        }
    }
}
