﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Threading;
using System.Threading.Tasks;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    /// <summary>
    /// キャンセル可能なバックグラウンドタスクの発行を制御します。
    /// ・新しいタスクを発行するときには、前のタスクはキャンセルされます。
    /// ・新しいタスクは、前のタスクの完了を待ってから、実行を開始します。
    /// ・新しいタスクは前のタスクの実行結果を受け取ることができます。
    /// </summary>
    /// <typeparam name="T">タスクの実行結果です。</typeparam>
    public class TaskController<T>
        where T : class // null許容型
    {
        private CancellationTokenSource _cts;
        private Task<T> _task;
        private readonly object _syncObject = new object();

        public delegate T BackgroundTask(CancellationToken ct, T lastResult);

        /// <summary>
        /// 新しいタスクを発行します。
        /// </summary>
        public void Request(BackgroundTask func)
        {
            lock (_syncObject)
            {
                if (_cts != null)
                {
                    _cts.Cancel();
                }

                _cts = new CancellationTokenSource();
                var ct = _cts.Token;

                if (_task != null)
                {
                    // タスクが実行を開始する前はキャンセルできないように
                    // ContinuwWith()の引数にctは与えない。
                    _task = _task.ContinueWith<T>((t) =>
                        {
                            if (t.IsCanceled)
                            {
                                return func(ct, null);
                            }
                            else
                            {
                                return func(ct, t.Result);
                            }
                        });
                }
                else
                {
                    _task = Task.Factory.StartNew<T>(() =>
                        {
                            return func(ct, null);
                        }, ct);
                }
            }
        }

        /// <summary>
        /// 実行中のタスクをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            lock (_syncObject)
            {
                if (_cts != null)
                {
                    _cts.Cancel();
                }

                _cts = null;
                _task = null;
            }
        }

        /// <summary>
        /// 定期的に呼び出します。
        /// 実行を完了したタスクへの参照が消去されます。
        /// </summary>
        public void Polling()
        {
            lock (_syncObject)
            {
                this.CheckCompletion();
            }
        }

        private void CheckCompletion()
        {
            if (_task != null)
            {
                if (_task.Exception != null)
                {
                    this.CheckException(_task.Exception);
                    _task = null;
                }
                else if (_task.IsCompleted)
                {
                    _task = null;
                }
            }
        }

        private void CheckException(Exception exception)
        {
            var e = exception;

            while (e != null)
            {
                if (e is AggregateException)
                {
                    e = e.InnerException;
                    continue;
                }

                if (e is TaskCanceledException)
                {
                    return;
                }

                throw new ApplicationException("Exception occured in background job.", exception);
            }
        }
    }
}
