﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using System;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public sealed class FinalOutPanelPresenter : FormsSpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private FinalOutControl _view;
        private FinalOutSpyModel _model;
        private int _grabToken = SpyPlaybackService.TokenGrabInvalid;

        //-----------------------------------------------------------------

        protected override Control CreateControl()
        {
            Assertion.Operation.Null(_view);

            _view = new FinalOutControl();
            _view.CurrentDraggedEvent += CurrentDraggedEvent;
            _view.GetFinalOut += this.GetFinalOut;

            return _view;
        }

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Begin,
                    (target, args) => this.UpdateFrameRange());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.End,
                    (target, args) => this.UpdateFrameRange());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => this.UpdateCurrentFrame());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.ShouldAutoScroll,
                    (target, args) => this.UpdatePauseFlag());

            this.UpdateFrameRange();
            this.UpdateCurrentFrame();
            this.UpdatePauseFlag();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            _view = null;

            base.OnUninitialize();
        }

        private void UpdateFrameRange()
        {
            _view.SetAudioFrameRange(this.GetPlaybackService().Begin.AudioFrame, this.GetPlaybackService().End.AudioFrame);
        }

        private void UpdateCurrentFrame()
        {
            _view.ScrollToAudioFramePos(this.GetPlaybackService().Current.AudioFrame);
        }

        private void UpdatePauseFlag()
        {
            _view.PauseFlag = !this.GetPlaybackService().ShouldAutoScroll;
        }

        private void UpdateSamples(object sender, EventArgs e)
        {
            if (_view == null)
            {
                return;
            }

            _view.UpdateSamples();
        }

        private void CurrentDraggedEvent(object sender, DragMoveEventArgs e)
        {
            switch (e.Action)
            {
                case DragMoveAction.Start:
                    _grabToken = this.GetPlaybackService().GetGrabForCurrent();
                    break;

                case DragMoveAction.Update:
                    long frame = _view.GetCurrentFrame();
                    if (frame >= 0)
                    {
                        this.GetPlaybackService().SetCurrentAudioFrame(new Frame(frame), _grabToken);
                    }
                    break;

                case DragMoveAction.End:
                    this.GetPlaybackService().ReleaseGrabForCurrent(_grabToken);
                    _grabToken = SpyPlaybackService.TokenGrabInvalid;
                    break;
            }
        }

        private void UpdateEventHandler(object sender, EventArgs e)
        {
            _view.Model = _model;
        }

        private static int ConvertWaveVisibilityToInt(WaveVisibility wv)
        {
            int channels = 0;
            switch (wv)
            {
                case WaveVisibility.None:
                    channels = 0;
                    break;
                case WaveVisibility.Stereo:
                    channels = 2;
                    break;
                case WaveVisibility.Surround4ch:
                    channels = 4;
                    break;
                case WaveVisibility.Surround6ch:
                    channels = 6;
                    break;
            }
            return channels;
        }

        /// <summary>
        /// 設定が更新されると実行されます。
        /// </summary>
        /// <param name="service">SettingsService を指定します。</param>
        protected override void OnSettingsApplied(SettingsService service)
        {
            var settings = service.GetCafeAudioSettings();

            bool changeFlag = false;

            int channels = ConvertWaveVisibilityToInt(settings.FinalOutTvWaveVisibility);
            if (channels != _view.DrawChannelsTv)
            {
                _view.DrawChannelsTv = channels;
                changeFlag = true;
            }
            channels = ConvertWaveVisibilityToInt(settings.FinalOutDrcWaveVisibility);
            if (channels != _view.DrawChannelsDrc)
            {
                _view.DrawChannelsDrc = channels;
                changeFlag = true;
            }
            if (changeFlag)
            {
                _view.Invalidate();
            }
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            if (object.ReferenceEquals(_model, model))
            {
                return;
            }

            if (_model != null)
            {
                _model.UpdateEvent -= this.UpdateEventHandler;
            }

            _model = model as FinalOutSpyModel;

            if (_model == null)
            {
                _view.Reset();
                return;
            }

            _model.UpdateEvent += this.UpdateEventHandler;

            _view.Model = _model;
            _view.UpdateSamples();
            _view.ScrollToAudioFramePos(this.GetPlaybackService().Current.AudioFrame);
        }

        private FinalOutSpyModel.FinalOut GetFinalOut(Frame audioFrame)
        {
            if (_model != null)
            {
                return _model.GetFinalOut(audioFrame);
            }
            return null;
        }
    }
}
