﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;

namespace NintendoWare.NwSoundSpyPlugin.Models
{
    /// <summary>
    /// NwSnd サウンドデータ Spy モデルです。
    /// </summary>
    public sealed class SoundDataInfoSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct SoundDataInfoPacket {
        ///     struct SoundInfo {
        ///         u32 soundId;
        ///         u32 playerId;
        ///         u8 soundType;
        ///         u8 volume;
        ///         u8 playerPriority;
        ///         u8 padding1[1];
        ///         PString label; // null文字を含みます。
        ///     };
        ///
        ///     struct PlayerInfo {
        ///         u32 playerId;
        ///         PString label; // null文字を含みます。
        ///     };
        ///
        ///     u32 itemType;
        ///     u32 itemCount;
        ///     u8 itemArea[]; // SoundInfo または PlayerInfo を４バイト境界で切り詰めて格納します。
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        public class SoundInfo
        {
            public uint PlayerId { get; set; }
            public SoundType SoundType { get; set; }
            /// <summary>
            /// 0 ~ 255。127が1.0倍。
            /// </summary>
            public uint Volume { get; set; }
            /// <summary>
            /// 0 ~ 127
            /// </summary>
            public uint PlayerPriority { get; set; }
            public string Label { get; set; }
        }

        public class PlayerInfo
        {
            public string Label { get; set; }
        }

        private readonly Dictionary<uint, SoundInfo> _soundInfoTable = new Dictionary<uint, SoundInfo>();
        private readonly Dictionary<uint, PlayerInfo> _playerInfoTable = new Dictionary<uint, PlayerInfo>();

        private enum ItemType
        {
            Sound = 0,
            Player = 1
        }

        public enum SoundType
        {
            Sequence = 0,
            Wave = 1,
            Stream = 2
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                return;
            }

            var reader = CreateDataReader(dataBlock);

            ItemType itemType = (ItemType)reader.ReadUInt32();
            if (itemType == ItemType.Sound)
            {
                OnPushDataSound(dataBlock, reader);
            }
            else if (itemType == ItemType.Player)
            {
                OnPushDataPlayer(dataBlock, reader);
            }
        }

        private void OnPushDataSound(SpyDataBlock dataBlock, BinaryReader reader)
        {
            uint soundCount = reader.ReadUInt32();

            for (int i = 0; i < soundCount; i++)
            {
                SoundInfo soundInfo = new SoundInfo();

                uint soundId = reader.ReadUInt32();
                soundInfo.PlayerId = reader.ReadUInt32();
                soundInfo.SoundType = (SoundType)reader.ReadByte();
                soundInfo.Volume = reader.ReadByte();
                soundInfo.PlayerPriority = reader.ReadByte();
                reader.ReadByte(); // padding1
                byte labelLength = reader.ReadByte();
                string label = new string(reader.ReadChars(labelLength));

                int validSize = 1 + labelLength; // byte + char[labelLength]
                int paddingSize = (~(validSize + 0x03)) & 0x03; // 4バイトアライメントの補数
                if (paddingSize > 0)
                {
                    reader.ReadBytes(paddingSize);
                }
                soundInfo.Label = label;

                _soundInfoTable[soundId] = soundInfo;
            }
        }

        private void OnPushDataPlayer(SpyDataBlock dataBlock, BinaryReader reader)
        {
            uint itemCount = reader.ReadUInt32();

            for (int i = 0; i < itemCount; i++)
            {
                PlayerInfo info = new PlayerInfo();

                uint playerId = reader.ReadUInt32();
                byte labelLength = reader.ReadByte();
                string label = new string(reader.ReadChars(labelLength));

                int validSize = 1 + labelLength; // byte + char[labelLength]
                int paddingSize = (~(validSize + 0x03)) & 0x03; // 4バイトアライメントの補数
                if (paddingSize > 0)
                {
                    reader.ReadBytes(paddingSize);
                }
                info.Label = label;

                _playerInfoTable[playerId] = info;
            }
        }

        public SoundInfo GetSoundInfo(uint soundId)
        {
            SoundInfo info;
            if (_soundInfoTable.TryGetValue(soundId, out info))
            {
                return info;
            }
            else
            {
                return null;
            }
        }

        public PlayerInfo GetPlayerInfo(uint playerId)
        {
            PlayerInfo info;
            if (_playerInfoTable.TryGetValue(playerId, out info))
            {
                return info;
            }
            else
            {
                return null;
            }
        }

        public string GetLabel(uint soundId)
        {
            try
            {
                return _soundInfoTable[soundId].Label;
            }
            catch (KeyNotFoundException)
            {
                return soundId.ToString();
            }
        }

        public uint GetPlayerId(uint soundId)
        {
            try
            {
                return _soundInfoTable[soundId].PlayerId;
            }
            catch (KeyNotFoundException)
            {
                return 0;
            }
        }

        public string GetPlayerName(uint playerId)
        {
            try
            {
                return _playerInfoTable[playerId].Label;
            }
            catch (KeyNotFoundException)
            {
                return string.Empty;
            }
        }
    }
}
