﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;

namespace NintendoWare.NnAtkSpyPlugin
{
    /// <summary>
    /// Atk の Spy モデルです。
    /// </summary>
    public sealed class AtkSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// // 共通ヘッダ
        /// struct PacketHeader {
        ///     u8 type;
        ///     u8 padding[3];
        /// }
        ///
        /// // 現在の状態
        /// struct SoundStatePacket {
        ///     struct Item {
        ///         u32 instanceId;
        ///         u32 soundId;
        ///         u8 statusBitFlag;
        ///         u8 padding[3];
        ///     }
        ///
        ///     PacketHeader header;
        ///     u32 itemCount;
        ///     Item items[itemCount];
        /// }
        ///
        /// // サウンドの静的データ
        /// struct SoundDataPacket {
        ///     struct Item {
        ///         u32 soundId;
        ///         u32 playerId;
        ///         u8 soundType;
        ///         u8 volume;
        ///         u8 playerPriority;
        ///         u8 padding[1];
        ///         string(u8, 255) label;
        ///     }
        ///
        ///     PacketHeader header
        ///     u32 itemCount;
        ///     u8 itemArea[]; // パックされた Item のリスト
        /// }
        ///
        /// // プレーヤーの静的データ
        /// struct PlayerDataPacket {
        ///     struct Item {
        ///         u32 playerId;
        ///         string(u8, 255) label;
        ///     }
        ///
        ///     PacketHeader header;
        ///     u32 itemCount;
        ///     u8 itemArea[]; // パックされた Item のリスト
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// Version 0.2.0.0
        /// <list type="bullet">
        /// <item>追加サウンドアーカイブ対応。</item>
        /// </list>
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// // 共通ヘッダ
        /// struct PacketHeader {
        ///     u8 type;
        ///     u8 padding[3];
        /// }
        ///
        /// // 現在の状態
        /// struct SoundStatePacket {
        ///     struct Item {
        ///         u32 instanceId;
        ///         u32 soundId;
        ///         u8 statusBitFlag;
        ///         u8 padding[3];
        ///+        u64 archiveInstanceId; // サウンドアーカイブのインスタンス Id
        ///     }
        ///
        ///     PacketHeader header;
        ///     u32 itemCount;
        ///     Item items[itemCount];
        /// }
        ///
        /// // サウンドの静的データ
        /// struct SoundDataPacket {
        ///     struct Item {
        ///         u32 soundId;
        ///         u32 playerId;
        ///         u8 soundType;
        ///         u8 volume;
        ///         u8 playerPriority;
        ///         u8 padding[1];
        ///         string(u8, 255) label;
        ///     }
        ///
        ///     PacketHeader header;
        ///+    u64 archiveInstanceId; // サウンドアーカイブのインスタンス Id
        ///     u32 itemCount;
        ///     u8 itemArea[]; // パックされた Item のリスト
        /// }
        ///
        /// // プレーヤーの静的データ
        /// struct PlayerDataPacket {
        ///     struct Item {
        ///         u32 playerId;
        ///         string(u8, 255) label;
        ///     }
        ///
        ///     PacketHeader header;
        ///     u32 itemCount;
        ///     u8 itemArea[]; // パックされた Item のリスト
        /// }
        ///
        ///+// 追加サウンドアーカイブの静的データ
        ///+struct AddonSoundArchiveDataPacket {
        ///+     PacketHeader header;
        ///+     u64 instanceId;
        ///+     string(u8, 67) name;
        ///+}
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_2_0_0 = new Version(0, 2, 0, 0);

        /// <summary>
        /// Version 0.2.1.0
        /// <list type="bullet">
        /// <item>シーケンス変数対応。</item>
        /// </list>
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// // グローバル変数
        /// struct GlobalSequenceVariablePacket {
        ///     PacketHeader header;
        ///     u8 variableCount;
        ///     u8 actualVariableCount;
        ///     u8 padding[2];
        ///     s16 variables[];  // パックされた変数 (変数の個数、データ...の順番でパック)
        /// }
        ///
        /// // ローカル、トラック変数
        /// struct SequenceVariablePacket {
        ///     struct Item {
        ///         u32 soundInstanceId;
        ///         u32 soundId;
        ///         u64 archiveInstanceId;
        ///         u8 availableVariable;  // !0 = 変数が有効
        ///         u8 trackCount;
        ///         u8 localVariableCount;
        ///         u8 trackVariableCount;
        ///         s16 variables[];  // パックされた変数のリスト
        ///     }
        ///
        ///     PacketHeader header;
        ///     u8 itemCount;
        ///     u8 padding[3];
        ///     u8 itemArea[];  // パックされた Item のリスト
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_2_1_0 = new Version(0, 2, 1, 0);

        /// <summary>
        /// Version 0.3.0.0
        /// <list type="bullet">
        /// <item>サウンドのパラメータ対応。</item>
        /// </list>
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// // 現在の状態
        /// struct SoundStatePacket {
        ///     struct Item {
        ///         u32 instanceId;
        ///         u32 soundId;
        ///         u8 statusBitFlag;
        ///         u8 padding[3];
        ///         u64 archiveInstanceId;
        ///+        float volume;
        ///+        float pitch;
        ///+        float lpf;
        ///+        s32 bqfType;
        ///+        float bqfValue;
        ///+        s32 playerPriority;
        ///+        float pan;
        ///+        float surroundPan;
        ///     }
        ///
        ///     PacketHeader header;
        ///+    u16 itemSize;
        ///+    u8 padding[2];
        ///     u32 itemCount;
        ///     Item items[itemCount];
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_3_0_0 = new Version(0, 3, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 4, 0, 0);

        public class SoundArchiveData
        {
            public string Name { get; internal set; }

            public bool IsAddon { get; internal set; }

            internal Dictionary<uint, SoundData> SoundDatas { get; } = new Dictionary<uint, SoundData>();
        }

        public class SoundData
        {
            public uint PlayerId { get; internal set; }

            public SoundType SoundType { get; internal set; }

            /// <summary>
            /// 0 ~ 255。127が1.0倍。
            /// </summary>
            public uint Volume { get; internal set; }

            /// <summary>
            /// 0 ~ 127
            /// </summary>
            public uint PlayerPriority { get; internal set; }

            public string Label { get; internal set; }

            public PlayerData PlayerData { get; internal set; }

            public SoundArchiveData SoundArchiveData { get; internal set; }
        }

        public class PlayerData
        {
            public string Label { get; internal set; }
        }

        public class SoundState
        {
            public const byte StatusStart = (1 << 0);
            public const byte StatusPause = (1 << 1);
            public const byte StatusPrepared = (1 << 2);

            public SpyGlobalTime Timestamp { get; internal set; }

            public SpyTime BelongingFrame { get; internal set; }

            public uint InstanceId { get; internal set; }

            public uint SoundId { get; internal set; }

            public byte StatusBitFlag { get; internal set; }

            public ulong ArchiveInstanceId { get; internal set; }

            public SoundData SoundData { get; internal set; }

            public float Volume { get; internal set; }

            public float Pitch { get; internal set; }

            public float Lpf { get; internal set; }

            public int BiquadFilterType { get; internal set; }

            public float BiquadFilterValue { get; internal set; }

            public int PlayerPriority { get; internal set; }

            public float Pan { get; internal set; }

            public float SurroundPan { get; internal set; }
        }

        public class SequenceVariable
        {
            public uint SoundInstanceId { get; internal set; }

            public uint SoundId { get; internal set; }

            public ulong ArchiveInstanceId { get; internal set; }

            // 変数が有効なのかを表します。
            // falseのときには LocalVariables, TrackVariablesには正しく取得できた値が入っていません。
            public bool IsVariableAvailable { get; internal set; }

            public IEnumerable<int> LocalVariables { get; internal set; }

            public IEnumerable<TrackVariable> TrackVariables { get; internal set; }

            public class TrackVariable
            {
                public TrackVariable(IEnumerable<int> variables)
                {
                    this.Variables = variables;
                }

                public IEnumerable<int> Variables { get; }
            }
        }

        public class SoundStateChangedEventArgs : EventArgs
        {
            public IList<SoundState> SoundStates { get; internal set; }
            public SpyGlobalTime Timestamp { get; internal set; }
            public SpyTime BelongingFrame { get; internal set; }
        }

        public class GlobalSequenceVariableChangedEventArgs : EventArgs
        {
            public SpyGlobalTime Timestamp { get; internal set; }

            public SpyTime BelongingFrame { get; internal set; }

            public int[] Variables { get; internal set; }
        }

        public class SequenceVariableChangedEventArgs : EventArgs
        {
            public SpyGlobalTime Timestamp { get; internal set; }

            public SpyTime BelongingFrame { get; internal set; }

            public IList<SequenceVariable> SequenceVariables { get; internal set; }
        }

        private readonly Dictionary<ulong, SoundArchiveData> _archiveDataTable = new Dictionary<ulong, SoundArchiveData>();
        private readonly Dictionary<uint, PlayerData> _playerDataTable = new Dictionary<uint, PlayerData>();
        private readonly List<SoundState> _newSoundStates = new List<SoundState>();
        private readonly List<SoundStateCollection> _allSoundStates = new List<SoundStateCollection>();
        private readonly List<SoundData> _listForResolve = new List<SoundData>();
        private bool _errorUnexpectedDataVersion = false;
        private readonly List<GlobalSequenceVariableCollection> _allGlobalSequenceVariables = new List<GlobalSequenceVariableCollection>();
        private readonly List<SequenceVariableCollection> _allSequenceVariables = new List<SequenceVariableCollection>();

        /// <summary>
        /// SoundData または PlayerData を受信したときに通知します。
        /// </summary>
        public event EventHandler<EventArgs> DataChanged;

        /// <summary>
        /// SoundState を受信したときに通知します。
        /// </summary>
        public event EventHandler<SoundStateChangedEventArgs> SoundStateChanged;

        /// <summary>
        /// SequenceVariable を受信したときに通知します。
        /// </summary>
        public event EventHandler<GlobalSequenceVariableChangedEventArgs> GlobalSequenceVariableChanged;

        /// <summary>
        /// SequenceVariable を受信したときに通知します。
        /// </summary>
        public event EventHandler<SequenceVariableChangedEventArgs> SequenceVariableChanged;

        public IList<SoundState> NewSoundStates
        {
            get { return _newSoundStates; }
        }

        /// <summary>
        /// サポートしないバージョンのデータを受信すると true に設定されます。
        /// </summary>
        public bool ErrorUnexpectedDataVersion
        {
            get { return _errorUnexpectedDataVersion; }
            private set { this.SetPropertyValue(ref _errorUnexpectedDataVersion, value); }
        }

        private enum PacketType
        {
            SoundData,
            PlayerData,
            SoundState,
            AddonSoundArchiveData,
            GlobalSequenceVariable,
            SequenceVariable,
        }

        private enum DataType
        {
            Sound = 0,
            Player = 1
        }

        public enum SoundType
        {
            Sequence = 0,
            Wave = 1,
            Stream = 2
        }

        private class SoundStateCollection
        {
            private readonly List<SoundState> _newSoundStates = new List<SoundState>();

            public SpyGlobalTime Timestamp { get; set; }

            public SpyTime BelongingFrame { get; set; }

            public uint SoundCount { get; set; }

            public IList<SoundState> NewSoundStates
            {
                get { return _newSoundStates; }
            }
        }

        private class GlobalSequenceVariableCollection
        {
            public SpyGlobalTime Timestamp { get; set; }

            public int[] Variables { get; set; }
        }

        private class SequenceVariableCollection
        {
            private readonly List<SequenceVariable> _sequenceVariables = new List<SequenceVariable>();

            public SpyGlobalTime Timestamp { get; set; }

            public IList<SequenceVariable> SequenceVariables
            {
                get { return _sequenceVariables; }
            }
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                this.ErrorUnexpectedDataVersion = true;
                return;
            }

            var reader = CreateDataReader(dataBlock);

            PacketType packetType = (PacketType)reader.ReadByte();
            reader.ReadBytes(3); // padding

            switch (packetType)
            {
                case PacketType.SoundData:
                    OnPushSoundData(dataBlock, reader);
                    break;

                case PacketType.PlayerData:
                    OnPushPlayerData(dataBlock, reader);
                    break;

                case PacketType.SoundState:
                    // StateData の受信時には SoundData と PlayerData は受信済みのはず。
                    ResolveReferences();

                    OnPushSoundState(dataBlock, reader);
                    break;

                case PacketType.AddonSoundArchiveData:
                    OnPushAddonSoundArchiveData(dataBlock, reader);
                    break;

                case PacketType.GlobalSequenceVariable:
                    OnPushGlobalSequenceVariable(dataBlock, reader);
                    break;

                case PacketType.SequenceVariable:
                    OnPushSequenceVariable(dataBlock, reader);
                    break;

                default:
                    break;
            }
        }

        private void OnPushSoundData(SpyDataBlock dataBlock, BinaryReader reader)
        {
            ulong instanceId = 0;
            if (this.DataVersion >= Version_0_2_0_0)
            {
                instanceId = reader.ReadUInt64();
            }

            SoundArchiveData archiveData;
            if (!_archiveDataTable.TryGetValue(instanceId, out archiveData))
            {
                archiveData = new SoundArchiveData();
                _archiveDataTable[instanceId] = archiveData;
            }

            uint soundCount = reader.ReadUInt32();

            for (int i = 0; i < soundCount; i++)
            {
                SoundData soundData = new SoundData();

                uint soundId = reader.ReadUInt32();
                soundData.PlayerId = reader.ReadUInt32();
                soundData.SoundType = (SoundType)reader.ReadByte();
                soundData.Volume = reader.ReadByte();
                soundData.PlayerPriority = reader.ReadByte();
                reader.ReadByte(); // padding1
                byte labelLength = reader.ReadByte();
                string label = new string(reader.ReadChars(labelLength));

                int validSize = 1 + labelLength; // byte + char[labelLength]
                int paddingSize = (~(validSize + 0x03)) & 0x03; // 4バイトアライメントの補数
                if (paddingSize > 0)
                {
                    reader.ReadBytes(paddingSize);
                }
                soundData.Label = label;
                soundData.SoundArchiveData = archiveData;

                archiveData.SoundDatas[soundId] = soundData;

                _listForResolve.Add(soundData);
            }

            NotifyDataChanged();
        }

        private void OnPushPlayerData(SpyDataBlock dataBlock, BinaryReader reader)
        {
            uint itemCount = reader.ReadUInt32();

            for (int i = 0; i < itemCount; i++)
            {
                PlayerData playerData = new PlayerData();

                uint playerId = reader.ReadUInt32();
                byte labelLength = reader.ReadByte();
                string label = new string(reader.ReadChars(labelLength));

                int validSize = 1 + labelLength; // byte + char[labelLength]
                int paddingSize = (~(validSize + 0x03)) & 0x03; // 4バイトアライメントの補数
                if (paddingSize > 0)
                {
                    reader.ReadBytes(paddingSize);
                }
                playerData.Label = label;

                _playerDataTable[playerId] = playerData;
            }

            NotifyDataChanged();
        }

        private void OnPushSoundState(SpyDataBlock dataBlock, BinaryReader reader)
        {
            uint soundSize = 0;
            if (this.DataVersion >= Version_0_3_0_0)
            {
                soundSize = reader.ReadUInt16();
                reader.ReadBytes(2);
            }

            uint soundCount = reader.ReadUInt32();

            var belongingFrame = GetBelongingFrame(dataBlock.Timestamp);

            if (SoundStateChanged != null)
            {
                NewSoundStates.Clear();

                for (int i = 0; i < soundCount; i++)
                {
                    SoundState newState = CreateSoundState(
                        dataBlock.Timestamp,
                        GetBelongingFrame(dataBlock.Timestamp),
                        reader, soundSize);

                    if (newState != null)
                    {
                        NewSoundStates.Add(newState);
                    }
                }

                OnSoundStateChanged(dataBlock.Timestamp, belongingFrame);
            }
            else
            {
                var infos = new SoundStateCollection()
                {
                    Timestamp = dataBlock.Timestamp,
                    BelongingFrame = belongingFrame,
                    SoundCount = soundCount
                };

                for (int i = 0; i < soundCount; i++)
                {
                    SoundState newState = CreateSoundState(
                        dataBlock.Timestamp,
                        belongingFrame,
                        reader, soundSize);

                    if (newState != null)
                    {
                        infos.NewSoundStates.Add(newState);
                    }
                }

                _allSoundStates.Add(infos);
            }
        }

        private void OnPushAddonSoundArchiveData(SpyDataBlock dataBlock, BinaryReader reader)
        {
            var archiveData = new SoundArchiveData();

            ulong instanceId = reader.ReadUInt64();
            byte nameLength = reader.ReadByte();
            archiveData.Name = new string(reader.ReadChars(nameLength));
            archiveData.IsAddon = true;

            _archiveDataTable[instanceId] = archiveData;
        }

        private void ResolveReferences()
        {
            foreach (var soundData in _listForResolve)
            {
                PlayerData playerData;
                _playerDataTable.TryGetValue(soundData.PlayerId, out playerData);
                soundData.PlayerData = playerData;
            }
            _listForResolve.Clear();
        }

        private void OnPushGlobalSequenceVariable(SpyDataBlock dataBlock, BinaryReader reader)
        {
            var variableCount = (int)reader.ReadByte();
            var actualVariableCount = (int)reader.ReadByte();
            reader.ReadBytes(2); // padding

            var variables = Enumerable.Range(0, actualVariableCount)
                .Select(_ => (int)reader.ReadInt16())
                .Concat(Enumerable.Repeat(-1, variableCount - actualVariableCount))
                .ToArray();

            var globalSequenceVariableCollection = new GlobalSequenceVariableCollection()
            {
                Timestamp = dataBlock.Timestamp,
                Variables = variables,
            };

            OnGlobalSequenceVariableChanged(dataBlock.Timestamp, variables);

            _allGlobalSequenceVariables.Add(globalSequenceVariableCollection);
        }

        private void OnPushSequenceVariable(SpyDataBlock dataBlock, BinaryReader reader)
        {
            var itemCount = (int)reader.ReadByte();
            reader.ReadBytes(3);

            var sequenceVariables = new SequenceVariable[itemCount];
            for (var index = 0; index < itemCount; index++)
            {
                sequenceVariables[index] = this.CreateSequenceVariable(reader);
            }

            OnSequenceVariableChanged(dataBlock.Timestamp, sequenceVariables);

            var sequenceVariableCollection = new SequenceVariableCollection()
            {
                Timestamp = dataBlock.Timestamp,
            };
            sequenceVariables
                .ForEach(v => sequenceVariableCollection.SequenceVariables.Add(v));

            _allSequenceVariables.Add(sequenceVariableCollection);
        }

        private SoundState CreateSoundState(
            SpyGlobalTime timestamp,
            SpyTime belongingFrame,
            BinaryReader reader,
            uint soundSize)
        {
            if (reader == null)
            {
                return null;
            }

            long stateBeginPosition = reader.BaseStream.Position;
            SoundState newState = new SoundState();

            newState.InstanceId = reader.ReadUInt32();
            newState.SoundId = reader.ReadUInt32();
            newState.StatusBitFlag = reader.ReadByte();
            reader.BaseStream.Position += 3; // パディング分進めておく
            newState.ArchiveInstanceId = 0;
            if (this.DataVersion >= Version_0_2_0_0)
            {
                newState.ArchiveInstanceId = reader.ReadUInt64();
            }

            if (this.DataVersion >= Version_0_3_0_0)
            {
                newState.Volume = reader.ReadSingle();
                newState.Pitch = reader.ReadSingle();
                newState.Lpf = reader.ReadSingle();
                newState.BiquadFilterType = reader.ReadInt32();
                newState.BiquadFilterValue = reader.ReadSingle();
                newState.PlayerPriority = reader.ReadInt32();
                newState.Pan = reader.ReadSingle();
                newState.SurroundPan = reader.ReadSingle();
            }

            if (this.DataVersion >= Version_0_3_0_0)
            {
                long readedSize = reader.BaseStream.Position - stateBeginPosition;
                if (readedSize < soundSize)
                {
                    reader.BaseStream.Seek(soundSize - readedSize, SeekOrigin.Current);
                }
            }

            newState.Timestamp = timestamp;
            newState.BelongingFrame = belongingFrame;

            newState.SoundData = this.GetSoundData(newState.SoundId, newState.ArchiveInstanceId);

            return newState;
        }

        private SequenceVariable CreateSequenceVariable(BinaryReader reader)
        {
            var beginPosition = reader.BaseStream.Position;
            uint instanceId = reader.ReadUInt32();
            uint soundId = reader.ReadUInt32();
            ulong archiveInstanceId = reader.ReadUInt64();

            bool isVariableAvailable = reader.ReadByte() != 0 ? true : false;
            int trackCount = reader.ReadByte();
            int localVariableCount = reader.ReadByte();
            int trackVariableCount = reader.ReadByte();

            var localVariables = DecompressVariable(reader, localVariableCount);

            var trackVariables = new List<SequenceVariable.TrackVariable>();
            for (var trackIndex = 0; trackIndex < trackCount; trackIndex++)
            {
                var values = DecompressVariable(reader, trackVariableCount);
                trackVariables.Add(new SequenceVariable.TrackVariable(values));
            }

            var padding = (reader.BaseStream.Position - beginPosition) % 4;
            if (padding > 0)
            {
                reader.ReadBytes(4 - (int)padding);
            }

            return new SequenceVariable()
            {
                SoundInstanceId = instanceId,
                SoundId = soundId,
                ArchiveInstanceId = archiveInstanceId,
                IsVariableAvailable = isVariableAvailable,
                LocalVariables = localVariables,
                TrackVariables = trackVariables,
            };
        }

        private int[] DecompressVariable(BinaryReader reader, int variableCount)
        {
            int count = reader.ReadInt16();
            return Enumerable.Range(0, count)
                .Select(_ => (int)reader.ReadInt16())
                .Concat(Enumerable.Repeat(-1, variableCount - count))
                .ToArray();
        }

        private void NotifyDataChanged()
        {
            this.DataChanged?.Invoke(this, EventArgs.Empty);
        }

        public SoundData GetSoundData(uint soundId, ulong soundArchiveInstanceId)
        {
            SoundArchiveData archiveData;
            if (!_archiveDataTable.TryGetValue(soundArchiveInstanceId, out archiveData))
            {
                return null;
            }

            SoundData soundData;
            if (!archiveData.SoundDatas.TryGetValue(soundId, out soundData))
            {
                return null;
            }

            return soundData;
        }

        public PlayerData GetPlayerData(uint playerId)
        {
            PlayerData data;
            if (_playerDataTable.TryGetValue(playerId, out data))
            {
                return data;
            }
            else
            {
                return null;
            }
        }

        private void OnSoundStateChanged(SpyGlobalTime timestamp, SpyTime belongingFrame)
        {
            SoundStateChanged?.Invoke(this, new SoundStateChangedEventArgs()
            {
                SoundStates = NewSoundStates,
                Timestamp = timestamp,
                BelongingFrame = belongingFrame,
            });
        }

        public void ReadAllSoundStates()
        {
            if (SoundStateChanged != null)
            {
                foreach (var states in _allSoundStates)
                {
                    SoundStateChanged(this, new SoundStateChangedEventArgs()
                    {
                        SoundStates = states.NewSoundStates,
                        Timestamp = states.Timestamp,
                        BelongingFrame = states.BelongingFrame
                    });
                }
            }
        }

        private void OnGlobalSequenceVariableChanged(SpyGlobalTime timestamp, int[] variables)
        {
            GlobalSequenceVariableChanged?.Invoke(this, new GlobalSequenceVariableChangedEventArgs()
            {
                Timestamp = timestamp,
                BelongingFrame = GetBelongingFrame(timestamp),
                Variables = variables,
            });
        }

        private void OnSequenceVariableChanged(SpyGlobalTime timestamp, SequenceVariable[] sequenceVariables)
        {
            SequenceVariableChanged?.Invoke(this, new SequenceVariableChangedEventArgs()
            {
                Timestamp = timestamp,
                BelongingFrame = GetBelongingFrame(timestamp),
                SequenceVariables = sequenceVariables,
            });
        }

        public void ReadAllSequenceVariables()
        {
            if (GlobalSequenceVariableChanged != null)
            {
                foreach (var glovalVariable in _allGlobalSequenceVariables)
                {
                    GlobalSequenceVariableChanged(this, new GlobalSequenceVariableChangedEventArgs()
                    {
                        Timestamp = glovalVariable.Timestamp,
                        BelongingFrame = GetBelongingFrame(glovalVariable.Timestamp),
                        Variables = glovalVariable.Variables,
                    });
                }
            }

            if (SequenceVariableChanged != null)
            {
                foreach (var sequenceVariableCollection in _allSequenceVariables)
                {
                    SequenceVariableChanged(this, new SequenceVariableChangedEventArgs()
                    {
                        Timestamp = sequenceVariableCollection.Timestamp,
                        BelongingFrame = GetBelongingFrame(sequenceVariableCollection.Timestamp),
                        SequenceVariables = sequenceVariableCollection.SequenceVariables,
                    });
                }
            }
        }
    }
}
