﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class StreamSoundItemViewModel : ObservableObject
    {
        public static readonly double InvalidFillingPercentage = -1.0f;

        private PlotSpyModel.PlotFloat _fillingPercentagePlotFloat;
        private PlotSpyModel.PlotState _streamStatePlotState;
        private PlotSpyModel.PlotFloat _remainingCachePercentagePlotFloat;

        private double _fillingPercentage;
        private string _loadingDelayState;
        private string _itemLabel;
        private readonly int _itemNum;
        private double _remainingCachePercentage;

        public double FillingPercentage
        {
            get
            {
                return _fillingPercentage;
            }
            private set
            {
                this.SetPropertyValue(ref _fillingPercentage, value);
            }
        }

        public double FillingPercentageMax => 100d;

        public bool IsDelayed
        {
            get
            {
                return LoadingDelayState == "Delayed" ? true : false;
            }
        }

        public string LoadingDelayState
        {
            get
            {
                return _loadingDelayState;
            }
            private set
            {
                if (this.SetPropertyValue(ref _loadingDelayState, value))
                {
                    this.NotifyPropertyChanged(nameof(this.IsDelayed));
                }
            }
        }

        public string ItemLabel
        {
            get
            {
                return _itemLabel;
            }
            private set
            {
                this.SetPropertyValue(ref _itemLabel, value);
            }
        }

        private string NodeName
        {
            get { return $"@nn/atk/atk StreamSound/{_itemNum.ToString()}"; }
        }

        public double RemainingCachePercentage
        {
            get
            {
                return _remainingCachePercentage;
            }
            private set
            {
                this.SetPropertyValue(ref _remainingCachePercentage, value);
            }
        }

        public bool IsRemainingCachePercentageVisible
        {
            get
            {
                return this.RemainingCachePercentage != InvalidFillingPercentage;
            }
        }

        public StreamSoundItemViewModel(int itemNum)
        {
            _fillingPercentage = InvalidFillingPercentage;
            _loadingDelayState = "Hidden";
            this.ItemLabel = string.Empty;
            _itemNum = itemNum;
            _remainingCachePercentage = InvalidFillingPercentage;
        }

        public bool FindFillingPercentagePlotFloat(PlotSpyModel plotSpyModel)
        {
            if (_fillingPercentagePlotFloat == null)
            {
                string fullName = PlotSpyModel.CreateFullName(this.NodeName, "FillingPercentage");
                _fillingPercentagePlotFloat = plotSpyModel.GetFloat(fullName);
            }

            return _fillingPercentagePlotFloat != null;
        }

        public bool FindDelayStatePlotState(PlotSpyModel plotSpyModel)
        {
            if (_streamStatePlotState == null)
            {
                string fullName = PlotSpyModel.CreateFullName(this.NodeName, "State");
                _streamStatePlotState = plotSpyModel.GetState(fullName);
            }

            return _streamStatePlotState != null;
        }

        public bool FindRemainingCachePercentagePlotFloat(PlotSpyModel plotSpyModel)
        {
            if (_remainingCachePercentagePlotFloat == null)
            {
                string fullName = PlotSpyModel.CreateFullName(this.NodeName, "RemainingCachePercentage");
                _remainingCachePercentagePlotFloat = plotSpyModel.GetFloat(fullName);
            }

            return _remainingCachePercentagePlotFloat != null;
        }

        public void UpdateCurrentTime(SpyPlaybackService playbackService)
        {
            if (_fillingPercentagePlotFloat == null)
            {
                this.FillingPercentage = -1.0d;
                return;
            }

            if (_streamStatePlotState == null)
            {
                return;
            }

            var fillingPercentageValue = this.FindNearestValue(_fillingPercentagePlotFloat, playbackService.Current.Timestamp);
            if (fillingPercentageValue != null)
            {
                this.FillingPercentage = fillingPercentageValue.Value;
            }

            var stateValue = this.FindNearestValue(_streamStatePlotState, playbackService.Current.Timestamp);
            if (stateValue != null)
            {
                // stateValue には "ラベル名" および "ストリームサウンドの充填状態" がスペース区切りで含まれる
                string value = (string)stateValue.Value;
                int spaceIndex = value.LastIndexOf(' ');
                if (spaceIndex >= 0)
                {
                    this.ItemLabel = value.Substring(0, spaceIndex);
                    this.LoadingDelayState = value.Substring(spaceIndex + 1);
                }
            }

            if (_remainingCachePercentagePlotFloat == null)
            {
                this.RemainingCachePercentage = InvalidFillingPercentage;
            }
            else
            {
                var remainingCachePercentageValue = this.FindNearestValue(_remainingCachePercentagePlotFloat, playbackService.Current.Timestamp);
                if (remainingCachePercentageValue != null)
                {
                    this.RemainingCachePercentage = remainingCachePercentageValue.Value;
                }
            }
        }

        private PlotSpyModel.PlotFloatValue FindNearestValue(PlotSpyModel.PlotFloat model, SpyGlobalTime time)
        {
            if (model.Values.Count == 0)
            {
                return null;
            }

            if (model.Values.Count == 1)
            {
                return model.Values[0];
            }

            var index = BinarySearchUtility.BinarySearch(model.Values, time, value => value.Timestamp, BinarySearchUtility.Options.SmallestIndex);

            if (index < 0)
            {
                index = Math.Max(0, ~index - 1);
            }

            try
            {
                return model.Values[index];
            }
            catch
            {
                return null;
            }
        }

        private PlotSpyModel.PlotStateValue FindNearestValue(PlotSpyModel.PlotState model, SpyGlobalTime time)
        {
            if (model.Values.Count == 0)
            {
                return null;
            }

            if (model.Values.Count == 1)
            {
                return model.Values[0];
            }

            var index = BinarySearchUtility.BinarySearch(model.Values, time, value => value.Time.Timestamp, BinarySearchUtility.Options.SmallestIndex);

            if (index < 0)
            {
                index = Math.Max(0, ~index - 1);
            }

            try
            {
                return model.Values[index];
            }
            catch
            {
                return null;
            }
        }

        public bool IsInvalid()
        {
            return this.FillingPercentage == -1.0f;
        }
    }
}
