﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Threading;
using System.Windows.Data;
using System.Windows.Markup;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    [ContentProperty("Items")]
    public class SequenceVariablePanelViewModel : ObservableObject
    {
        private const int InitialCollectionCount = 30;

        private readonly WeakReference<AtkSpyModel> _refModel = new WeakReference<AtkSpyModel>(null);

        private readonly List<SequenceVariableItemViewModel> _allItems = new List<SequenceVariableItemViewModel>();

        private readonly ObservableCollection<SequenceVariableItemViewModel> _collection = new ObservableCollection<SequenceVariableItemViewModel>();
        private readonly ObservableCollection<SoundInformation> _soundInformations = new ObservableCollection<SoundInformation>();

        private readonly object _lockObject = new object();
        private ICollectionView _viewSoundInformations = null;

        private readonly Manager _manager = new Manager();
        private Manager.Group _currentGroup;
        private uint? _currentInstanceId = null;
        private SpyTimeUnit _timeUnit;
        private SpyTime _currentTime = SpyTime.InvalidValue;
        private SoundInformation _selectedSoundInformation = null;

        public SequenceVariablePanelViewModel()
        {
        }

        protected override void DisposeManagedInstance()
        {
            _allItems.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public SynchronizationContext SyncContext
        {
            get;
            set;
        }

        private AtkSpyModel TargetModel
        {
            get { return _refModel.GetTarget(); }
            set { _refModel.SetTarget(value); }
        }

        public ICollectionView ViewItems
        {
            get { return CollectionViewSource.GetDefaultView(this.Items); }
        }

        public ICollectionView ViewSoundInformations
        {
            get
            {
                if (_viewSoundInformations == null)
                {
                    lock (_lockObject)
                    {
                        this.SetupViewSoundInformations(this.SoundInformations);
                    }
                }

                return _viewSoundInformations;
                //return CollectionViewSource.GetDefaultView(this.SoundInformations);
            }
        }

        public ObservableCollection<SequenceVariableItemViewModel> Items
        {
            get { return _collection; }
        }

        public ObservableCollection<SoundInformation> SoundInformations
        {
            get { return _soundInformations; }
        }

        public SoundInformation SelectedSoundInformation
        {
            get { return _selectedSoundInformation; }
            set
            {
                this.SetPropertyValue(ref _selectedSoundInformation, value);
                UpdateItems();
            }
        }

        public SpyTime CurrentTime
        {
            get { return _currentTime; }
            set
            {
                if (value == _currentTime)
                {
                    return;
                }

                _currentTime = value;
                UpdateItems();
            }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        ///
        public void SetModel(AtkSpyModel model)
        {
            var lastModel = this.TargetModel;

            // モデルが切り替わるときは一旦 null に設定される必要があります。
            Assertion.Operation.True(model == null || (lastModel == null && model != null));

            if (lastModel == model)
            {
                return;
            }

            this.TargetModel = model;

            this.Clear();

            if (lastModel != null)
            {
                lastModel.GlobalSequenceVariableChanged -= this.OnGlobalSequenceVariableChanged;
                lastModel.SequenceVariableChanged -= this.OnSequenceVariableChanged;
            }

            if (model != null)
            {
                model.GlobalSequenceVariableChanged += this.OnGlobalSequenceVariableChanged;
                model.SequenceVariableChanged += this.OnSequenceVariableChanged;
                model.ReadAllSequenceVariables();
            }

            UpdateItems();
        }

        private void SetupViewSoundInformations(object source)
        {
            var viewItems = CollectionViewSource.GetDefaultView(source);
            viewItems.SortDescriptions.Add(new SortDescription("Time.Timestamp", ListSortDirection.Descending));
            viewItems.GroupDescriptions.Add(new PropertyGroupDescription("Time.Timestamp"));
            viewItems.Refresh();
            _viewSoundInformations = viewItems;
        }

        private void Clear()
        {
            this.SelectedSoundInformation = null;
            this.SoundInformations.Clear();
            _currentGroup = null;
            _currentInstanceId = null;
            this.Items.Clear();
            _allItems.ForEach(it => it.Dispose());
            _allItems.Clear();
            _manager.Clear();
        }

        /// <summary>
        /// CurrentTime に応じて Items を更新します。
        /// </summary>
        private void UpdateItems()
        {
            if (!this.CurrentTime.Timestamp.IsValid)
            {
                return;
            }

            this.KillItems();
            var globalVariable = _manager.FindGlobalVariable(this.CurrentTime.Timestamp);
            if (globalVariable != null)
            {
                this.SetItem(globalVariable);
            }

            var group = _manager.Find(this.CurrentTime.Timestamp);
            var instanceId = this.SelectedSoundInformation != null ? this.SelectedSoundInformation.InstanceId : _currentInstanceId;

            if (_currentGroup != group || _currentInstanceId != instanceId)
            {
                _currentGroup = group;

                if (group != null)
                {
                    if (this.SelectedSoundInformation != null)
                    {
                        var id = this.SelectedSoundInformation.InstanceId;
                        group.Items
                            .Where(i => i.SoundInstanceId == id)
                            .ForEach(i => this.SetItem(i));
                    }
                }
            }
        }

        /// <summary>
        /// Items のすべての要素を未使用状態にします。
        /// </summary>
        private void KillItems()
        {
            this.Items
                .ForEach(i => i.IsAlive = false);
        }

        private void SetItem(SequenceVariableItemViewModel item)
        {
            var freeItem = this.Items
                .FirstOrDefault(i => i.IsAlive == false);

            if (freeItem == null)
            {
                freeItem = ExpandCollection();
            }

            freeItem.CopyFrom(item);
            freeItem.IsAlive = true;
        }

        ///
        private SequenceVariableItemViewModel ExpandCollection()
        {
            var item = new SequenceVariableItemViewModel();
            this.Items.Add(item);
            return item;
        }

        private void AddSoundInformation(SequenceVariableItemViewModel item)
        {
            var soundData = this.TargetModel.GetSoundData(item.SoundId, item.ArchiveInstanceId);
            var name = soundData != null ? soundData.Label : item.SoundId.ToString();

            this.SoundInformations.Add(new SoundInformation
            {
                Name = name,
                SoundId = item.SoundId,
                InstanceId = item.SoundInstanceId,
                ArchiveInstanceId = item.ArchiveInstanceId,
                Time = item.Time,
            });
        }

        private IEnumerable<SequenceVariableItemViewModel> CreateItem(SpyTime time, AtkSpyModel.SequenceVariable sequenceVariable)
        {
            var items = new List<SequenceVariableItemViewModel>();
            var variableId = 0U;

            {
                var item = new SequenceVariableItemViewModel()
                {
                    SoundId = sequenceVariable.SoundId,
                    SoundInstanceId = sequenceVariable.SoundInstanceId,
                    ArchiveInstanceId = sequenceVariable.ArchiveInstanceId,
                    VariableId = variableId++,
                    IsVariableAvailable = sequenceVariable.IsVariableAvailable,
                    Time = time,
                    Name = Resources.Labels.LocalSequenceVariable,
                    Variables = (int[])sequenceVariable.LocalVariables.ToArray().Clone(),
                };
                items.Add(item);
            }

            var index = 0;
            foreach (var trackVariable in sequenceVariable.TrackVariables)
            {
                var item = new SequenceVariableItemViewModel()
                {
                    SoundId = sequenceVariable.SoundId,
                    SoundInstanceId = sequenceVariable.SoundInstanceId,
                    ArchiveInstanceId = sequenceVariable.ArchiveInstanceId,
                    VariableId = variableId++,
                    IsVariableAvailable = sequenceVariable.IsVariableAvailable,
                    Name = string.Format($"{Resources.Labels.TrackSequenceVariable}{index}"),
                    Variables = (int[])trackVariable.Variables.ToArray().Clone(),
                };
                items.Add(item);
                index++;
            }

            return items;
        }

        private void OnGlobalSequenceVariableChanged(object sender, AtkSpyModel.GlobalSequenceVariableChangedEventArgs e)
        {
            var timestamp = e.Timestamp;
            var time = new SpyTime(e.Timestamp, e.BelongingFrame.AppFrame, e.BelongingFrame.AudioFrame);

            {
                var item = new SequenceVariableItemViewModel()
                {
                    SoundId = 0,
                    SoundInstanceId = 0,
                    ArchiveInstanceId = 0,
                    VariableId = 0,
                    IsVariableAvailable = true,
                    Time = time,
                    Name = Resources.Labels.GlobalSequenceVariable,
                    Variables = (int[])e.Variables.ToArray().Clone(),
                };
                _manager.AddGlobalVariable(timestamp, item);
            }
        }

        private void OnSequenceVariableChanged(object sender, AtkSpyModel.SequenceVariableChangedEventArgs e)
        {
            var timestamp = e.Timestamp;
            var time = new SpyTime(e.Timestamp, e.BelongingFrame.AppFrame, e.BelongingFrame.AudioFrame);

            Manager.Group playingGroup = _manager.PlayingGroup;
            if (playingGroup == null)
            {
                foreach (var variable in e.SequenceVariables)
                {
                    // 再生開始。
                    var items = CreateItem(time, variable);
                    _manager.Add(e.Timestamp, items);
                    foreach (var item in items)
                    {
                        _allItems.Add(item);
                    }
                    this.AddSoundInformation(items.First());
                }
            }
            else
            {
                List<SequenceVariableItemViewModel> removedItems = null;
                List<SequenceVariableItemViewModel> changedItems = null;
                foreach (var item in playingGroup.Items)
                {
                    var variable = e.SequenceVariables.FirstOrDefault(i => i.SoundInstanceId == item.SoundInstanceId);
                    if (variable == null)
                    {
                        // 再生終了
                        if (removedItems == null)
                        {
                            removedItems = new List<SequenceVariableItemViewModel>();
                        }
                        removedItems.Add(item);
                    }
                    else if (CheckItemModify(item, variable))
                    {
                        // パラメータが変化した。
                        if (changedItems == null)
                        {
                            changedItems = new List<SequenceVariableItemViewModel>();
                        }
                        var items = CreateItem(time, variable);
                        foreach (var changedItem in items)
                        {
                            changedItems.Add(changedItem);
                        }
                    }
                }

                if (removedItems != null)
                {
                    foreach (var item in removedItems)
                    {
                        _manager.Remove(timestamp, item.SoundInstanceId);
                    }
                }

                if (changedItems != null)
                {
                    _manager.Add(timestamp, changedItems);
                    foreach (var changedItem in changedItems)
                    {
                        _allItems.Add(changedItem);
                    }
                }

                foreach (var variable in e.SequenceVariables)
                {
                    if (!playingGroup.Contains(variable.SoundInstanceId))
                    {
                        // 再生開始。
                        var items = CreateItem(time, variable);
                        _manager.Add(timestamp, items);
                        foreach (var item in items)
                        {
                            _allItems.Add(item);
                        }
                        this.AddSoundInformation(items.First());
                    }
                }
            }

            if (this.SelectedSoundInformation == null)
            {
                this.SelectedSoundInformation = this.SoundInformations.FirstOrDefault();
            }
        }

        private bool CheckItemModify(SequenceVariableItemViewModel item, AtkSpyModel.SequenceVariable sequenceVariable)
        {
            if (item.SoundId != sequenceVariable.SoundId ||
                item.Variables.SequenceEqual(sequenceVariable.LocalVariables) == false)
            {
                return true;
            }

            return false;
        }

        public class SoundInformation
        {
            public SpyTime Time { get; set; }

            public uint SoundId { get; set; }

            public uint InstanceId { get; set; }

            public ulong ArchiveInstanceId { get; set; }

            public string Name { get; set; }
        }

        ///
        public class Manager
        {
            private readonly List<GlobalVariable> _globalVariables = new List<GlobalVariable>();
            private readonly List<Group> _groups = new List<Group>();
            private Group _playingGroup = null;

            ///
            public Group PlayingGroup
            {
                get { return _playingGroup; }
            }

            ///
            public void Clear()
            {
                _globalVariables.Clear();
                _groups.Clear();
                _playingGroup = null;
            }

            public void AddGlobalVariable(SpyGlobalTime position, SequenceVariableItemViewModel item)
            {
                _globalVariables.Add(new GlobalVariable(position, item));
            }

            public void Add(SpyGlobalTime position, IEnumerable<SequenceVariableItemViewModel> items)
            {
                if (_playingGroup != null && _playingGroup.Position == position)
                {
                    items
                        .ForEach(i => _playingGroup.Add(i));
                }
                else
                {
                    var group = new Group(position);

                    if (_playingGroup != null)
                    {
                        group.Items.AddRange(_playingGroup.Items);
                    }

                    items
                        .ForEach(i => group.Add(i));
                    _groups.Add(group);
                    _playingGroup = group;
                }
            }

            public void Remove(SpyGlobalTime position, uint instanceID)
            {
                if (_playingGroup != null && _playingGroup.Contains(instanceID))
                {
                    if (_playingGroup.Position == position)
                    {
                        _playingGroup.Remove(instanceID);
                    }
                    else
                    {
                        var group = new Group(position);
                        group.Items.AddRange(_playingGroup.Items);
                        group.Remove(instanceID);
                        _groups.Add(group);
                        _playingGroup = group;
                    }
                }
            }

            public SequenceVariableItemViewModel FindGlobalVariable(SpyGlobalTime position)
            {
                var index = BinarySearchUtility.BinarySearch(_globalVariables, position, g => g.Position);

                index = Math.Max(index, ~index - 1);
                if (index >= 0)
                {
                    return _globalVariables[index].Item;
                }
                else
                {
                    return null;
                }
            }

            /// <summary>
            /// 指定したposition以前の最新のグループを返します。
            /// </summary>
            /// <param name="position"></param>
            /// <returns></returns>
            public Group Find(SpyGlobalTime position)
            {
                var index = BinarySearchUtility.BinarySearch(_groups, position, g => g.Position);

                index = Math.Max(index, ~index - 1);
                if (index >= 0)
                {
                    return _groups[index];
                }
                else
                {
                    return null;
                }
            }

            ///
            public class Group
            {
                private readonly List<SequenceVariableItemViewModel> _items = new List<SequenceVariableItemViewModel>();
                private readonly SpyGlobalTime _position;

                public Group(SpyGlobalTime position)
                {
                    _position = position;
                }

                public SpyGlobalTime Position
                {
                    get
                    {
                        return _position;
                    }
                }

                public List<SequenceVariableItemViewModel> Items
                {
                    get
                    {
                        return _items;
                    }
                }

                public bool Contains(uint instanceId)
                {
                    return _items.Any(i => i.SoundInstanceId == instanceId);
                }

                public void Add(SequenceVariableItemViewModel item)
                {
                    var index = _items.FindIndex(i => i.SoundInstanceId == item.SoundInstanceId && i.VariableId == item.VariableId);
                    if (index >= 0)
                    {
                        _items[index] = item;
                    }
                    else
                    {
                        _items.Add(item);
                    }
                }

                public void Remove(uint instanceID)
                {
                    _items.RemoveAll(i => i.SoundInstanceId == instanceID);
                }
            }

            private class GlobalVariable
            {
                public GlobalVariable(SpyGlobalTime position, SequenceVariableItemViewModel item)
                {
                    this.Position = position;
                    this.Item = item;
                }

                public SpyGlobalTime Position
                {
                    get;
                    private set;
                }

                public SequenceVariableItemViewModel Item
                {
                    get;
                    private set;
                }
            }
        }
    }
}
