﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using NintendoWare.Spy;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class LoudnessMeterLevelControl : Control
    {
        private class LevelItem
        {
            public double Y { get; set; }

            public float ColorCoefficient { get; set; } = 1.0f;

            public LevelItem(double y)
            {
                this.Y = y;
            }

            public LevelItem(double y, float colorScale)
            {
                this.Y = y;
                this.ColorCoefficient = colorScale;
            }
        }

        private static readonly SolidColorBrush DefaultColor = Brushes.Green;

        public static readonly DependencyProperty MaximumProperty =
            DependencyProperty.Register(
                nameof(Maximum),
                typeof(double),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    1.0d,
                    (s, e) => Self(s).OnMaximumChanged(e),
                    (s, v) => Self(s).CoerceMaximum((double)v)),
                IsValidDoubleValue);

        public static readonly DependencyProperty MinimumProperty =
            DependencyProperty.Register(
                nameof(Minimum),
                typeof(double),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    0.0d,
                    (s, e) => Self(s).OnMinimumChanged(e),
                    (s, v) => Self(s).CoerceMinimum((double)v)),
                IsValidDoubleValue);

        public static readonly DependencyProperty ValueProperty =
            DependencyProperty.Register(
                nameof(Value),
                typeof(double),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    0.0d,
                    (s, e) => Self(s).OnValueChanged(e)),
                IsValidDoubleValueOrNegativeInfinity);

        private static readonly DependencyPropertyKey PeakHoldPropertyKey =
            DependencyProperty.RegisterReadOnly(
                nameof(PeakHold),
                typeof(double),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    0.0d,
                    (s, e) => Self(s).OnPeakHoldChanged(e)),
                IsValidDoubleValueOrNegativeInfinity);

        public static readonly DependencyProperty PeakHoldProperty =
            PeakHoldPropertyKey.DependencyProperty;

        private static readonly DependencyPropertyKey PeakHoldBrushPropertyKey =
            DependencyProperty.RegisterReadOnly(
                nameof(PeakHoldBrush),
                typeof(Brush),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(DefaultColor));

        public static readonly DependencyProperty PeakHoldBrushProperty =
            PeakHoldBrushPropertyKey.DependencyProperty;

        public static readonly DependencyProperty ThresholdsProperty =
            DependencyProperty.Register(
                nameof(Thresholds),
                typeof(IEnumerable<LoudnessMeterThreshold>),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    null,
                    (s, e) => Self(s).OnThresholdsChanged(e)));

        public static readonly DependencyProperty PeakHoldHeightProperty =
            DependencyProperty.Register(
                nameof(PeakHoldHeight),
                typeof(double),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    2.0,
                    (s, e) => Self(s).OnPeakHoldHeightChanged(e)),
                v => IsValidDoubleValue(v, d => d >= 0.0));

        public static readonly DependencyProperty PeakHoldColorCoefficientProperty =
            DependencyProperty.Register(
                nameof(PeakHoldColorCoefficient),
                typeof(float),
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(
                    2.0f,
                    (s, e) => Self(s).OnPeakHoldColorCoefficientChanged(e)),
                v => IsValidFloatValue(v, f => f >= 0.0f));

        public double Maximum
        {
            get { return (double)this.GetValue(MaximumProperty); }
            set { this.SetValue(MaximumProperty, value); }
        }

        public double Minimum
        {
            get { return (double)this.GetValue(MinimumProperty); }
            set { this.SetValue(MinimumProperty, value); }
        }

        public double Value
        {
            get { return (double)this.GetValue(ValueProperty); }
            set { this.SetValue(ValueProperty, value); }
        }

        public double PeakHold
        {
            get { return (double)this.GetValue(PeakHoldProperty); }
            private set { this.SetValue(PeakHoldPropertyKey, value); }
        }

        public Brush PeakHoldBrush
        {
            get { return (Brush)this.GetValue(PeakHoldBrushProperty); }
            private set { this.SetValue(PeakHoldBrushPropertyKey, value); }
        }

        public IEnumerable<LoudnessMeterThreshold> Thresholds
        {
            get { return (IEnumerable<LoudnessMeterThreshold>)this.GetValue(ThresholdsProperty); }
            set { this.SetValue(ThresholdsProperty, value); }
        }

        public double PeakHoldHeight
        {
            get { return (double)this.GetValue(PeakHoldHeightProperty); }
            set { this.SetValue(PeakHoldHeightProperty, value); }
        }

        public float PeakHoldColorCoefficient
        {
            get { return (float)this.GetValue(PeakHoldColorCoefficientProperty); }
            set { this.SetValue(PeakHoldColorCoefficientProperty, value); }
        }

        private LoudnessMeterPeakHold _peakHold = new LoudnessMeterPeakHold();
        private SolidColorBrush _valueBrush;
        private List<LoudnessMeterThreshold> _sortedThresholds;

        static LoudnessMeterLevelControl()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(LoudnessMeterLevelControl),
                new FrameworkPropertyMetadata(typeof(LoudnessMeterLevelControl)));
        }

        public LoudnessMeterLevelControl()
        {
            _peakHold.HoldChanged += (o, s) => this.InvalidateVisual();

            // コントロールに設定されたタグを _peakHold にもコピーすることで
            // デバッグ時に用途を識別できるようにしておきます。
            DependencyPropertyDescriptor
                .FromProperty(FrameworkElement.TagProperty, typeof(LoudnessMeterLevelControl))
                .AddValueChanged(this, (o, a) => _peakHold.Tag = this.Tag);
        }

        private void OnMaximumChanged(DependencyPropertyChangedEventArgs e)
        {
            _sortedThresholds = null;
            this.InvalidateVisual();
        }

        private object CoerceMaximum(object v)
        {
            return Math.Max(this.Minimum, (double)v);
        }

        private void OnMinimumChanged(DependencyPropertyChangedEventArgs e)
        {
            _peakHold.Minimum = this.Minimum;
            _sortedThresholds = null;
            this.CoerceValue(MaximumProperty);
            this.InvalidateVisual();
        }

        private object CoerceMinimum(double v)
        {
            return v;
        }

        private void OnValueChanged(DependencyPropertyChangedEventArgs e)
        {
            _peakHold.Current = Math.Max(this.Value, this.Minimum);
            this.InvalidateVisual();
        }

        private void OnPeakHoldChanged(DependencyPropertyChangedEventArgs e)
        {
            this.UpdatePeakHoldBrush();
        }

        private void OnThresholdsChanged(DependencyPropertyChangedEventArgs e)
        {
            if (e.OldValue is INotifyCollectionChanged)
            {
                ((INotifyCollectionChanged)e.OldValue).CollectionChanged -= this.OnThresholsCollectionChanged;
            }

            if (e.NewValue is INotifyCollectionChanged)
            {
                ((INotifyCollectionChanged)e.NewValue).CollectionChanged += this.OnThresholsCollectionChanged;
            }

            _sortedThresholds = null;
            this.InvalidateVisual();
        }

        private void OnThresholsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            e.OldItems
                .OfType<INotifyPropertyChanged>()
                .ForEach(it => it.PropertyChanged -= this.OnThresholdsItemPropertyChanged);

            e.NewItems
                .OfType<INotifyPropertyChanged>()
                .ForEach(it => it.PropertyChanged += this.OnThresholdsItemPropertyChanged);

            _sortedThresholds = null;
            this.InvalidateVisual();
        }

        private void OnThresholdsItemPropertyChanged(object sender, PropertyChangedEventArgs args)
        {
            _sortedThresholds = null;
            this.InvalidateVisual();
        }

        private void OnPeakHoldHeightChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateVisual();
        }

        private void OnPeakHoldColorCoefficientChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateVisual();
        }

        /// <summary>
        /// 内部処理に適した形に閾値データを加工します。
        /// <list type="bullet">
        /// <item>Threshold の昇順にソート。</item>
        /// <item>Minimum, Maximum でかならず区間が区切られるように要素を追加。</item>
        /// </list>
        /// </summary>
        /// <returns></returns>
        private List<LoudnessMeterThreshold> GetSortedThresholds()
        {
            if (_sortedThresholds != null)
            {
                return _sortedThresholds;
            }

            List<LoudnessMeterThreshold> thresholds;
            if (this.Thresholds != null)
            {
                thresholds = this.Thresholds
                    .Where(it => !double.IsNaN(it.Threshold) && it.Fill != null)
                    .OrderBy(it => it.Threshold)
                    .ToList();
            }
            else
            {
                thresholds = new List<LoudnessMeterThreshold>();
            }

            // [this.Minimum, this.Maximum] の範囲のどの値に対しても色が指定されているようにします。
            if (thresholds.IsEmpty())
            {
                var maxThreshold = new LoudnessMeterThreshold
                {
                    Threshold = this.Maximum,
                    Fill = DefaultColor,
                };

                thresholds.Add(maxThreshold);
            }
            else
            {
                var maxThreshold = thresholds.Last();
                if (maxThreshold.Threshold < this.Maximum)
                {
                    maxThreshold = new LoudnessMeterThreshold
                    {
                        Threshold = this.Maximum,
                        Fill = maxThreshold.Fill,
                    };

                    thresholds.Add(maxThreshold);
                }
            }

            // (this.Minimum, this.Maximum] の範囲のどの値も２つの Threshold に囲まれるようにします。
            var minThreshold = thresholds.First();
            if (minThreshold.Threshold > this.Minimum)
            {
                minThreshold = new LoudnessMeterThreshold
                {
                    Threshold = this.Minimum,
                    Fill = minThreshold.Fill,
                };

                thresholds.Insert(0, minThreshold);
            }

            _sortedThresholds = thresholds;
            return _sortedThresholds;
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            base.OnRender(drawingContext);

            this.PeakHold = _peakHold.Hold;

            var width = this.ActualWidth;
            var viewHeight = this.ActualHeight;

            if (!IsValidDoubleValue(width) || !IsValidDoubleValue(viewHeight))
            {
                return;
            }

            var maximum = this.Maximum - this.Minimum;

            if (maximum <= double.Epsilon)
            {
                return;
            }

            IEnumerable<LoudnessMeterThreshold> thresholds = GetSortedThresholds();

            if (viewHeight >= maximum * 3)
            {
                // 1dB 毎に 1 ピクセルの隙間を開けて矩形を描きます。
                // (ピーク値の矩形を含みます)
                this.EnumerateLevelItemValues().ForEach(it =>
                    {
                        var top = (it.Y + 1) * viewHeight / maximum;
                        var bottom = it.Y * viewHeight / maximum;
                        var height = top - bottom - 1;

                        var fill = this.GetLevelItemBrush(
                            thresholds.First(th => th.Threshold >= it.Y + 1 + this.Minimum)?.Fill ?? thresholds.Last().Fill,
                            it.ColorCoefficient);
                        var rect = new Rect(0, viewHeight - top + 1, width, height);

                        this.DrawRectangle(drawingContext, fill, rect);
                    });
            }
            else
            {
                // 隙間をあけるのに十分な高さが無いので Threshold 毎にボックスを描きます。
                if (IsValidDoubleValue(this.Value))
                {
                    var value = MathUtility.Clamp(this.Value, this.Minimum, this.Maximum);
                    thresholds
                        .SuccessingPair((i, j) => new { Bottom = i, Top = j })
                        .Where(it => it.Bottom.Threshold < it.Top.Threshold)
                        .Where(it => this.Minimum < it.Top.Threshold)
                        .Where(it => it.Bottom.Threshold < value)
                        .ForEach(it =>
                        {
                            var top = Math.Min(it.Top.Threshold, value);
                            var bottom = Math.Max(it.Bottom.Threshold, this.Minimum);

                            if (top > bottom)
                            {
                                top = (top - this.Minimum) * viewHeight / maximum;
                                bottom = (bottom - this.Minimum) * viewHeight / maximum;
                                var height = top - bottom;

                                var fill = it.Top.Fill;
                                var rect = new Rect(0, viewHeight - top, width, height);

                                this.DrawRectangle(drawingContext, fill, rect);
                            }
                        });
                }

                // ピークホールド値のボックスを描きます。
                if (this.PeakHoldHeight > 0.0)
                {
                    var peakHold = _peakHold.Hold;
                    if (IsValidDoubleValue(peakHold) && peakHold > this.Minimum)
                    {
                        peakHold = Math.Min(peakHold, this.Maximum);

                        var y = peakHold - this.Minimum;

                        var top = y * viewHeight / maximum;
                        var height = this.PeakHoldHeight;

                        var fill = this.GetLevelItemBrush(
                            thresholds.First(it => it.Threshold >= peakHold).Fill,
                            this.PeakHoldColorCoefficient);
                        var rect = new Rect(0, viewHeight - top, width, height);

                        this.DrawRectangle(drawingContext, fill, rect);
                    }
                }
            }
        }

        private IEnumerable<LevelItem> EnumerateLevelItemValues()
        {
            if (IsValidDoubleValue(this.Value))
            {
                var value = MathUtility.Clamp(this.Value, this.Minimum, this.Maximum);
                for (double y = 0; y < value - this.Minimum; y++)
                {
                    yield return new LevelItem(y);
                }
            }

            if (this.PeakHoldHeight > 0.0)
            {
                var peakValue = MathUtility.Clamp(_peakHold.Hold, this.Minimum, this.Maximum);
                if (IsValidDoubleValue(peakValue) && peakValue > this.Minimum)
                {
                    yield return new LevelItem(
                        Math.Ceiling(peakValue - this.Minimum) - 1,
                        this.PeakHoldColorCoefficient);
                }
            }
        }

        private void DrawRectangle(DrawingContext drawingContext, Brush fill, Rect rect)
        {
            var guidelines = new GuidelineSet();
            guidelines.GuidelinesX.Add(rect.Left);
            guidelines.GuidelinesX.Add(rect.Right);
            guidelines.GuidelinesY.Add(rect.Top);
            guidelines.GuidelinesY.Add(rect.Bottom);

            drawingContext.PushGuidelineSet(guidelines);
            drawingContext.DrawRectangle(fill, null, rect);
            drawingContext.Pop();
        }

        private Brush GetLevelItemBrush(Brush brush, float coefficient)
        {
            if (coefficient != 1.0f && brush is SolidColorBrush)
            {
                var color = (brush as SolidColorBrush).Color;
                var newBrush = new SolidColorBrush(Color.Multiply(color, coefficient));
                newBrush.Freeze();
                return newBrush;
            }
            else
            {
                return brush;
            }
        }
        private void UpdatePeakHoldBrush()
        {
            if (this.Thresholds != null)
            {
                var peakHold = MathUtility.Clamp(this.PeakHold, this.Minimum, this.Maximum);

                var threshold = this.GetSortedThresholds()
                    .First(it => peakHold <= it.Threshold);

                if (_valueBrush != threshold.Fill)
                {
                    this.PeakHoldBrush = this.GetLevelItemBrush(threshold.Fill, this.PeakHoldColorCoefficient);
                    _valueBrush = threshold.Fill;
                }
            }
            else
            {
                if (_valueBrush != DefaultColor)
                {
                    this.PeakHoldBrush = this.GetLevelItemBrush(DefaultColor, this.PeakHoldColorCoefficient);
                    _valueBrush = DefaultColor;
                }
            }
        }

        private static bool IsValidDoubleValue(object v)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidDoubleValue(object v, Func<double, bool> predicate)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d) && predicate(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidDoubleValueOrNegativeInfinity(object v)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsPositiveInfinity(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidFloatValue(object v, Func<float, bool> predicate)
        {
            if (v is float)
            {
                var f = (float)v;
                return !float.IsNaN(f) && !float.IsInfinity(f) && predicate(f);
            }
            else
            {
                return false;
            }
        }

        private static LoudnessMeterLevelControl Self(object obj)
        {
            return (LoudnessMeterLevelControl)obj;
        }
    }
}
